export interface SubscriptionTier {
  id: string;
  name: string;
  price: number;
  billingPeriod: 'month' | 'year';
  interval: 'month' | 'year'; // Alias for billingPeriod for backward compatibility
  description: string;
  features: string[];
  popular?: boolean;
  highlighted?: boolean;
  badge?: string;
  cta: string;
  maxConsultationsPerMonth?: number;
  maxTrainingSessionsPerMonth?: number;
  supportLevel: 'basic' | 'priority' | 'premium' | 'vip';
}

export const SUBSCRIPTION_TIERS: SubscriptionTier[] = [
  {
    id: 'free',
    name: 'Free',
    price: 0,
    billingPeriod: 'month',
    description: 'Get started with basic government contracting resources',
    features: [
      'Access to community discussions',
      'Basic training video library',
      'Government contracting guides',
      'Email support',
      'Basic Section 508 resources'
    ],
    cta: 'Get Started Free',
    supportLevel: 'basic'
  },
  {
    id: 'starter',
    name: 'Starter',
    price: 97,
    billingPeriod: 'month',
    interval: 'month',
    description: 'Essential tools for small government contractors',
    features: [
      'Everything in Free',
      'Advanced training video library',
      'Monthly group training sessions',
      'Section 508 compliance checklist',
      'Priority email support',
      'Basic VPAT templates',
      'Government contracting roadmap'
    ],
    cta: 'Start Growing',
    maxTrainingSessionsPerMonth: 2,
    supportLevel: 'priority'
  },
  {
    id: 'professional',
    name: 'Professional',
    price: 197,
    billingPeriod: 'month',
    interval: 'month',
    popular: true,
    highlighted: true,
    badge: 'Most Popular',
    description: 'Complete solution for established government contractors',
    features: [
      'Everything in Starter',
      'Unlimited live training sessions',
      'Interactive training workshops',
      'Advanced Section 508 monitoring',
      'Custom VPAT generation',
      'Phone support',
      '1 consultation per month',
      'Advanced compliance tools',
      'SAM registration assistance'
    ],
    cta: 'Go Professional',
    maxConsultationsPerMonth: 1,
    maxTrainingSessionsPerMonth: -1, // unlimited
    supportLevel: 'premium'
  },
  {
    id: 'premium',
    name: 'Premium',
    price: 297,
    billingPeriod: 'month',
    interval: 'month',
    highlighted: false,
    badge: 'Best Value',
    description: 'Enterprise-level support for serious government contractors',
    features: [
      'Everything in Professional',
      'Unlimited one-on-one consultations',
      'White-label compliance reports',
      'Dedicated account manager',
      'Custom training programs',
      'Advanced analytics and reporting',
      'Priority Zoom support',
      'Direct access to compliance experts',
      'Custom contract review assistance'
    ],
    cta: 'Get Premium',
    maxConsultationsPerMonth: -1, // unlimited
    maxTrainingSessionsPerMonth: -1, // unlimited
    supportLevel: 'vip'
  }
];

export function getTierById(tierId: string): SubscriptionTier | undefined {
  return SUBSCRIPTION_TIERS.find(tier => tier.id === tierId);
}

export function getUserTierLevel(tierName: string): number {
  const tierOrder = ['free', 'starter', 'professional', 'premium'];
  return tierOrder.indexOf(tierName);
}

export function hasAccessToTier(userTier: string, requiredTier: string): boolean {
  const userLevel = getUserTierLevel(userTier);
  const requiredLevel = getUserTierLevel(requiredTier);
  return userLevel >= requiredLevel;
}

export function getTierUpgradePath(currentTier: string): SubscriptionTier | null {
  const currentIndex = SUBSCRIPTION_TIERS.findIndex(tier => tier.id === currentTier);
  if (currentIndex === -1 || currentIndex >= SUBSCRIPTION_TIERS.length - 1) {
    return null;
  }
  return SUBSCRIPTION_TIERS[currentIndex + 1];
}