import { sql } from "drizzle-orm";
import { pgTable, text, varchar, timestamp, boolean, integer, jsonb } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

export const users = pgTable("users", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  username: text("username").notNull().unique(),
  password: text("password").notNull(),
  email: text("email").unique(),
  firstName: text("first_name"),
  lastName: text("last_name"),
  profileImageUrl: text("profile_image_url"),
  bio: text("bio"),
  company: text("company"),
  isPremium: boolean("is_premium").default(false),
  premiumExpiresAt: timestamp("premium_expires_at"),
  squareCustomerId: text("square_customer_id"),
  paypalCustomerId: text("paypal_customer_id"),
  phone: text("phone"),
  smsOptIn: boolean("sms_opt_in").default(false),
  smsPreferences: text("sms_preferences").default("all"), // all, training, promotions, reminders
  zoomPmi: text("zoom_pmi"),
  zoomPmiPassword: text("zoom_pmi_password"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const posts = pgTable("posts", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  title: text("title").notNull(),
  content: text("content").notNull(),
  imageUrl: text("image_url"),
  category: text("category").default("general"),
  isPublic: boolean("is_public").default(true),
  isPremiumOnly: boolean("is_premium_only").default(false),
  requiredTier: text("required_tier").default("free"), // free, starter, professional, premium
  likesCount: integer("likes_count").default(0),
  commentsCount: integer("comments_count").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const comments = pgTable("comments", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  postId: varchar("post_id").notNull().references(() => posts.id, { onDelete: "cascade" }),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  content: text("content").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
});

export const likes = pgTable("likes", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  postId: varchar("post_id").notNull().references(() => posts.id, { onDelete: "cascade" }),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  createdAt: timestamp("created_at").defaultNow(),
});

export const follows = pgTable("follows", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  followerId: varchar("follower_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  followingId: varchar("following_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  createdAt: timestamp("created_at").defaultNow(),
});

export const subscriptions = pgTable("subscriptions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  paymentProvider: text("payment_provider").notNull(), // square, paypal, venmo, cashapp
  subscriptionId: text("subscription_id"),
  status: text("status").default("active"), // active, cancelled, expired
  planType: text("plan_type").default("starter"), // starter, professional, premium, enterprise
  amount: integer("amount").notNull(),
  currency: text("currency").default("USD"),
  startDate: timestamp("start_date").defaultNow(),
  expiresAt: timestamp("expires_at"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const systemMetrics = pgTable("system_metrics", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  metricType: text("metric_type").notNull(), // api_requests, storage_usage, user_activity, errors
  metricName: text("metric_name").notNull(), // specific metric name
  value: integer("value").notNull(),
  metadata: text("metadata"), // JSON string for additional data
  timestamp: timestamp("timestamp").defaultNow(),
});

export const usageAlerts = pgTable("usage_alerts", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  alertType: text("alert_type").notNull(), // limit_warning, limit_exceeded, high_usage
  resource: text("resource").notNull(), // database, storage, api_requests, users
  currentValue: integer("current_value").notNull(),
  limitValue: integer("limit_value").notNull(),
  threshold: integer("threshold").notNull(), // percentage threshold that triggered alert
  message: text("message").notNull(),
  isResolved: boolean("is_resolved").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const contacts = pgTable("contacts", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  firstName: text("first_name").notNull(),
  lastName: text("last_name").notNull(),
  email: text("email").notNull(),
  phone: text("phone"),
  serviceInterest: text("service_interest"),
  message: text("message").notNull(),
  isUrgent: boolean("is_urgent").default(false),
  smsOptIn: boolean("sms_opt_in").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

// SMS Campaign Management
export const smsCampaigns = pgTable("sms_campaigns", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  message: text("message").notNull(),
  campaignType: text("campaign_type").notNull(), // training, promotion, reminder, announcement
  targetAudience: text("target_audience").notNull(), // all, members, non-members, tier-specific
  tierTarget: text("tier_target"), // free, starter, professional, premium (for tier-specific campaigns)
  status: text("status").default("draft"), // draft, scheduled, sending, completed, cancelled
  scheduledAt: timestamp("scheduled_at"),
  sentAt: timestamp("sent_at"),
  totalRecipients: integer("total_recipients").default(0),
  successfulSends: integer("successful_sends").default(0),
  failedSends: integer("failed_sends").default(0),
  createdBy: varchar("created_by").references(() => users.id, { onDelete: "set null" }),
  createdAt: timestamp("created_at").defaultNow(),
});

// SMS Subscribers (for non-members)
export const smsSubscribers = pgTable("sms_subscribers", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  phone: text("phone").notNull().unique(),
  firstName: text("first_name"),
  lastName: text("last_name"),
  email: text("email"),
  source: text("source").default("website"), // website, consultation, contact-form
  preferences: text("preferences").default("all"), // all, training, promotions, reminders
  isActive: boolean("is_active").default(true),
  optInDate: timestamp("opt_in_date").defaultNow(),
  optOutDate: timestamp("opt_out_date"),
  createdAt: timestamp("created_at").defaultNow(),
});

// SMS Campaign Logs
export const smsLogs = pgTable("sms_logs", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  campaignId: varchar("campaign_id").references(() => smsCampaigns.id, { onDelete: "cascade" }),
  recipientPhone: text("recipient_phone").notNull(),
  recipientType: text("recipient_type").notNull(), // member, subscriber
  recipientId: varchar("recipient_id"), // user_id or subscriber_id
  message: text("message").notNull(),
  status: text("status").notNull(), // sent, delivered, failed, bounced
  ringcentralMessageId: text("ringcentral_message_id"),
  errorMessage: text("error_message"),
  deliveredAt: timestamp("delivered_at"),
  sentAt: timestamp("sent_at").defaultNow(),
});

export const consultations = pgTable("consultations", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  firstName: text("first_name").notNull(),
  lastName: text("last_name").notNull(),
  email: text("email").notNull(),
  phone: text("phone"),
  companyName: text("company_name"),
  serviceType: text("service_type").notNull(),
  preferredDate: text("preferred_date"),
  preferredTime: text("preferred_time"),
  message: text("message"),
  calendarEventId: text("calendar_event_id"), // Google Calendar event ID
  createdAt: timestamp("created_at").defaultNow(),
});

export const trainingVideos = pgTable("training_videos", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  title: text("title").notNull(),
  description: text("description"),
  videoUrl: text("video_url").notNull(),
  captionsUrl: text("captions_url"),
  transcriptUrl: text("transcript_url"),
  category: text("category").notNull(),
  isPublic: boolean("is_public").default(false),
  requiredTier: text("required_tier").default("free"), // free, starter, professional, premium
  createdAt: timestamp("created_at").defaultNow(),
});

export const fileUploads = pgTable("file_uploads", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  originalName: text("original_name").notNull(),
  fileName: text("file_name").notNull(),
  fileSize: text("file_size").notNull(),
  mimeType: text("mime_type").notNull(),
  uploadedBy: text("uploaded_by"),
  uploadedAt: timestamp("uploaded_at").defaultNow(),
});

// Section 508 Compliance Monitoring Tables
export const complianceScans = pgTable("compliance_scans", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  url: text("url").notNull(),
  scanType: text("scan_type").notNull(), // 'manual', 'automated', 'scheduled'
  status: text("status").notNull(), // 'running', 'completed', 'failed'
  overallScore: integer("overall_score"), // 0-100 compliance score
  totalIssues: integer("total_issues").default(0),
  criticalIssues: integer("critical_issues").default(0),
  warningIssues: integer("warning_issues").default(0),
  infoIssues: integer("info_issues").default(0),
  scanDuration: integer("scan_duration"), // milliseconds
  userAgent: text("user_agent"),
  createdAt: timestamp("created_at").defaultNow(),
  completedAt: timestamp("completed_at"),
});

export const complianceIssues = pgTable("compliance_issues", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  scanId: varchar("scan_id").notNull().references(() => complianceScans.id, { onDelete: "cascade" }),
  ruleId: text("rule_id").notNull(), // WCAG rule identifier
  severity: text("severity").notNull(), // 'critical', 'warning', 'info'
  wcagLevel: text("wcag_level").notNull(), // 'A', 'AA', 'AAA'
  section508Rule: text("section508_rule"), // Section 508 rule reference
  element: text("element"), // CSS selector of problematic element
  message: text("message").notNull(),
  helpUrl: text("help_url"),
  xpath: text("xpath"),
  html: text("html"),
  status: text("status").default('open'), // 'open', 'resolved', 'false_positive'
  resolvedAt: timestamp("resolved_at"),
  resolvedBy: text("resolved_by"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const complianceReports = pgTable("compliance_reports", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  title: text("title").notNull(),
  description: text("description"),
  reportType: text("report_type").notNull(), // 'vpat', 'section508', 'wcag', 'audit'
  status: text("status").notNull().default('draft'), // 'draft', 'published', 'archived'
  complianceLevel: text("compliance_level"), // 'A', 'AA', 'AAA', 'partial', 'non_compliant'
  dateRange: jsonb("date_range"), // { start: date, end: date }
  includedPages: jsonb("included_pages"), // array of URLs
  scanIds: jsonb("scan_ids"), // array of scan IDs included in report
  generatedBy: text("generated_by"),
  publishedAt: timestamp("published_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const complianceAlerts = pgTable("compliance_alerts", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  alertType: text("alert_type").notNull(), // 'critical_issue', 'compliance_drop', 'new_violations'
  severity: text("severity").notNull(), // 'high', 'medium', 'low'
  title: text("title").notNull(),
  message: text("message").notNull(),
  scanId: varchar("scan_id").references(() => complianceScans.id),
  url: text("url"),
  isRead: boolean("is_read").default(false),
  isResolved: boolean("is_resolved").default(false),
  resolvedAt: timestamp("resolved_at"),
  resolvedBy: text("resolved_by"),
  createdAt: timestamp("created_at").defaultNow(),
});

// Insert schemas
export const insertUserSchema = createInsertSchema(users).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  username: z.string().min(3, "Username must be at least 3 characters"),
  password: z.string().min(6, "Password must be at least 6 characters"),
  email: z.string().email("Please enter a valid email address").optional(),
});

export const insertPostSchema = createInsertSchema(posts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  likesCount: true,
  commentsCount: true,
}).extend({
  title: z.string().min(1, "Title is required"),
  content: z.string().min(10, "Content must be at least 10 characters"),
  category: z.string().optional(),
});

export const insertCommentSchema = createInsertSchema(comments).omit({
  id: true,
  createdAt: true,
}).extend({
  content: z.string().min(1, "Comment cannot be empty"),
});

export const insertSubscriptionSchema = createInsertSchema(subscriptions).omit({
  id: true,
  createdAt: true,
}).extend({
  paymentProvider: z.enum(["square", "paypal", "venmo", "cashapp"]),
  amount: z.number().positive("Amount must be positive"),
});

export const insertContactSchema = createInsertSchema(contacts).omit({
  id: true,
  createdAt: true,
}).extend({
  firstName: z.string().min(1, "First name is required"),
  lastName: z.string().min(1, "Last name is required"),
  email: z.string().email("Please enter a valid email address"),
  phone: z.string().optional(),
  message: z.string().min(10, "Message must be at least 10 characters"),
});

export const insertConsultationSchema = createInsertSchema(consultations).omit({
  id: true,
  createdAt: true,
}).extend({
  firstName: z.string().min(1, "First name is required"),
  lastName: z.string().min(1, "Last name is required"),
  email: z.string().email("Please enter a valid email address"),
  serviceType: z.string().min(1, "Please select a consultation type"),
  message: z.string().min(10, "Please provide details about your consultation needs (minimum 10 characters)"),
});

export const insertTrainingVideoSchema = createInsertSchema(trainingVideos).omit({
  id: true,
  createdAt: true,
}).extend({
  title: z.string().min(1, "Title is required"),
  videoUrl: z.string().url("Please enter a valid video URL"),
  category: z.string().min(1, "Category is required"),
});

export const insertComplianceScanSchema = createInsertSchema(complianceScans).omit({
  id: true,
  createdAt: true,
  completedAt: true,
}).extend({
  url: z.string().url("Please enter a valid URL"),
  scanType: z.enum(["manual", "automated", "scheduled"]),
  status: z.enum(["running", "completed", "failed"]),
});

export const insertComplianceIssueSchema = createInsertSchema(complianceIssues).omit({
  id: true,
  createdAt: true,
  resolvedAt: true,
}).extend({
  ruleId: z.string().min(1, "Rule ID is required"),
  severity: z.enum(["critical", "warning", "info"]),
  wcagLevel: z.enum(["A", "AA", "AAA"]),
  message: z.string().min(1, "Message is required"),
});

export const insertComplianceReportSchema = createInsertSchema(complianceReports).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  publishedAt: true,
}).extend({
  title: z.string().min(1, "Title is required"),
  reportType: z.enum(["vpat", "section508", "wcag", "audit"]),
  status: z.enum(["draft", "published", "archived"]),
});

export const insertComplianceAlertSchema = createInsertSchema(complianceAlerts).omit({
  id: true,
  createdAt: true,
  resolvedAt: true,
}).extend({
  alertType: z.enum(["critical_issue", "compliance_drop", "new_violations"]),
  severity: z.enum(["high", "medium", "low"]),
  title: z.string().min(1, "Title is required"),
  message: z.string().min(1, "Message is required"),
});

// Types
export type InsertUser = z.infer<typeof insertUserSchema>;
export type User = typeof users.$inferSelect;

export type InsertPost = z.infer<typeof insertPostSchema>;
export type Post = typeof posts.$inferSelect;

export type InsertComment = z.infer<typeof insertCommentSchema>;
export type Comment = typeof comments.$inferSelect;

export type Like = typeof likes.$inferSelect;
export type Follow = typeof follows.$inferSelect;

export type InsertSubscription = z.infer<typeof insertSubscriptionSchema>;
export type Subscription = typeof subscriptions.$inferSelect;

export type InsertContact = z.infer<typeof insertContactSchema>;
export type Contact = typeof contacts.$inferSelect;

export type InsertConsultation = z.infer<typeof insertConsultationSchema>;
export type Consultation = typeof consultations.$inferSelect;

export type InsertTrainingVideo = z.infer<typeof insertTrainingVideoSchema>;
export type TrainingVideo = typeof trainingVideos.$inferSelect;

export type InsertOAuthToken = z.infer<typeof insertOAuthTokenSchema>;
export type OAuthToken = typeof oauthTokens.$inferSelect;

// Settings table for storing application configuration
export const settings = pgTable("settings", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  key: text("key").notNull().unique(),
  value: text("value"),
  encrypted: boolean("encrypted").default(false),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertSettingsSchema = createInsertSchema(settings);
export type InsertSettings = z.infer<typeof insertSettingsSchema>;
export type Settings = typeof settings.$inferSelect;

export type FileUpload = typeof fileUploads.$inferSelect;

// Section 508 Compliance Types
export type InsertComplianceScan = z.infer<typeof insertComplianceScanSchema>;
export type ComplianceScan = typeof complianceScans.$inferSelect;

export type InsertComplianceIssue = z.infer<typeof insertComplianceIssueSchema>;
export type ComplianceIssue = typeof complianceIssues.$inferSelect;

export type InsertComplianceReport = z.infer<typeof insertComplianceReportSchema>;
export type ComplianceReport = typeof complianceReports.$inferSelect;

export type InsertComplianceAlert = z.infer<typeof insertComplianceAlertSchema>;
export type ComplianceAlert = typeof complianceAlerts.$inferSelect;

// Monitoring types
export type SystemMetric = typeof systemMetrics.$inferSelect;
export type UsageAlert = typeof usageAlerts.$inferSelect;

// Live Training Sessions table - Zoom-powered training sessions
export const liveTrainingSessions = pgTable("live_training_sessions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  title: varchar("title", { length: 255 }).notNull(),
  description: text("description"),
  category: varchar("category", { length: 100 }).notNull(), // 'compliance', 'tax', 'hr', 'accounting'
  sessionType: varchar("session_type", { length: 50 }).notNull(), // 'meeting', 'webinar', 'workshop'
  
  // Zoom integration
  zoomMeetingId: varchar("zoom_meeting_id", { length: 50 }), // Zoom meeting ID
  zoomMeetingUuid: varchar("zoom_meeting_uuid", { length: 100 }), // Zoom meeting UUID
  zoomJoinUrl: varchar("zoom_join_url", { length: 500 }), // Join URL for participants
  zoomStartUrl: varchar("zoom_start_url", { length: 500 }), // Start URL for host
  zoomPassword: varchar("zoom_password", { length: 50 }), // Meeting password
  
  // Schedule and duration
  scheduledStartTime: timestamp("scheduled_start_time").notNull(),
  scheduledEndTime: timestamp("scheduled_end_time").notNull(),
  duration: integer("duration").notNull(), // Minutes
  timezone: varchar("timezone", { length: 50 }).notNull().default('America/New_York'),
  
  // Instructor and access
  instructorId: varchar("instructor_id", { length: 255 }).notNull(),
  instructorName: varchar("instructor_name", { length: 255 }).notNull(),
  isPremiumOnly: boolean("is_premium_only").notNull().default(false),
  maxParticipants: integer("max_participants").default(100),
  
  // Session details
  agenda: text("agenda"),
  learningObjectives: text("learning_objectives").array(), // Array of objectives
  materials: text("materials").array(), // URLs to resources
  recordingEnabled: boolean("recording_enabled").notNull().default(true),
  recordingUrl: varchar("recording_url", { length: 500 }), // Post-session recording
  
  // Status and management
  status: varchar("status", { length: 50 }).notNull().default('scheduled'), // 'scheduled', 'live', 'completed', 'cancelled'
  actualStartTime: timestamp("actual_start_time"),
  actualEndTime: timestamp("actual_end_time"),
  
  // Metadata
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

// Training Session Registrations table - Who's attending
export const trainingRegistrations = pgTable("training_registrations", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  sessionId: varchar("session_id", { length: 255 }).notNull().references(() => liveTrainingSessions.id),
  userId: varchar("user_id", { length: 255 }).notNull().references(() => users.id),
  
  // Registration details
  registeredAt: timestamp("registered_at").defaultNow(),
  status: varchar("status", { length: 50 }).notNull().default('registered'), // 'registered', 'attended', 'no-show', 'cancelled'
  
  // Zoom integration
  zoomRegistrantId: varchar("zoom_registrant_id", { length: 100 }), // For webinars
  zoomJoinUrl: varchar("zoom_join_url", { length: 500 }), // Personal join URL
  
  // Attendance tracking
  joinedAt: timestamp("joined_at"),
  leftAt: timestamp("left_at"),
  attendanceDuration: integer("attendance_duration"), // Minutes attended
  
  // Feedback and completion
  completed: boolean("completed").notNull().default(false),
  feedbackRating: integer("feedback_rating"), // 1-5 stars
  feedbackComments: text("feedback_comments"),
  certificateIssued: boolean("certificate_issued").notNull().default(false),
  
  // Metadata
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

// Training Session Resources table - Shared materials
export const trainingResources = pgTable("training_resources", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  sessionId: varchar("session_id", { length: 255 }).notNull().references(() => liveTrainingSessions.id),
  
  // Resource details
  title: varchar("title", { length: 255 }).notNull(),
  description: text("description"),
  resourceType: varchar("resource_type", { length: 50 }).notNull(), // 'slide', 'document', 'video', 'link'
  fileUrl: varchar("file_url", { length: 500 }), // Cloud storage URL
  fileSize: integer("file_size"), // File size in bytes
  mimeType: varchar("mime_type", { length: 100 }),
  
  // Access control
  availableFrom: timestamp("available_from").defaultNow(),
  availableUntil: timestamp("available_until"), // Optional expiry
  downloadAllowed: boolean("download_allowed").notNull().default(true),
  
  // Metadata
  uploadedBy: varchar("uploaded_by", { length: 255 }).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

// OAuth Token Storage for persistent authentication
export const oauthTokens = pgTable("oauth_tokens", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  provider: text("provider").notNull(), // 'google', 'ringcentral'
  accessToken: text("access_token").notNull(),
  refreshToken: text("refresh_token"),
  expiresAt: timestamp("expires_at"),
  scope: text("scope"),
  accountId: text("account_id"), // Google email, RingCentral extension/owner
  metadata: jsonb("metadata"), // Provider-specific data like defaultFromNumber
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  userProviderUnique: sql`UNIQUE(${table.userId}, ${table.provider})`
}));

// OAuth Token schema for validation
export const insertOAuthTokenSchema = createInsertSchema(oauthTokens).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  provider: z.enum(["google", "ringcentral"]),
  accessToken: z.string().min(1, "Access token is required"),
  refreshToken: z.string().optional(),
});

// Zoom Zod schemas for validation
const insertLiveTrainingSessionSchema = createInsertSchema(liveTrainingSessions);
const insertTrainingRegistrationSchema = createInsertSchema(trainingRegistrations);
const insertTrainingResourceSchema = createInsertSchema(trainingResources);

// Zoom insert and select types for live training
export type InsertLiveTrainingSession = z.infer<typeof insertLiveTrainingSessionSchema>;
export type LiveTrainingSession = typeof liveTrainingSessions.$inferSelect;

export type InsertTrainingRegistration = z.infer<typeof insertTrainingRegistrationSchema>;
export type TrainingRegistration = typeof trainingRegistrations.$inferSelect;

export type InsertTrainingResource = z.infer<typeof insertTrainingResourceSchema>;
export type TrainingResource = typeof trainingResources.$inferSelect;
