import { db, withRetry } from './db.js';
import { users } from '@shared/schema';
import { eq } from 'drizzle-orm';
import type { User, InsertUser } from '@shared/schema';

/**
 * Database-backed user storage for persistence
 * Handles user data that needs to survive server restarts
 */
export class DatabaseUserStorage {
  
  // Get user from database
  async getUser(id: string): Promise<User | undefined> {
    try {
      return await withRetry(async () => {
        return await db.query.users.findFirst({
          where: eq(users.id, id)
        });
      });
    } catch (error) {
      console.error('Error fetching user after retries:', error);
      return undefined;
    }
  }

  // Get user by username
  async getUserByUsername(username: string): Promise<User | undefined> {
    try {
      return await withRetry(async () => {
        return await db.query.users.findFirst({
          where: eq(users.username, username)
        });
      });
    } catch (error) {
      console.error('Error fetching user by username after retries:', error);
      return undefined;
    }
  }

  // Create user in database
  async createUser(userData: InsertUser): Promise<User> {
    try {
      return await withRetry(async () => {
        const [newUser] = await db.insert(users)
          .values({
            ...userData,
            createdAt: new Date(),
            updatedAt: new Date()
          })
          .returning();
        return newUser;
      });
    } catch (error) {
      console.error('Error creating user after retries:', error);
      throw new Error('Failed to create user');
    }
  }

  // Update user settings (PMI, etc.)
  async updateUserSettings(id: string, settings: { zoomPmi?: string; zoomPmiPassword?: string }): Promise<User> {
    try {
      return await withRetry(async () => {
        const [updatedUser] = await db.update(users)
          .set({
            zoomPmi: settings.zoomPmi,
            zoomPmiPassword: settings.zoomPmiPassword,
            updatedAt: new Date()
          })
          .where(eq(users.id, id))
          .returning();

        if (!updatedUser) {
          throw new Error('User not found');
        }

        return updatedUser;
      });
    } catch (error) {
      console.error('Error updating user settings after retries:', error);
      throw new Error('Failed to update user settings');
    }
  }

  // Ensure admin user exists (migration helper)
  async ensureAdminUser(): Promise<User> {
    try {
      // Try to find existing admin user
      let adminUser = await this.getUserByUsername('tmh_admin');
      
      if (!adminUser) {
        // Create admin user if it doesn't exist
        adminUser = await this.createUser({
          username: 'tmh_admin',
          password: 'admin123',
          email: 'admin@tmhglobal.com',
          firstName: 'TMH',
          lastName: 'Global',
          company: 'TMH Global LLC',
          bio: 'Professional accounting and business services for government contractors',
          isPremium: true,
          smsOptIn: false,
          smsPreferences: 'all'
        });
        console.log('✅ Admin user created in database');
      }
      
      return adminUser;
    } catch (error) {
      console.error('Error ensuring admin user:', error);
      throw error;
    }
  }
}

// Export singleton instance
export const databaseUserStorage = new DatabaseUserStorage();