import { 
  type User, 
  type InsertUser, 
  type Contact, 
  type InsertContact,
  type Consultation, 
  type InsertConsultation,
  type TrainingVideo,
  type InsertTrainingVideo,
  type FileUpload,
  type Post,
  type InsertPost,
  type Comment,
  type InsertComment,
  type Like,
  type Follow,
  type Subscription,
  type InsertSubscription,
  type SystemMetric,
  type UsageAlert,
  type ComplianceScan,
  type InsertComplianceScan,
  type ComplianceIssue,
  type InsertComplianceIssue,
  type ComplianceAlert,
  type InsertComplianceAlert,
  type ComplianceReport,
  type InsertComplianceReport,
  type LiveTrainingSession,
  type InsertLiveTrainingSession,
  type TrainingRegistration,
  type InsertTrainingRegistration,
  type TrainingResource,
  type InsertTrainingResource,
  type OAuthToken,
  type InsertOAuthToken
} from "@shared/schema";
import { randomUUID } from "crypto";
import { databaseUserStorage } from './user-storage.js';
import { db } from './db.js';
import { oauthTokens, trainingVideos } from '@shared/schema';
import { eq, and, desc } from "drizzle-orm";

export interface IStorage {
  // User management
  getUser(id: string): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  createUser(user: InsertUser): Promise<User>;
  updateUserSettings(id: string, settings: { zoomPmi?: string; zoomPmiPassword?: string }): Promise<User>;

  // Social features - Posts
  createPost(post: InsertPost): Promise<Post>;
  getAllPosts(): Promise<Post[]>;
  getPost(id: string): Promise<Post | undefined>;
  updatePostCounts(postId: string, likesCount: number, commentsCount: number): Promise<void>;

  // Social features - Comments
  createComment(comment: InsertComment): Promise<Comment>;
  getCommentsByPost(postId: string): Promise<Comment[]>;

  // Social features - Likes
  createLike(postId: string, userId: string): Promise<Like>;
  deleteLike(postId: string, userId: string): Promise<void>;
  getLike(postId: string, userId: string): Promise<Like | undefined>;

  // Social features - Follows
  createFollow(followerId: string, followingId: string): Promise<Follow>;
  deleteFollow(followerId: string, followingId: string): Promise<void>;
  getFollows(userId: string): Promise<Follow[]>;

  // Premium subscriptions
  createSubscription(subscription: InsertSubscription): Promise<Subscription>;
  getSubscription(userId: string): Promise<Subscription | undefined>;
  updateSubscription(userId: string, status: string): Promise<void>;

  // Contact management
  createContact(contact: InsertContact): Promise<Contact>;
  getAllContacts(): Promise<Contact[]>;
  getContact(id: string): Promise<Contact | undefined>;

  // Consultation scheduling
  createConsultation(consultation: InsertConsultation): Promise<Consultation>;
  getAllConsultations(): Promise<Consultation[]>;
  getConsultation(id: string): Promise<Consultation | undefined>;

  // Training videos
  createTrainingVideo(video: InsertTrainingVideo): Promise<TrainingVideo>;
  getAllTrainingVideos(): Promise<TrainingVideo[]>;
  getPublicTrainingVideos(): Promise<TrainingVideo[]>;
  getTrainingVideo(id: string): Promise<TrainingVideo | undefined>;
  updateTrainingVideo(id: string, video: Partial<InsertTrainingVideo>): Promise<TrainingVideo | undefined>;
  deleteTrainingVideo(id: string): Promise<void>;

  // File uploads
  createFileUpload(fileData: Omit<FileUpload, 'id' | 'uploadedAt'>): Promise<FileUpload>;
  getAllFileUploads(): Promise<FileUpload[]>;
  getFileUpload(id: string): Promise<FileUpload | undefined>;

  // Section 508 Compliance monitoring
  createComplianceScan(scanData: Partial<ComplianceScan>): Promise<ComplianceScan>;
  updateComplianceScan(id: string, updates: Partial<ComplianceScan>): Promise<ComplianceScan | null>;
  getComplianceScan(id: string): Promise<ComplianceScan | null>;
  getComplianceScans(url?: string, limit?: number): Promise<ComplianceScan[]>;
  getLatestComplianceScan(url: string): Promise<ComplianceScan | null>;
  
  createComplianceIssue(issueData: Partial<ComplianceIssue>): Promise<ComplianceIssue>;
  getComplianceIssues(scanId?: string, status?: string): Promise<ComplianceIssue[]>;
  
  createComplianceAlert(alertData: Partial<ComplianceAlert>): Promise<ComplianceAlert>;
  getComplianceAlerts(onlyUnread?: boolean): Promise<ComplianceAlert[]>;
  markComplianceAlertAsRead(id: string): Promise<ComplianceAlert | null>;

  // Live Training Sessions with Zoom integration
  createLiveTrainingSession(sessionData: Partial<LiveTrainingSession>): Promise<LiveTrainingSession>;
  updateLiveTrainingSession(id: string, updates: Partial<LiveTrainingSession>): Promise<LiveTrainingSession | null>;
  getLiveTrainingSession(id: string): Promise<LiveTrainingSession | null>;
  getLiveTrainingSessions(category?: string, isPremiumOnly?: boolean): Promise<LiveTrainingSession[]>;
  getUpcomingTrainingSessions(): Promise<LiveTrainingSession[]>;
  
  createTrainingRegistration(registrationData: Partial<TrainingRegistration>): Promise<TrainingRegistration>;
  getTrainingRegistration(sessionId: string, userId: string): Promise<TrainingRegistration | null>;
  getTrainingRegistrations(sessionId?: string, userId?: string): Promise<TrainingRegistration[]>;
  updateTrainingRegistration(id: string, updates: Partial<TrainingRegistration>): Promise<TrainingRegistration | null>;
  
  createTrainingResource(resourceData: Partial<TrainingResource>): Promise<TrainingResource>;
  getTrainingResources(sessionId: string): Promise<TrainingResource[]>;

  // Training Interactive Features
  getTrainingPolls(sessionId: string): Promise<any[]>;
  createTrainingPoll(pollData: any): Promise<any>;
  voteOnTrainingPoll(pollId: string, optionIndex: number, userId: string): Promise<any>;
  
  getTrainingQAQuestions(sessionId: string): Promise<any[]>;
  createTrainingQAQuestion(questionData: any): Promise<any>;
  answerTrainingQAQuestion(questionId: string, answer: string, answeredBy: string): Promise<any>;
  
  getTrainingBreakoutRooms(sessionId: string): Promise<any[]>;
  createTrainingBreakoutRoom(roomData: any): Promise<any>;

  // Enhanced Consultation Management
  getConsultationAvailability(date: string): Promise<any[]>;
  updateConsultationStatus(id: string, status: string): Promise<Consultation | null>;
  rescheduleConsultation(id: string, newDate: string, newTime: string): Promise<Consultation | null>;
  completeConsultation(id: string, notes?: string): Promise<Consultation | null>;

  // Compliance Reports
  getComplianceReport(id: string): Promise<ComplianceReport | null>;

  // SMS Campaign Management
  getSMSCampaigns(): Promise<any[]>;
  getSMSCampaignStats(): Promise<any>;
  createSMSCampaign(campaign: any): Promise<any>;
  getSMSCampaign(id: string): Promise<any>;
  updateSMSCampaignStatus(id: string, status: string, totalRecipients?: number): Promise<void>;
  completeSMSCampaign(id: string, successCount: number, failureCount: number): Promise<void>;
  getUsersWithSMSOptIn(tierTarget?: string): Promise<any[]>;
  getActiveSMSSubscribers(): Promise<any[]>;
  createSMSLog(log: any): Promise<any>;
  getAllSMSSubscribers(): Promise<any[]>;
  createSMSSubscriber(subscriber: any): Promise<any>;
  optOutSMSSubscriber(phone: string): Promise<void>;

  // OAuth Token Management for persistent authentication
  upsertOAuthTokens(userId: string, provider: string, tokens: Omit<InsertOAuthToken, 'userId' | 'provider'>): Promise<OAuthToken>;
  getOAuthTokens(userId: string, provider: string): Promise<OAuthToken | undefined>;
  deleteOAuthTokens(userId: string, provider: string): Promise<void>;
}

export class MemStorage implements IStorage {
  private users: Map<string, User>;
  private contacts: Map<string, Contact>;
  private consultations: Map<string, Consultation>;
  private fileUploads: Map<string, FileUpload>;
  private posts: Map<string, Post>;
  private comments: Map<string, Comment>;
  private likes: Map<string, Like>;
  private follows: Map<string, Follow>;
  private subscriptions: Map<string, Subscription>;
  private systemMetrics: Map<string, SystemMetric>;
  private usageAlerts: Map<string, UsageAlert>;
  private complianceScans: ComplianceScan[];
  private complianceIssues: ComplianceIssue[];
  private complianceAlerts: ComplianceAlert[];
  private liveTrainingSessions: LiveTrainingSession[];
  private trainingRegistrations: TrainingRegistration[];
  private trainingResources: TrainingResource[];
  private requestCount: number;

  constructor() {
    this.users = new Map();
    this.contacts = new Map();
    this.consultations = new Map();
    this.fileUploads = new Map();
    this.posts = new Map();
    this.comments = new Map();
    this.likes = new Map();
    this.follows = new Map();
    this.subscriptions = new Map();
    this.systemMetrics = new Map();
    this.usageAlerts = new Map();
    this.complianceScans = [];
    this.complianceIssues = [];
    this.complianceAlerts = [];
    this.liveTrainingSessions = [];
    this.trainingRegistrations = [];
    this.trainingResources = [];
    this.requestCount = 0;

    // Initialize with some sample data
    this.initializeSampleData();
    this.initializeSampleMonitoring();
    
    // Ensure admin user exists in database for persistence
    this.ensureDatabaseAdmin();
  }

  private initializeSampleData() {
    // Sample admin user for posting
    const adminUser: User = {
      id: randomUUID(),
      username: "tmh_admin",
      password: "admin123",
      email: "admin@tmhglobal.com",
      firstName: "TMH",
      lastName: "Global",
      profileImageUrl: null,
      bio: "Professional accounting and business services for government contractors",
      company: "TMH Global LLC",
      isPremium: true,
      premiumExpiresAt: null,
      squareCustomerId: null,
      paypalCustomerId: null,
      phone: null,
      smsOptIn: false,
      smsPreferences: "all",
      zoomPmi: null,
      zoomPmiPassword: null,
      createdAt: new Date(),
      updatedAt: new Date()
    };
    this.users.set(adminUser.id, adminUser);

    // Sample posts
    const samplePosts: Post[] = [
      {
        id: randomUUID(),
        userId: adminUser.id,
        title: "Welcome to TMH Global Community",
        content: "We're excited to launch our exclusive community platform for government contractors. Here you'll find expert insights, compliance updates, and connect with fellow professionals in the industry.",
        imageUrl: null,
        category: "general",
        isPublic: true,
        isPremiumOnly: false,
        requiredTier: "free",
        likesCount: 12,
        commentsCount: 5,
        createdAt: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000), // 2 days ago
        updatedAt: new Date()
      },
      {
        id: randomUUID(),
        userId: adminUser.id,
        title: "SAM Registration Updates - Important Changes",
        content: "Recent updates to SAM.gov have changed some key requirements for registration. Make sure your business information is current and complete to avoid any delays in contract opportunities.",
        imageUrl: null,
        category: "sam-registration",
        isPublic: true,
        isPremiumOnly: false,
        requiredTier: "free",
        likesCount: 8,
        commentsCount: 3,
        createdAt: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000), // 1 day ago
        updatedAt: new Date()
      },
      {
        id: randomUUID(),
        userId: adminUser.id,
        title: "Advanced Compliance Strategies",
        content: "For our premium members: Advanced techniques for maintaining compliance across multiple contract vehicles and agencies. This includes documentation best practices and audit preparation.",
        imageUrl: null,
        category: "compliance",
        isPublic: true,
        isPremiumOnly: true,
        requiredTier: "premium",
        likesCount: 15,
        commentsCount: 8,
        createdAt: new Date(Date.now() - 3 * 60 * 60 * 1000), // 3 hours ago
        updatedAt: new Date()
      }
    ];

    samplePosts.forEach(post => {
      this.posts.set(post.id, post);
    });

    // Add sample live training sessions
    const sampleSessions: LiveTrainingSession[] = [
      {
        id: randomUUID(),
        title: "Section 508 Compliance Workshop",
        description: "Interactive workshop covering digital accessibility requirements for federal contractors",
        category: "compliance",
        sessionType: "webinar",
        
        zoomMeetingId: "123456789",
        zoomMeetingUuid: "abc123",
        zoomJoinUrl: "https://zoom.us/j/123456789",
        zoomStartUrl: "https://zoom.us/s/123456789",
        zoomPassword: "workshop2025",
        
        scheduledStartTime: new Date(Date.now() + 24 * 60 * 60 * 1000), // Tomorrow
        scheduledEndTime: new Date(Date.now() + 24 * 60 * 60 * 1000 + 2 * 60 * 60 * 1000), // Tomorrow + 2 hours
        duration: 120,
        timezone: "America/New_York",
        
        instructorId: "instructor-1",
        instructorName: "Sarah Johnson",
        isPremiumOnly: false,
        maxParticipants: 100,
        
        agenda: "1. Understanding Section 508\n2. WCAG 2.1 AA Standards\n3. Testing Tools and Techniques\n4. Common Compliance Issues\n5. Q&A Session",
        learningObjectives: [
          "Understand Section 508 requirements for government contractors",
          "Learn WCAG 2.1 AA compliance standards",
          "Master accessibility testing tools and techniques",
          "Identify and fix common accessibility issues"
        ],
        materials: null,
        recordingEnabled: true,
        recordingUrl: null,
        
        status: "scheduled",
        actualStartTime: null,
        actualEndTime: null,
        
        createdAt: new Date(),
        updatedAt: new Date()
      },
      {
        id: randomUUID(),
        title: "Government Contracting Basics",
        description: "Essential knowledge for small businesses entering government contracting",
        category: "business",
        sessionType: "meeting",
        
        zoomMeetingId: "987654321",
        zoomMeetingUuid: "xyz789",
        zoomJoinUrl: "https://zoom.us/j/987654321",
        zoomStartUrl: "https://zoom.us/s/987654321",
        zoomPassword: "govcon101",
        
        scheduledStartTime: new Date(Date.now() + 3 * 24 * 60 * 60 * 1000), // 3 days from now
        scheduledEndTime: new Date(Date.now() + 3 * 24 * 60 * 60 * 1000 + 90 * 60 * 1000), // 3 days + 90 minutes
        duration: 90,
        timezone: "America/New_York",
        
        instructorId: "instructor-2",
        instructorName: "Michael Chen",
        isPremiumOnly: true,
        maxParticipants: 50,
        
        agenda: "1. Introduction to Government Contracting\n2. S.A.M. Registration Process\n3. NAICS Codes and Set-Asides\n4. Proposal Writing Basics\n5. Compliance Requirements",
        learningObjectives: [
          "Navigate the government contracting landscape",
          "Complete S.A.M. registration successfully",
          "Understand NAICS codes and set-aside programs",
          "Write competitive proposals"
        ],
        materials: null,
        recordingEnabled: true,
        recordingUrl: null,
        
        status: "scheduled",
        actualStartTime: null,
        actualEndTime: null,
        
        createdAt: new Date(),
        updatedAt: new Date()
      },
      {
        id: randomUUID(),
        title: "Tax Preparation for Contractors - COMPLETED",
        description: "Year-end tax strategies and deductions for government contractors",
        category: "tax",
        sessionType: "webinar",
        
        zoomMeetingId: "555666777",
        zoomMeetingUuid: "completed123",
        zoomJoinUrl: "https://zoom.us/j/555666777",
        zoomStartUrl: "https://zoom.us/s/555666777",
        zoomPassword: "tax2024",
        
        scheduledStartTime: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000), // 7 days ago
        scheduledEndTime: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000 + 2 * 60 * 60 * 1000), // 7 days ago + 2 hours
        duration: 120,
        timezone: "America/New_York",
        
        instructorId: "instructor-3",
        instructorName: "Jennifer Martinez",
        isPremiumOnly: false,
        maxParticipants: 200,
        
        agenda: "1. Contractor Tax Obligations\n2. Deductible Business Expenses\n3. Quarterly Payment Strategies\n4. Record Keeping Best Practices",
        learningObjectives: [
          "Understand contractor tax obligations",
          "Maximize deductible business expenses",
          "Plan quarterly tax payments effectively",
          "Implement proper record keeping"
        ],
        materials: null,
        recordingEnabled: true,
        recordingUrl: "https://zoom.us/rec/share/tax-recording-2024",
        
        status: "completed",
        actualStartTime: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000),
        actualEndTime: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000 + 2 * 60 * 60 * 1000),
        
        createdAt: new Date(Date.now() - 14 * 24 * 60 * 60 * 1000),
        updatedAt: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000)
      }
    ];

    this.liveTrainingSessions = sampleSessions;
  }

  // Ensure admin user exists in database for PMI persistence
  private async ensureDatabaseAdmin() {
    try {
      await databaseUserStorage.ensureAdminUser();
      console.log('✅ Admin user verified in database for PMI persistence');
    } catch (error) {
      console.error('❌ Failed to ensure admin user in database:', error);
    }
  }

  // User methods - Now using database storage for persistence
  async getUser(id: string): Promise<User | undefined> {
    return await databaseUserStorage.getUser(id);
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    return await databaseUserStorage.getUserByUsername(username);
  }

  async createUser(insertUser: InsertUser): Promise<User> {
    return await databaseUserStorage.createUser(insertUser);
  }

  async updateUserSettings(id: string, settings: { zoomPmi?: string; zoomPmiPassword?: string }): Promise<User> {
    return await databaseUserStorage.updateUserSettings(id, settings);
  }

  // Social features - Posts
  async createPost(insertPost: InsertPost): Promise<Post> {
    const id = randomUUID();
    const post: Post = {
      ...insertPost,
      id,
      imageUrl: insertPost.imageUrl || null,
      category: insertPost.category || "general",
      isPublic: insertPost.isPublic ?? true,
      isPremiumOnly: insertPost.isPremiumOnly ?? false,
      likesCount: 0,
      commentsCount: 0,
      createdAt: new Date(),
      updatedAt: new Date()
    };
    this.posts.set(id, post);
    return post;
  }

  async getAllPosts(): Promise<Post[]> {
    return Array.from(this.posts.values()).sort(
      (a, b) => (b.createdAt?.getTime() || 0) - (a.createdAt?.getTime() || 0)
    );
  }

  async getPost(id: string): Promise<Post | undefined> {
    return this.posts.get(id);
  }

  async updatePostCounts(postId: string, likesCount: number, commentsCount: number): Promise<void> {
    const post = this.posts.get(postId);
    if (post) {
      post.likesCount = likesCount;
      post.commentsCount = commentsCount;
      post.updatedAt = new Date();
      this.posts.set(postId, post);
    }
  }

  // Social features - Comments
  async createComment(insertComment: InsertComment): Promise<Comment> {
    const id = randomUUID();
    const comment: Comment = {
      ...insertComment,
      id,
      createdAt: new Date()
    };
    this.comments.set(id, comment);

    // Update post comment count
    const comments = await this.getCommentsByPost(insertComment.postId);
    const post = await this.getPost(insertComment.postId);
    if (post) {
      await this.updatePostCounts(insertComment.postId, post.likesCount || 0, comments.length);
    }

    return comment;
  }

  async getCommentsByPost(postId: string): Promise<Comment[]> {
    return Array.from(this.comments.values())
      .filter(comment => comment.postId === postId)
      .sort((a, b) => (a.createdAt?.getTime() || 0) - (b.createdAt?.getTime() || 0));
  }

  // Social features - Likes
  async createLike(postId: string, userId: string): Promise<Like> {
    const id = randomUUID();
    const like: Like = {
      id,
      postId,
      userId,
      createdAt: new Date()
    };
    this.likes.set(id, like);

    // Update post like count
    const likes = Array.from(this.likes.values()).filter(l => l.postId === postId);
    const post = await this.getPost(postId);
    if (post) {
      await this.updatePostCounts(postId, likes.length, post.commentsCount || 0);
    }

    return like;
  }

  async deleteLike(postId: string, userId: string): Promise<void> {
    const like = Array.from(this.likes.values()).find(
      l => l.postId === postId && l.userId === userId
    );
    if (like) {
      this.likes.delete(like.id);

      // Update post like count
      const likes = Array.from(this.likes.values()).filter(l => l.postId === postId);
      const post = await this.getPost(postId);
      if (post) {
        await this.updatePostCounts(postId, likes.length, post.commentsCount || 0);
      }
    }
  }

  async getLike(postId: string, userId: string): Promise<Like | undefined> {
    return Array.from(this.likes.values()).find(
      like => like.postId === postId && like.userId === userId
    );
  }

  // Social features - Follows
  async createFollow(followerId: string, followingId: string): Promise<Follow> {
    const id = randomUUID();
    const follow: Follow = {
      id,
      followerId,
      followingId,
      createdAt: new Date()
    };
    this.follows.set(id, follow);
    return follow;
  }

  async deleteFollow(followerId: string, followingId: string): Promise<void> {
    const follow = Array.from(this.follows.values()).find(
      f => f.followerId === followerId && f.followingId === followingId
    );
    if (follow) {
      this.follows.delete(follow.id);
    }
  }

  async getFollows(userId: string): Promise<Follow[]> {
    return Array.from(this.follows.values()).filter(
      follow => follow.followerId === userId
    );
  }

  // Premium subscriptions
  async createSubscription(insertSubscription: InsertSubscription): Promise<Subscription> {
    const id = randomUUID();
    const subscription: Subscription = {
      ...insertSubscription,
      id,
      subscriptionId: insertSubscription.subscriptionId || null,
      status: insertSubscription.status || "active",
      planType: insertSubscription.planType || "premium",
      currency: insertSubscription.currency || "USD",
      startDate: new Date(),
      expiresAt: insertSubscription.expiresAt || null,
      createdAt: new Date()
    };
    this.subscriptions.set(id, subscription);
    return subscription;
  }

  async getSubscription(userId: string): Promise<Subscription | undefined> {
    return Array.from(this.subscriptions.values()).find(
      sub => sub.userId === userId
    );
  }

  async updateSubscription(userId: string, status: string): Promise<void> {
    const subscription = await this.getSubscription(userId);
    if (subscription) {
      subscription.status = status;
      this.subscriptions.set(subscription.id, subscription);
    }
  }

  // Contact methods
  async createContact(insertContact: InsertContact): Promise<Contact> {
    const id = randomUUID();
    const contact: Contact = { 
      ...insertContact, 
      serviceInterest: insertContact.serviceInterest || null,
      phone: insertContact.phone || null,
      id, 
      createdAt: new Date() 
    };
    this.contacts.set(id, contact);
    return contact;
  }

  async getAllContacts(): Promise<Contact[]> {
    return Array.from(this.contacts.values()).sort(
      (a, b) => (b.createdAt?.getTime() || 0) - (a.createdAt?.getTime() || 0)
    );
  }

  async getContact(id: string): Promise<Contact | undefined> {
    return this.contacts.get(id);
  }

  // Consultation methods
  async createConsultation(insertConsultation: InsertConsultation): Promise<Consultation> {
    const id = randomUUID();
    const consultation: Consultation = { 
      ...insertConsultation, 
      phone: insertConsultation.phone || null,
      message: insertConsultation.message || null,
      companyName: insertConsultation.companyName || null,
      preferredDate: insertConsultation.preferredDate || null,
      preferredTime: insertConsultation.preferredTime || null,
      id, 
      createdAt: new Date() 
    };
    this.consultations.set(id, consultation);
    return consultation;
  }

  async getAllConsultations(): Promise<Consultation[]> {
    return Array.from(this.consultations.values()).sort(
      (a, b) => (b.createdAt?.getTime() || 0) - (a.createdAt?.getTime() || 0)
    );
  }

  async getConsultation(id: string): Promise<Consultation | undefined> {
    return this.consultations.get(id);
  }

  // Training video methods - using database
  async createTrainingVideo(insertVideo: InsertTrainingVideo): Promise<TrainingVideo> {
    const [video] = await db.insert(trainingVideos).values(insertVideo).returning();
    return video;
  }

  async getAllTrainingVideos(): Promise<TrainingVideo[]> {
    return await db.select().from(trainingVideos).orderBy(desc(trainingVideos.createdAt));
  }

  async getPublicTrainingVideos(): Promise<TrainingVideo[]> {
    return await db.select().from(trainingVideos)
      .where(eq(trainingVideos.isPublic, true))
      .orderBy(desc(trainingVideos.createdAt));
  }

  async getTrainingVideo(id: string): Promise<TrainingVideo | undefined> {
    const [video] = await db.select().from(trainingVideos).where(eq(trainingVideos.id, id));
    return video;
  }

  async updateTrainingVideo(id: string, video: Partial<InsertTrainingVideo>): Promise<TrainingVideo | undefined> {
    const [updatedVideo] = await db.update(trainingVideos)
      .set(video)
      .where(eq(trainingVideos.id, id))
      .returning();
    return updatedVideo;
  }

  async deleteTrainingVideo(id: string): Promise<void> {
    await db.delete(trainingVideos).where(eq(trainingVideos.id, id));
  }

  // Monitoring methods
  async recordMetric(metricType: string, metricName: string, value: number, metadata?: string): Promise<SystemMetric> {
    const id = randomUUID();
    const metric: SystemMetric = {
      id,
      metricType,
      metricName,
      value,
      metadata: metadata || null,
      timestamp: new Date()
    };
    this.systemMetrics.set(id, metric);
    return metric;
  }

  async getMetrics(type?: string, hours?: number): Promise<SystemMetric[]> {
    let metrics = Array.from(this.systemMetrics.values());
    
    if (type) {
      metrics = metrics.filter(m => m.metricType === type);
    }
    
    if (hours) {
      const cutoff = new Date(Date.now() - hours * 60 * 60 * 1000);
      metrics = metrics.filter(m => m.timestamp && m.timestamp > cutoff);
    }
    
    return metrics.sort((a, b) => (b.timestamp ? b.timestamp.getTime() : 0) - (a.timestamp ? a.timestamp.getTime() : 0));
  }

  async createAlert(alertType: string, resource: string, currentValue: number, limitValue: number, threshold: number, message: string): Promise<UsageAlert> {
    const id = randomUUID();
    const alert: UsageAlert = {
      id,
      alertType,
      resource,
      currentValue,
      limitValue,
      threshold,
      message,
      isResolved: false,
      createdAt: new Date()
    };
    this.usageAlerts.set(id, alert);
    return alert;
  }

  async getAlerts(activeOnly: boolean = false): Promise<UsageAlert[]> {
    let alerts = Array.from(this.usageAlerts.values());
    
    if (activeOnly) {
      alerts = alerts.filter(a => !a.isResolved);
    }
    
    return alerts.sort((a, b) => (b.createdAt ? b.createdAt.getTime() : 0) - (a.createdAt ? a.createdAt.getTime() : 0));
  }

  async incrementRequestCount(): Promise<void> {
    this.requestCount++;
    
    // Record metric every 100 requests
    if (this.requestCount % 100 === 0) {
      await this.recordMetric('api_requests', 'total_requests', this.requestCount);
    }
  }

  async getRequestCount(): Promise<number> {
    return this.requestCount;
  }

  async getSummaryStats(): Promise<{
    totalUsers: number;
    totalPosts: number;
    totalComments: number;
    apiRequestsToday: number;
    storageUsed: number;
    premiumUsers: number;
  }> {
    const totalUsers = this.users.size;
    const totalPosts = this.posts.size;
    const totalComments = this.comments.size;
    const premiumUsers = Array.from(this.users.values()).filter(u => u.isPremium).length;
    
    // Mock storage calculation (in bytes)
    const storageUsed = (totalUsers * 1024) + (totalPosts * 2048) + (totalComments * 512);
    
    return {
      totalUsers,
      totalPosts,
      totalComments,
      apiRequestsToday: this.requestCount,
      storageUsed,
      premiumUsers
    };
  }

  private async initializeSampleMonitoring() {
    // Add some sample metrics
    await this.recordMetric('api_requests', 'total_requests', 1250, '{"endpoint": "/api/posts", "method": "GET"}');
    await this.recordMetric('user_activity', 'new_registrations', 23, '{"source": "organic"}');
    await this.recordMetric('user_activity', 'active_users', 187, '{"timeframe": "daily"}');
    await this.recordMetric('storage_usage', 'database_size', 2.1 * 1024 * 1024, '{"table": "posts"}');
    
    // Add sample alerts for demo
    await this.createAlert(
      'high_usage',
      'api_requests', 
      1250,
      1500,
      83,
      'API request volume is approaching daily limits. Consider optimizing client-side caching.'
    );

    // Set request count to show meaningful data
    this.requestCount = 1250;
  }

  // File upload methods
  async createFileUpload(fileData: Omit<FileUpload, 'id' | 'uploadedAt'>): Promise<FileUpload> {
    const id = randomUUID();
    const fileUpload: FileUpload = { 
      ...fileData, 
      id, 
      uploadedAt: new Date() 
    };
    this.fileUploads.set(id, fileUpload);
    return fileUpload;
  }

  async getAllFileUploads(): Promise<FileUpload[]> {
    return Array.from(this.fileUploads.values()).sort(
      (a, b) => (b.uploadedAt?.getTime() || 0) - (a.uploadedAt?.getTime() || 0)
    );
  }

  async getFileUpload(id: string): Promise<FileUpload | undefined> {
    return this.fileUploads.get(id);
  }

  // Section 508 Compliance operations
  async createComplianceScan(scanData: Partial<ComplianceScan>): Promise<ComplianceScan> {
    const scan: ComplianceScan = {
      id: randomUUID(),
      url: scanData.url || '',
      scanType: scanData.scanType || 'manual',
      status: scanData.status || 'running',
      overallScore: scanData.overallScore || null,
      totalIssues: scanData.totalIssues || 0,
      criticalIssues: scanData.criticalIssues || 0,
      warningIssues: scanData.warningIssues || 0,
      infoIssues: scanData.infoIssues || 0,
      scanDuration: scanData.scanDuration || null,
      userAgent: scanData.userAgent || null,
      createdAt: new Date(),
      completedAt: scanData.completedAt || null
    };
    this.complianceScans.push(scan);
    return scan;
  }

  async updateComplianceScan(id: string, updates: Partial<ComplianceScan>): Promise<ComplianceScan | null> {
    const scanIndex = this.complianceScans.findIndex(s => s.id === id);
    if (scanIndex === -1) return null;

    this.complianceScans[scanIndex] = { ...this.complianceScans[scanIndex], ...updates };
    return this.complianceScans[scanIndex];
  }

  async getComplianceScan(id: string): Promise<ComplianceScan | null> {
    return this.complianceScans.find(s => s.id === id) || null;
  }

  async getComplianceScans(url?: string, limit: number = 50): Promise<ComplianceScan[]> {
    let scans = [...this.complianceScans];
    if (url) {
      scans = scans.filter(s => s.url === url);
    }
    return scans
      .sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime())
      .slice(0, limit);
  }

  async getLatestComplianceScan(url: string): Promise<ComplianceScan | null> {
    const scans = this.complianceScans
      .filter(s => s.url === url && s.status === 'completed')
      .sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
    return scans[0] || null;
  }

  async createComplianceIssue(issueData: Partial<ComplianceIssue>): Promise<ComplianceIssue> {
    const issue: ComplianceIssue = {
      id: randomUUID(),
      scanId: issueData.scanId || '',
      ruleId: issueData.ruleId || '',
      severity: issueData.severity || 'info',
      wcagLevel: issueData.wcagLevel || 'A',
      section508Rule: issueData.section508Rule || null,
      element: issueData.element || null,
      message: issueData.message || '',
      helpUrl: issueData.helpUrl || null,
      xpath: issueData.xpath || null,
      html: issueData.html || null,
      status: issueData.status || 'open',
      resolvedAt: issueData.resolvedAt || null,
      resolvedBy: issueData.resolvedBy || null,
      createdAt: new Date()
    };
    this.complianceIssues.push(issue);
    return issue;
  }

  async getComplianceIssues(scanId?: string, status?: string): Promise<ComplianceIssue[]> {
    let issues = [...this.complianceIssues];
    if (scanId) {
      issues = issues.filter(i => i.scanId === scanId);
    }
    if (status) {
      issues = issues.filter(i => i.status === status);
    }
    return issues.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }

  async createComplianceAlert(alertData: Partial<ComplianceAlert>): Promise<ComplianceAlert> {
    const alert: ComplianceAlert = {
      id: randomUUID(),
      alertType: alertData.alertType || 'new_violations',
      severity: alertData.severity || 'medium',
      title: alertData.title || '',
      message: alertData.message || '',
      scanId: alertData.scanId || null,
      url: alertData.url || null,
      isRead: alertData.isRead || false,
      isResolved: alertData.isResolved || false,
      resolvedAt: alertData.resolvedAt || null,
      resolvedBy: alertData.resolvedBy || null,
      createdAt: new Date()
    };
    this.complianceAlerts.push(alert);
    return alert;
  }

  async getComplianceAlerts(onlyUnread: boolean = false): Promise<ComplianceAlert[]> {
    let alerts = [...this.complianceAlerts];
    if (onlyUnread) {
      alerts = alerts.filter(a => !a.isRead);
    }
    return alerts.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }

  async markComplianceAlertAsRead(id: string): Promise<ComplianceAlert | null> {
    const alertIndex = this.complianceAlerts.findIndex(a => a.id === id);
    if (alertIndex === -1) return null;

    this.complianceAlerts[alertIndex].isRead = true;
    return this.complianceAlerts[alertIndex];
  }

  // Live Training Sessions with Zoom integration
  async createLiveTrainingSession(sessionData: Partial<LiveTrainingSession>): Promise<LiveTrainingSession> {
    const session: LiveTrainingSession = {
      id: randomUUID(),
      title: sessionData.title || '',
      description: sessionData.description || null,
      category: sessionData.category || 'compliance',
      sessionType: sessionData.sessionType || 'meeting',
      
      // Zoom integration fields
      zoomMeetingId: sessionData.zoomMeetingId || null,
      zoomMeetingUuid: sessionData.zoomMeetingUuid || null,
      zoomJoinUrl: sessionData.zoomJoinUrl || null,
      zoomStartUrl: sessionData.zoomStartUrl || null,
      zoomPassword: sessionData.zoomPassword || null,
      
      // Schedule and duration
      scheduledStartTime: sessionData.scheduledStartTime || new Date(),
      scheduledEndTime: sessionData.scheduledEndTime || new Date(),
      duration: sessionData.duration || 60,
      timezone: sessionData.timezone || 'America/New_York',
      
      // Instructor and access
      instructorId: sessionData.instructorId || '',
      instructorName: sessionData.instructorName || '',
      isPremiumOnly: sessionData.isPremiumOnly || false,
      maxParticipants: sessionData.maxParticipants || 100,
      
      // Session details
      agenda: sessionData.agenda || null,
      learningObjectives: sessionData.learningObjectives || null,
      materials: sessionData.materials || null,
      recordingEnabled: sessionData.recordingEnabled || true,
      recordingUrl: sessionData.recordingUrl || null,
      
      // Status and management
      status: sessionData.status || 'scheduled',
      actualStartTime: sessionData.actualStartTime || null,
      actualEndTime: sessionData.actualEndTime || null,
      
      // Metadata
      createdAt: new Date(),
      updatedAt: new Date()
    };
    this.liveTrainingSessions.push(session);
    return session;
  }

  async updateLiveTrainingSession(id: string, updates: Partial<LiveTrainingSession>): Promise<LiveTrainingSession | null> {
    const sessionIndex = this.liveTrainingSessions.findIndex(s => s.id === id);
    if (sessionIndex === -1) return null;

    this.liveTrainingSessions[sessionIndex] = { 
      ...this.liveTrainingSessions[sessionIndex], 
      ...updates,
      updatedAt: new Date()
    };
    return this.liveTrainingSessions[sessionIndex];
  }

  async getLiveTrainingSession(id: string): Promise<LiveTrainingSession | null> {
    return this.liveTrainingSessions.find(s => s.id === id) || null;
  }

  async getLiveTrainingSessions(category?: string, isPremiumOnly?: boolean): Promise<LiveTrainingSession[]> {
    let sessions = [...this.liveTrainingSessions];
    
    if (category) {
      sessions = sessions.filter(s => s.category === category);
    }
    
    if (isPremiumOnly !== undefined) {
      sessions = sessions.filter(s => s.isPremiumOnly === isPremiumOnly);
    }
    
    return sessions.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }

  async getUpcomingTrainingSessions(): Promise<LiveTrainingSession[]> {
    const now = new Date();
    return this.liveTrainingSessions
      .filter(s => s.scheduledStartTime > now && s.status === 'scheduled')
      .sort((a, b) => a.scheduledStartTime.getTime() - b.scheduledStartTime.getTime());
  }

  async createTrainingRegistration(registrationData: Partial<TrainingRegistration>): Promise<TrainingRegistration> {
    const registration: TrainingRegistration = {
      id: randomUUID(),
      sessionId: registrationData.sessionId || '',
      userId: registrationData.userId || '',
      
      // Registration details
      registeredAt: new Date(),
      status: registrationData.status || 'registered',
      
      // Zoom integration
      zoomRegistrantId: registrationData.zoomRegistrantId || null,
      zoomJoinUrl: registrationData.zoomJoinUrl || null,
      
      // Attendance tracking
      joinedAt: registrationData.joinedAt || null,
      leftAt: registrationData.leftAt || null,
      attendanceDuration: registrationData.attendanceDuration || null,
      
      // Feedback and completion
      completed: registrationData.completed || false,
      feedbackRating: registrationData.feedbackRating || null,
      feedbackComments: registrationData.feedbackComments || null,
      certificateIssued: registrationData.certificateIssued || false,
      
      // Metadata
      createdAt: new Date(),
      updatedAt: new Date()
    };
    this.trainingRegistrations.push(registration);
    return registration;
  }

  async getTrainingRegistration(sessionId: string, userId: string): Promise<TrainingRegistration | null> {
    return this.trainingRegistrations.find(r => r.sessionId === sessionId && r.userId === userId) || null;
  }

  async getTrainingRegistrations(sessionId?: string, userId?: string): Promise<TrainingRegistration[]> {
    let registrations = [...this.trainingRegistrations];
    
    if (sessionId) {
      registrations = registrations.filter(r => r.sessionId === sessionId);
    }
    
    if (userId) {
      registrations = registrations.filter(r => r.userId === userId);
    }
    
    return registrations.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }

  async updateTrainingRegistration(id: string, updates: Partial<TrainingRegistration>): Promise<TrainingRegistration | null> {
    const registrationIndex = this.trainingRegistrations.findIndex(r => r.id === id);
    if (registrationIndex === -1) return null;

    this.trainingRegistrations[registrationIndex] = {
      ...this.trainingRegistrations[registrationIndex],
      ...updates,
      updatedAt: new Date()
    };
    return this.trainingRegistrations[registrationIndex];
  }

  async createTrainingResource(resourceData: Partial<TrainingResource>): Promise<TrainingResource> {
    const resource: TrainingResource = {
      id: randomUUID(),
      sessionId: resourceData.sessionId || '',
      
      // Resource details
      title: resourceData.title || '',
      description: resourceData.description || null,
      resourceType: resourceData.resourceType || 'document',
      fileUrl: resourceData.fileUrl || null,
      fileSize: resourceData.fileSize || null,
      mimeType: resourceData.mimeType || null,
      
      // Access control
      availableFrom: resourceData.availableFrom || new Date(),
      availableUntil: resourceData.availableUntil || null,
      downloadAllowed: resourceData.downloadAllowed || true,
      
      // Metadata
      uploadedBy: resourceData.uploadedBy || '',
      createdAt: new Date(),
      updatedAt: new Date()
    };
    this.trainingResources.push(resource);
    return resource;
  }

  async getTrainingResources(sessionId: string): Promise<TrainingResource[]> {
    return this.trainingResources
      .filter(r => r.sessionId === sessionId)
      .sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }

  // ===== SMS CAMPAIGN MANAGEMENT =====
  
  async getSMSCampaigns(): Promise<any[]> {
    // For now, return empty array - will be implemented with database
    return [];
  }

  async getSMSCampaignStats(): Promise<any> {
    return {
      totalCampaigns: 0,
      totalSent: 0,
      activeSubscribers: 0,
      avgDeliveryRate: 0
    };
  }

  async createSMSCampaign(campaign: any): Promise<any> {
    const newCampaign = {
      id: randomUUID(),
      ...campaign,
      createdAt: new Date(),
      updatedAt: new Date(),
      totalRecipients: 0,
      successfulSends: 0,
      failedSends: 0
    };
    
    // For now, just return the campaign - will store in database
    return newCampaign;
  }

  async getSMSCampaign(id: string): Promise<any> {
    // For now, return null - will be implemented with database
    return null;
  }

  async updateSMSCampaignStatus(id: string, status: string, totalRecipients?: number): Promise<void> {
    // For now, no-op - will be implemented with database
  }

  async completeSMSCampaign(id: string, successCount: number, failureCount: number): Promise<void> {
    // For now, no-op - will be implemented with database
  }

  async getUsersWithSMSOptIn(tierTarget?: string): Promise<any[]> {
    const users = Array.from(this.users.values());
    return users.filter(user => 
      user.phone && 
      user.smsOptIn && 
      (!tierTarget || this.getUserTier(user) === tierTarget)
    );
  }

  async getActiveSMSSubscribers(): Promise<any[]> {
    // For now, return empty array - will be implemented with database
    return [];
  }

  async createSMSLog(log: any): Promise<any> {
    const newLog = {
      id: randomUUID(),
      ...log,
      sentAt: new Date()
    };
    return newLog;
  }

  async getAllSMSSubscribers(): Promise<any[]> {
    // For now, return empty array - will be implemented with database
    return [];
  }

  async createSMSSubscriber(subscriber: any): Promise<any> {
    const newSubscriber = {
      id: randomUUID(),
      ...subscriber,
      optInDate: new Date(),
      createdAt: new Date()
    };
    return newSubscriber;
  }

  async optOutSMSSubscriber(phone: string): Promise<void> {
    // For now, no-op - will be implemented with database
  }

  // ===== TRAINING INTERACTIVE FEATURES =====
  
  async getTrainingPolls(sessionId: string): Promise<any[]> {
    // For now, return empty array - will be implemented with database
    return [];
  }

  async createTrainingPoll(pollData: any): Promise<any> {
    const newPoll = {
      id: randomUUID(),
      ...pollData,
      createdAt: new Date(),
      votes: []
    };
    return newPoll;
  }

  async voteOnTrainingPoll(pollId: string, optionIndex: number, userId: string): Promise<any> {
    // For now, return success - will be implemented with database
    return { success: true, pollId, optionIndex, userId };
  }

  async getTrainingQAQuestions(sessionId: string): Promise<any[]> {
    // For now, return empty array - will be implemented with database
    return [];
  }

  async createTrainingQAQuestion(questionData: any): Promise<any> {
    const newQuestion = {
      id: randomUUID(),
      ...questionData,
      createdAt: new Date(),
      status: 'pending'
    };
    return newQuestion;
  }

  async answerTrainingQAQuestion(questionId: string, answer: string, answeredBy: string): Promise<any> {
    // For now, return updated question - will be implemented with database
    return {
      id: questionId,
      answer,
      answeredBy,
      answeredAt: new Date(),
      status: 'answered'
    };
  }

  async getTrainingBreakoutRooms(sessionId: string): Promise<any[]> {
    // For now, return empty array - will be implemented with database
    return [];
  }

  async createTrainingBreakoutRoom(roomData: any): Promise<any> {
    const newRoom = {
      id: randomUUID(),
      ...roomData,
      createdAt: new Date(),
      participants: []
    };
    return newRoom;
  }

  // ===== ENHANCED CONSULTATION MANAGEMENT =====
  
  async getConsultationAvailability(date: string): Promise<any[]> {
    const requestedDate = new Date(date);
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Reset time to midnight for date comparison
    
    // Calculate 1 day from today (more reasonable)
    const minDate = new Date(today);
    minDate.setDate(today.getDate() + 1);
    
    // Check if requested date is at least 1 day in the future
    if (requestedDate < minDate) {
      return []; // No availability if too soon (same-day booking not allowed)
    }
    
    // Check if requested date is on a closed day (Tuesday=2, Saturday=6, Sunday=0)
    const dayOfWeek = requestedDate.getDay();
    if (dayOfWeek === 0 || dayOfWeek === 2 || dayOfWeek === 6) {
      return []; // No availability on closed days
    }
    
    // Return available time slots for valid days
    const availableSlots = [
      { time: "09:00", available: true },
      { time: "10:00", available: true },
      { time: "11:00", available: true },
      { time: "13:00", available: true },
      { time: "14:00", available: true },
      { time: "15:00", available: true },
      { time: "16:00", available: true }
    ];
    
    // TODO: Check existing consultations and mark slots as unavailable if booked
    // This would require querying the consultations for the specific date
    
    return availableSlots;
  }

  async updateConsultationStatus(id: string, status: string): Promise<Consultation | null> {
    const consultation = this.consultations.get(id);
    if (consultation) {
      const updated = { ...consultation, status, updatedAt: new Date() };
      this.consultations.set(id, updated as Consultation);
      return updated as Consultation;
    }
    return null;
  }

  async rescheduleConsultation(id: string, newDate: string, newTime: string): Promise<Consultation | null> {
    const consultation = this.consultations.get(id);
    if (consultation) {
      const updated = { 
        ...consultation, 
        preferredDate: newDate, 
        preferredTime: newTime,
        updatedAt: new Date() 
      };
      this.consultations.set(id, updated as Consultation);
      return updated as Consultation;
    }
    return null;
  }

  async completeConsultation(id: string, notes?: string): Promise<Consultation | null> {
    const consultation = this.consultations.get(id);
    if (consultation) {
      const updated = { 
        ...consultation, 
        status: 'completed',
        notes,
        completedAt: new Date(),
        updatedAt: new Date() 
      };
      this.consultations.set(id, updated as Consultation);
      return updated as Consultation;
    }
    return null;
  }

  // ===== COMPLIANCE REPORTS =====
  
  async getComplianceReport(id: string): Promise<ComplianceReport | null> {
    // For now, return null - will be implemented with database
    return null;
  }

  // ===== OAUTH TOKEN MANAGEMENT =====
  
  async upsertOAuthTokens(userId: string, provider: string, tokens: Omit<InsertOAuthToken, 'userId' | 'provider'>): Promise<OAuthToken> {
    const tokenData: InsertOAuthToken = {
      userId,
      provider,
      ...tokens
    };

    // Upsert using database - insert or update if exists
    const [result] = await db
      .insert(oauthTokens)
      .values(tokenData)
      .onConflictDoUpdate({
        target: [oauthTokens.userId, oauthTokens.provider],
        set: {
          accessToken: tokenData.accessToken,
          refreshToken: tokenData.refreshToken,
          expiresAt: tokenData.expiresAt,
          scope: tokenData.scope,
          accountId: tokenData.accountId,
          metadata: tokenData.metadata,
          updatedAt: new Date()
        }
      })
      .returning();

    return result;
  }

  async getOAuthTokens(userId: string, provider: string): Promise<OAuthToken | undefined> {
    const [result] = await db
      .select()
      .from(oauthTokens)
      .where(and(
        eq(oauthTokens.userId, userId),
        eq(oauthTokens.provider, provider)
      ))
      .limit(1);

    return result;
  }

  async deleteOAuthTokens(userId: string, provider: string): Promise<void> {
    await db
      .delete(oauthTokens)
      .where(and(
        eq(oauthTokens.userId, userId),
        eq(oauthTokens.provider, provider)
      ));
  }

  private getUserTier(user: User): string {
    // Check subscription status to determine tier
    // This is a simplified version - full implementation would check subscriptions table
    if (user.isPremium) {
      return 'premium';
    }
    return 'free';
  }
}

export const storage = new MemStorage();
