import type { Express, Request } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import { insertContactSchema, insertConsultationSchema, insertTrainingVideoSchema, insertPostSchema, insertCommentSchema, smsCampaigns, smsSubscribers, smsLogs } from "@shared/schema";
import { ringCentralService } from "./integrations/ringcentral";
import { squareService } from "./payments/square";
import { paypalService } from "./payments/paypal";
import { zoomService } from "./integrations/zoom";
import { complianceScanner } from "./compliance/scanner";
import { googleCalendarService } from "./integrations/google-calendar";
import { gmailService } from "./integrations/gmail";
import multer from "multer";
import path from "path";
import { randomUUID } from "crypto";
import { createPaypalOrder, capturePaypalOrder, loadPaypalDefault } from "./paypal";
import bcrypt from "bcryptjs";

// Session type enhancement for Google Calendar and RingCentral integration
declare module "express-session" {
  interface SessionData {
    userId?: string;
    isAdmin?: boolean;
    googleTokens?: {
      access_token: string;
      refresh_token?: string;
      scope: string;
      token_type: string;
      expiry_date?: number;
    };
    ringCentralTokens?: {
      accessToken: string;
      refreshToken?: string;
      expiresAt: Date;
    };
  }
}

// Configure multer for file uploads
const upload = multer({
  storage: multer.diskStorage({
    destination: (req, file, cb) => {
      cb(null, 'uploads/');
    },
    filename: (req, file, cb) => {
      const uniqueName = `${randomUUID()}-${file.originalname}`;
      cb(null, uniqueName);
    }
  }),
  limits: {
    fileSize: 10 * 1024 * 1024, // 10MB limit
  },
  fileFilter: (req, file, cb) => {
    // Allow common document and image types
    const allowedTypes = [
      'application/pdf',
      'application/msword',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
      'application/vnd.ms-excel',
      'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
      'image/jpeg',
      'image/png',
      'image/gif',
      'text/plain'
    ];
    
    if (allowedTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error('Invalid file type. Please upload documents, images, or text files only.'));
    }
  }
});

// Middleware to track API requests
const trackApiRequest = async (req: any, res: any, next: any) => {
  await storage.incrementRequestCount();
  next();
};

export async function registerRoutes(app: Express): Promise<Server> {
  // Apply tracking middleware to all API routes
  app.use('/api', trackApiRequest);

  // Admin Authentication Routes
  app.post("/api/auth/login", async (req, res) => {
    try {
      const { username, password } = req.body;
      
      if (!username || !password) {
        return res.status(400).json({ message: "Username and password required" });
      }

      const user = await storage.getUserByUsername(username);
      
      if (!user) {
        return res.status(401).json({ message: "Invalid credentials" });
      }

      const isPasswordValid = await bcrypt.compare(password, user.password);
      
      if (!isPasswordValid) {
        return res.status(401).json({ message: "Invalid credentials" });
      }

      req.session.userId = user.id;
      req.session.isAdmin = true;
      
      res.json({ 
        success: true, 
        user: { 
          id: user.id, 
          username: user.username,
          firstName: user.firstName,
          lastName: user.lastName 
        } 
      });
    } catch (error) {
      console.error('Login error:', error);
      res.status(500).json({ message: "Login failed" });
    }
  });

  app.post("/api/auth/logout", (req, res) => {
    req.session.destroy((err) => {
      if (err) {
        return res.status(500).json({ message: "Logout failed" });
      }
      res.json({ success: true });
    });
  });

  app.get("/api/auth/check", (req, res) => {
    if (req.session?.isAdmin) {
      res.json({ isAuthenticated: true });
    } else {
      res.json({ isAuthenticated: false });
    }
  });

  // Contact form submission
  app.post("/api/contact", async (req, res) => {
    try {
      const validatedData = insertContactSchema.parse(req.body);
      const contact = await storage.createContact(validatedData);
      
      // Send email notification if Google tokens are available
      if (req.session?.googleTokens) {
        try {
          gmailService.setCredentials(req.session.googleTokens);
          await gmailService.sendContactNotification(contact);
          console.log('✅ Email notification sent for new contact:', contact.firstName, contact.lastName);
        } catch (error) {
          console.error('❌ Failed to send email notification:', error);
        }
      } else {
        console.log('ℹ️ No Google tokens available - email notification skipped');
      }

      // Send SMS notification if RingCentral is configured and client opted in
      if (contact.smsOptIn && ringCentralService.isConfigured()) {
        // Load tokens from session if available
        if (req.session?.ringCentralTokens) {
          ringCentralService.setTokens(req.session.ringCentralTokens);
        }
        
        if (ringCentralService.hasValidTokens()) {
          try {
            // Send SMS to customer if they opted in and provided phone
            if (contact.phone) {
              const formattedPhone = ringCentralService.formatPhoneNumber(contact.phone);
              await ringCentralService.sendSMS({
                to: [formattedPhone],
                text: `TMH Global LLC: Thank you ${contact.firstName}! We received your inquiry about ${contact.serviceInterest}. We'll respond within 24 hours. Support: support@tmhglobal.com or 703-829-7277. Reply STOP to opt out.`
              });
              console.log(`✅ SMS sent to customer: ${formattedPhone}`);
            }
            
            // Also send SMS alert to business phone
            await ringCentralService.sendSMS({
              to: ['+17038297277'], // Your business phone number
              text: `🚨 NEW LEAD: ${contact.firstName} ${contact.lastName} interested in ${contact.serviceInterest}. Email: ${contact.email}${contact.phone ? `, Phone: ${contact.phone}` : ''}. Add to SMS campaigns!`
            });
            console.log('✅ SMS alert sent for new contact with SMS opt-in');
          } catch (error) {
            console.error('❌ Failed to send SMS alert:', error);
          }
        } else {
          console.log('ℹ️ RingCentral OAuth tokens not available - SMS notification skipped. Please authorize RingCentral SMS.');
        }
      }
      
      console.log('New contact submission:', contact);
      
      res.json({ success: true, message: "Contact form submitted successfully", id: contact.id });
    } catch (error: any) {
      res.status(400).json({ 
        success: false, 
        message: "Invalid form data", 
        errors: error.errors || error.message 
      });
    }
  });


  // Get all contacts (admin endpoint)
  app.get("/api/contacts", async (req, res) => {
    try {
      const contacts = await storage.getAllContacts();
      res.json(contacts);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve contacts" });
    }
  });

  // Get all consultations (admin endpoint)
  app.get("/api/consultations", async (req, res) => {
    try {
      const consultations = await storage.getAllConsultations();
      res.json(consultations);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve consultations" });
    }
  });

  // Google Calendar Authentication Routes
  app.get("/api/auth/google", (req, res) => {
    try {
      const authUrl = googleCalendarService.getAuthUrl();
      res.json({ authUrl });
    } catch (error) {
      res.status(500).json({ message: "Failed to generate auth URL" });
    }
  });

  // Handle Google OAuth redirect (GET request from Google)
  app.get("/api/auth/google/callback", async (req, res) => {
    try {
      const { code, error } = req.query;
      
      if (error) {
        return res.send(`
          <html>
            <script>
              window.close();
            </script>
          </html>
        `);
      }
      
      if (!code) {
        return res.status(400).send("Authorization code required");
      }

      const tokens = await googleCalendarService.getTokens(code as string);
      
      // Store tokens in session
      if (!req.session) {
        req.session = {};
      }
      req.session.googleTokens = tokens;

      // Also save tokens to database for persistence
      try {
        const adminUser = await storage.getUserByUsername("tmh_admin");
        if (adminUser) {
          await storage.upsertOAuthTokens(adminUser.id, "google", {
            accessToken: tokens.access_token,
            refreshToken: tokens.refresh_token || undefined,
            expiresAt: tokens.expiry_date ? new Date(tokens.expiry_date) : undefined,
            scope: tokens.scope,
            accountId: null, // Could be email if needed
            metadata: null
          });
          console.log('✅ Google tokens saved to database for persistence');
        }
      } catch (error) {
        console.error('❌ Failed to save Google tokens to database:', error);
      }
      
      // Close the popup window and notify parent
      res.send(`
        <html>
          <script>
            // Notify parent window and close popup
            if (window.opener) {
              window.opener.postMessage({ type: 'GOOGLE_AUTH_SUCCESS' }, '*');
            }
            window.close();
          </script>
        </html>
      `);
    } catch (error: any) {
      console.error('Google auth callback error:', error);
      res.send(`
        <html>
          <script>
            // Notify parent window of error and close popup
            if (window.opener) {
              window.opener.postMessage({ type: 'GOOGLE_AUTH_ERROR', error: '${error.message}' }, '*');
            }
            window.close();
          </script>
        </html>
      `);
    }
  });

  // Legacy POST endpoint (keeping for compatibility)
  app.post("/api/auth/google/callback", async (req, res) => {
    try {
      const { code } = req.body;
      
      if (!code) {
        return res.status(400).json({ message: "Authorization code required" });
      }

      const tokens = await googleCalendarService.getTokens(code);
      
      // Store tokens in session
      if (!req.session) {
        req.session = {};
      }
      req.session.googleTokens = tokens;
      
      res.json({ 
        success: true, 
        message: "Google Calendar connected successfully",
        hasRefreshToken: !!tokens.refresh_token
      });
    } catch (error: any) {
      console.error('Google auth callback error:', error);
      res.status(500).json({ 
        message: "Failed to authenticate with Google Calendar",
        error: error.message 
      });
    }
  });

  // RingCentral configuration endpoint (ADMIN ONLY)
  app.post("/api/sms/configure", async (req, res) => {
    // Basic authentication check - replace with your actual auth
    const authHeader = req.headers.authorization;
    if (!authHeader || authHeader !== 'Bearer ADMIN_SECRET_KEY') {
      return res.status(401).json({ message: "Unauthorized - Admin access required" });
    }
    try {
      const { clientId, clientSecret } = req.body;
      
      console.log(`📥 Received RingCentral config: ClientID=${clientId?.substring(0,10)}..., Secret=${clientSecret ? '[PROVIDED]' : '[MISSING]'}`);
      
      if (!clientId || !clientSecret) {
        return res.status(400).json({ message: "Client ID and Client Secret are required" });
      }
      
      // Update environment variables
      process.env.RINGCENTRAL_CLIENT_ID = clientId;
      process.env.RINGCENTRAL_CLIENT_SECRET = clientSecret;
      
      // Update the service configuration dynamically  
      ringCentralService.updateConfig(clientId, clientSecret);
      
      // Test the new configuration immediately
      const testUrl = ringCentralService.getAuthorizationUrl();
      console.log(`🔧 New OAuth URL: ${testUrl.substring(0,100)}...`);
      console.log('🔧 RingCentral credentials updated successfully');
      
      res.json({ message: "RingCentral configuration updated successfully" });
    } catch (error: any) {
      console.error('RingCentral configuration error:', error);
      res.status(500).json({ message: "Failed to update RingCentral configuration" });
    }
  });

  // RingCentral configuration debug endpoint
  app.get("/api/sms/debug", (req, res) => {
    try {
      const config = ringCentralService.getConfig();
      res.json({ 
        clientId: config.clientId ? `${config.clientId.substring(0,10)}...` : 'NOT SET',
        hasSecret: !!config.clientSecret,
        redirectUri: config.redirectUri
      });
    } catch (error) {
      res.status(500).json({ message: "Failed to get RingCentral config" });
    }
  });

  // RingCentral OAuth endpoints
  app.get("/api/auth/ringcentral", (req, res) => {
    try {
      const authUrl = ringCentralService.getAuthorizationUrl();
      res.json({ authUrl });
    } catch (error) {
      res.status(500).json({ message: "Failed to generate RingCentral auth URL" });
    }
  });

  // Handle RingCentral OAuth redirect
  app.get("/api/auth/ringcentral/callback", async (req, res) => {
    try {
      // Prevent caching to avoid 304 responses
      res.set('Cache-Control', 'no-store');
      console.log('🔧 RingCentral callback received with query:', req.query);
      
      const { code, error } = req.query;
      
      if (error) {
        return res.send(`
          <html>
            <script>
              if (window.opener) {
                window.opener.postMessage({ type: 'RINGCENTRAL_AUTH_ERROR', error: '${error}' }, '*');
              }
              window.close();
            </script>
          </html>
        `);
      }
      
      if (!code) {
        return res.status(400).send("Authorization code required");
      }

      console.log('🔧 RingCentral: Starting token exchange with code:', code);
      const tokens = await ringCentralService.exchangeCodeForToken(code as string);
      console.log('🔧 RingCentral: Token exchange successful:', !!tokens.accessToken);
      
      // Store tokens in session
      if (!req.session) {
        req.session = {};
      }
      req.session.ringCentralTokens = tokens;
      console.log('🔧 RingCentral: Tokens stored in session');
      
      // Also save tokens to database for persistence
      try {
        const adminUser = await storage.getUserByUsername("tmh_admin");
        if (adminUser) {
          await storage.upsertOAuthTokens(adminUser.id, "ringcentral", {
            accessToken: tokens.accessToken,
            refreshToken: tokens.refreshToken || undefined,
            expiresAt: tokens.expiresAt,
            scope: null, // RingCentral doesn't provide scope in response
            accountId: null, // Could store extension ID if available
            metadata: null
          });
          console.log('✅ RingCentral tokens saved to database for persistence');
        }
      } catch (error) {
        console.error('❌ Failed to save RingCentral tokens to database:', error);
      }
      
      // Set tokens in the service for immediate use
      ringCentralService.setTokens(tokens);
      console.log('🔧 RingCentral: Tokens set in service, hasValidTokens:', ringCentralService.hasValidTokens());
      
      // Close the popup window and notify parent
      res.send(`
        <html>
          <body>
            <p style="font-family: Arial, sans-serif; text-align: center; margin-top: 100px;">
              ✅ RingCentral SMS connected successfully! <br><br>
              <a href="javascript:void(0)" onclick="window.close()" style="color: #0066cc;">Close this window</a>
            </p>
          </body>
          <script>
            try {
              if (window.opener) {
                window.opener.postMessage({ type: 'RINGCENTRAL_AUTH_SUCCESS' }, '*');
              }
            } catch (e) {}
            
            // Multiple close attempts for browser compatibility
            window.close();
            setTimeout(function() {
              try {
                window.open('', '_self');
                window.close();
              } catch (e) {}
            }, 200);
          </script>
        </html>
      `);
    } catch (error: any) {
      console.error('RingCentral auth callback error:', error);
      res.send(`
        <html>
          <script>
            // Notify parent window of error and close popup
            if (window.opener) {
              window.opener.postMessage({ type: 'RINGCENTRAL_AUTH_ERROR', error: '${error.message}' }, '*');
            }
            window.close();
          </script>
        </html>
      `);
    }
  });

  app.get("/api/calendar/status", (req, res) => {
    const isConnected = !!(req.session?.googleTokens);
    res.json({ 
      connected: isConnected,
      email: isConnected ? 'tmhglobal@gmail.com' : null
    });
  });

  app.post("/api/calendar/disconnect", (req, res) => {
    if (req.session?.googleTokens) {
      delete req.session.googleTokens;
    }
    res.json({ success: true, message: "Google Calendar disconnected" });
  });

  // RingCentral SMS status endpoint
  app.get("/api/sms/status", (req, res) => {
    // Restore tokens from session if available
    if (req.session?.ringCentralTokens && !ringCentralService.hasValidTokens()) {
      ringCentralService.setTokens(req.session.ringCentralTokens);
    }
    
    const isConfigured = ringCentralService.isConfigured();
    const hasTokens = !!(req.session?.ringCentralTokens);
    const hasValidTokens = hasTokens && ringCentralService.hasValidTokens();
    
    res.json({ 
      configured: isConfigured,
      connected: hasValidTokens,
      service: "RingCentral",
      features: isConfigured ? ["SMS Notifications", "Contact Form Alerts"] : [],
      authRequired: isConfigured && !hasValidTokens
    });
  });

  app.post("/api/sms/disconnect", (req, res) => {
    if (req.session?.ringCentralTokens) {
      delete req.session.ringCentralTokens;
    }
    res.json({ success: true, message: "RingCentral SMS disconnected" });
  });

  // Debug endpoint to see SMS-enabled numbers  
  app.get("/api/sms/debug/senders", async (req, res) => {
    try {
      // Restore tokens from session if available
      if (req.session?.ringCentralTokens && !ringCentralService.hasValidTokens()) {
        ringCentralService.setTokens(req.session.ringCentralTokens);
      }
      
      if (!ringCentralService.hasValidTokens()) {
        return res.status(401).json({ message: "RingCentral not authenticated" });
      }
      
      res.json({ message: "SMS senders discovery triggered. Check logs for results." });
    } catch (error) {
      res.status(500).json({ message: "Failed to discover SMS senders" });
    }
  });

  // PayPal integration routes
  app.get("/paypal/setup", async (req, res) => {
    await loadPaypalDefault(req, res);
  });

  app.post("/paypal/order", async (req, res) => {
    // Request body should contain: { intent, amount, currency }
    await createPaypalOrder(req, res);
  });

  app.post("/paypal/order/:orderID/capture", async (req, res) => {
    await capturePaypalOrder(req, res);
  });

  // Square payment processing endpoint
  app.post("/api/square/create-payment", async (req, res) => {
    try {
      const { amount, planId, customer } = req.body;
      
      // Validate required fields
      if (!amount || !planId || !customer) {
        return res.status(400).json({ 
          success: false, 
          message: "Missing required fields: amount, planId, and customer information" 
        });
      }

      // Create customer record first
      const squareCustomer = await squareService.createCustomer(
        customer.email,
        customer.firstName,
        customer.lastName
      );

      // For now, we'll create a subscription record in our database
      // In a full Square integration, this would create actual Square payment
      const subscriptionData = {
        userId: squareCustomer.id,
        paymentProvider: "square" as const,
        subscriptionId: randomUUID(),
        status: "active",
        planType: planId,
        amount: parseInt(amount),
        currency: "USD",
        startDate: new Date(),
        expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000), // 30 days
        customerInfo: {
          firstName: customer.firstName,
          lastName: customer.lastName,
          email: customer.email,
          company: customer.company
        }
      };

      const subscription = await storage.createSubscription(subscriptionData);

      res.json({
        success: true,
        subscription,
        customer: squareCustomer,
        message: "Payment processed successfully. Welcome to TMH Global!"
      });

    } catch (error) {
      console.error('Square payment creation error:', error);
      res.status(500).json({ 
        success: false, 
        message: "Failed to process payment. Please try again." 
      });
    }
  });

  // Payment services status endpoints
  app.get("/api/payments/status", (req, res) => {
    const squareConfigured = !!(process.env.SQUARE_ACCESS_TOKEN && process.env.SQUARE_APPLICATION_ID);
    const paypalConfigured = !!(process.env.PAYPAL_CLIENT_ID && process.env.PAYPAL_CLIENT_SECRET);
    
    res.json({ 
      square: {
        configured: squareConfigured,
        features: squareConfigured ? ["Credit Card Processing", "One-time Payments"] : []
      },
      paypal: {
        configured: paypalConfigured,
        features: paypalConfigured ? ["PayPal Payments", "Subscription Management"] : []
      }
    });
  });

  // Zoom integration status endpoint
  app.get("/api/zoom/status", (req, res) => {
    const zoomConfigured = !!(process.env.ZOOM_API_KEY && process.env.ZOOM_API_SECRET && process.env.ZOOM_ACCOUNT_ID);
    
    res.json({ 
      configured: zoomConfigured,
      features: zoomConfigured ? ["Virtual Meetings", "Webinars", "Training Sessions"] : [],
      service: "Zoom"
    });
  });

  // Calendar availability checking
  app.post("/api/calendar/check-availability", async (req, res) => {
    try {
      if (!req.session?.googleTokens) {
        return res.status(401).json({ message: "Google Calendar not connected" });
      }

      const { startTime, endTime } = req.body;
      
      if (!startTime || !endTime) {
        return res.status(400).json({ message: "Start time and end time required" });
      }

      googleCalendarService.setCredentials(req.session.googleTokens);
      
      const isAvailable = await googleCalendarService.checkAvailability(
        startTime,
        endTime,
        'primary'
      );
      
      res.json({ available: isAvailable });
    } catch (error: any) {
      console.error('Availability check error:', error);
      res.status(500).json({ 
        message: "Failed to check calendar availability",
        error: error.message 
      });
    }
  });

  // Get calendar events
  app.get("/api/calendar/events", async (req, res) => {
    try {
      if (!req.session?.googleTokens) {
        return res.status(401).json({ message: "Google Calendar not connected" });
      }

      const { startDate, endDate } = req.query;
      
      googleCalendarService.setCredentials(req.session.googleTokens);
      
      const events = await googleCalendarService.getEvents(
        startDate as string || new Date().toISOString(),
        endDate as string || new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString(),
        'primary'
      );
      
      res.json({ events });
    } catch (error: any) {
      console.error('Get events error:', error);
      res.status(500).json({ 
        message: "Failed to fetch calendar events",
        error: error.message 
      });
    }
  });

  // Training videos endpoints
  app.get("/api/training-videos", async (req, res) => {
    try {
      const videos = await storage.getPublicTrainingVideos();
      res.json(videos);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve training videos" });
    }
  });

  app.get("/api/training-videos/:id", async (req, res) => {
    try {
      const video = await storage.getTrainingVideo(req.params.id);
      if (!video) {
        return res.status(404).json({ message: "Training video not found" });
      }
      res.json(video);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve training video" });
    }
  });

  app.post("/api/training-videos", async (req, res) => {
    try {
      const validatedData = insertTrainingVideoSchema.parse(req.body);
      const video = await storage.createTrainingVideo(validatedData);
      res.json({ success: true, video });
    } catch (error: any) {
      res.status(400).json({ 
        success: false, 
        message: "Invalid video data", 
        errors: error.errors || error.message 
      });
    }
  });

  app.patch("/api/training-videos/:id", async (req, res) => {
    try {
      if (!req.body || Object.keys(req.body).length === 0) {
        return res.status(400).json({ 
          success: false, 
          message: "Request body cannot be empty" 
        });
      }
      const partialSchema = insertTrainingVideoSchema.partial();
      const validatedData = partialSchema.parse(req.body);
      const video = await storage.updateTrainingVideo(req.params.id, validatedData);
      if (!video) {
        return res.status(404).json({ message: "Training video not found" });
      }
      res.json({ success: true, video });
    } catch (error: any) {
      res.status(400).json({ 
        success: false, 
        message: "Failed to update video", 
        error: error.message 
      });
    }
  });

  app.delete("/api/training-videos/:id", async (req, res) => {
    try {
      const existingVideo = await storage.getTrainingVideo(req.params.id);
      if (!existingVideo) {
        return res.status(404).json({ 
          success: false,
          message: "Training video not found" 
        });
      }
      await storage.deleteTrainingVideo(req.params.id);
      res.json({ success: true, message: "Video deleted successfully" });
    } catch (error: any) {
      res.status(500).json({ 
        success: false, 
        message: "Failed to delete video", 
        error: error.message 
      });
    }
  });

  // File upload endpoint
  app.post("/api/upload", upload.single('file'), async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ message: "No file uploaded" });
      }

      const fileUpload = await storage.createFileUpload({
        originalName: req.file.originalname,
        fileName: req.file.filename,
        fileSize: req.file.size.toString(),
        mimeType: req.file.mimetype,
        uploadedBy: req.body.uploadedBy || 'anonymous'
      });

      res.json({ 
        success: true, 
        message: "File uploaded successfully", 
        file: fileUpload 
      });
    } catch (error: any) {
      res.status(500).json({ 
        success: false, 
        message: "Failed to upload file", 
        error: error.message 
      });
    }
  });

  // Get uploaded files
  app.get("/api/uploads", async (req, res) => {
    try {
      const files = await storage.getAllFileUploads();
      res.json(files);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve uploaded files" });
    }
  });

  // Emergency contact endpoint
  app.post("/api/emergency-contact", async (req, res) => {
    try {
      const { name, email, phone, urgentMessage } = req.body;
      
      const emergencyContact = await storage.createContact({
        firstName: name.split(' ')[0] || name,
        lastName: name.split(' ').slice(1).join(' ') || '',
        email,
        phone,
        message: urgentMessage,
        isUrgent: true,
        serviceInterest: 'emergency'
      });

      // In a real application, this would trigger immediate notifications
      console.log('URGENT: Emergency contact submission:', emergencyContact);
      
      res.json({ 
        success: true, 
        message: "Emergency contact submitted. You will be contacted within 1 hour.", 
        id: emergencyContact.id 
      });
    } catch (error: any) {
      res.status(500).json({ 
        success: false, 
        message: "Failed to submit emergency contact", 
        error: error.message 
      });
    }
  });

  // Community/Social features endpoints
  
  // Get all posts with user data
  app.get("/api/posts", async (req, res) => {
    try {
      const posts = await storage.getAllPosts();
      const postsWithUsers = await Promise.all(
        posts.map(async (post) => {
          const user = await storage.getUser(post.userId);
          return { ...post, user };
        })
      );
      res.json(postsWithUsers);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve posts" });
    }
  });

  // Create new post
  app.post("/api/posts", async (req, res) => {
    try {
      const validatedData = insertPostSchema.parse(req.body);
      const post = await storage.createPost(validatedData);
      const user = await storage.getUser(post.userId);
      res.json({ ...post, user });
    } catch (error: any) {
      res.status(400).json({ 
        message: "Invalid post data", 
        errors: error.errors || error.message 
      });
    }
  });

  // Like/unlike a post
  app.post("/api/posts/:id/like", async (req, res) => {
    try {
      const postId = req.params.id;
      const { userId } = req.body;

      if (!userId) {
        return res.status(400).json({ message: "User ID required" });
      }

      // Check if already liked
      const existingLike = await storage.getLike(postId, userId);
      
      if (existingLike) {
        // Unlike the post
        await storage.deleteLike(postId, userId);
        res.json({ liked: false, message: "Post unliked" });
      } else {
        // Like the post
        await storage.createLike(postId, userId);
        res.json({ liked: true, message: "Post liked" });
      }
    } catch (error: any) {
      res.status(500).json({ message: "Failed to process like", error: error.message });
    }
  });

  // Get comments for a post
  app.get("/api/posts/:id/comments", async (req, res) => {
    try {
      const postId = req.params.id;
      const comments = await storage.getCommentsByPost(postId);
      const commentsWithUsers = await Promise.all(
        comments.map(async (comment) => {
          const user = await storage.getUser(comment.userId);
          return { ...comment, user };
        })
      );
      res.json(commentsWithUsers);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve comments" });
    }
  });

  // Add comment to post
  app.post("/api/posts/:id/comments", async (req, res) => {
    try {
      const postId = req.params.id;
      const { content, userId } = req.body;
      
      if (!userId || !content) {
        return res.status(400).json({ message: "User ID and content are required" });
      }

      const commentData = { postId, userId, content };
      const validatedData = insertCommentSchema.parse(commentData);
      const comment = await storage.createComment(validatedData);
      const user = await storage.getUser(comment.userId);
      
      res.json({ ...comment, user });
    } catch (error: any) {
      res.status(400).json({ 
        message: "Invalid comment data", 
        errors: error.errors || error.message 
      });
    }
  });

  // Get user profile
  app.get("/api/user", async (req, res) => {
    // Return admin user from database with PMI settings
    try {
      const currentUser = await storage.getUserByUsername("tmh_admin");
      if (currentUser) {
        res.json(currentUser);
      } else {
        res.status(404).json({ message: "User not found" });
      }
    } catch (error) {
      console.error('Error getting user profile:', error);
      res.status(500).json({ message: "Failed to get user" });
    }
  });

  // Get user profile (auth endpoint for frontend compatibility)
  app.get("/api/auth/user", async (req, res) => {
    // Return admin user from database with PMI settings
    try {
      const currentUser = await storage.getUserByUsername("tmh_admin");
      if (currentUser) {
        res.json(currentUser);
      } else {
        res.status(404).json({ message: "User not found" });
      }
    } catch (error) {
      console.error('Error getting auth user profile:', error);
      res.status(500).json({ message: "Failed to get user" });
    }
  });

  // Update user settings (PMI configuration)
  app.patch("/api/auth/user/settings", async (req, res) => {
    try {
      const { zoomPmi, zoomPmiPassword } = req.body;
      
      // For now, get the admin user as the current user from database
      const currentUser = await storage.getUserByUsername("tmh_admin");
      if (!currentUser) {
        return res.status(404).json({ message: "Admin user not found" });
      }
      const updatedUser = await storage.updateUserSettings(currentUser.id, {
        zoomPmi,
        zoomPmiPassword
      });

      res.json(updatedUser);
    } catch (error) {
      console.error("Error updating user settings:", error);
      res.status(500).json({ message: "Failed to update settings" });
    }
  });

  // Monitoring endpoints
  app.get("/api/monitoring", async (req, res) => {
    try {
      const [metrics, alerts, summary] = await Promise.all([
        storage.getMetrics(),
        storage.getAlerts(),
        storage.getSummaryStats()
      ]);

      // Check for usage warnings and create alerts
      const databaseLimit = 10 * 1024 * 1024 * 1024; // 10GB
      const userLimit = 10000;
      const apiLimit = 100000;

      const databaseUsage = (summary.storageUsed / databaseLimit) * 100;
      const userUsage = (summary.totalUsers / userLimit) * 100;
      const apiUsage = (summary.apiRequestsToday / apiLimit) * 100;

      // Create alerts if thresholds exceeded
      if (databaseUsage > 75 && !alerts.some(a => a.resource === 'database' && !a.isResolved)) {
        await storage.createAlert(
          'limit_warning',
          'database',
          summary.storageUsed,
          databaseLimit,
          75,
          `Database storage at ${databaseUsage.toFixed(1)}% capacity`
        );
      }

      if (apiUsage > 75 && !alerts.some(a => a.resource === 'api_requests' && !a.isResolved)) {
        await storage.createAlert(
          'high_usage',
          'api_requests',
          summary.apiRequestsToday,
          apiLimit,
          75,
          `API requests at ${apiUsage.toFixed(1)}% of daily limit`
        );
      }

      // Get updated alerts after potential creation
      const updatedAlerts = await storage.getAlerts();

      res.json({
        metrics,
        alerts: updatedAlerts,
        summary
      });
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve monitoring data" });
    }
  });

  app.get("/api/monitoring/metrics", async (req, res) => {
    try {
      const { type, hours } = req.query;
      const metrics = await storage.getMetrics(
        type as string,
        hours ? parseInt(hours as string) : undefined
      );
      res.json(metrics);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve metrics" });
    }
  });

  app.get("/api/monitoring/alerts", async (req, res) => {
    try {
      const { activeOnly } = req.query;
      const alerts = await storage.getAlerts(activeOnly === 'true');
      res.json(alerts);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve alerts" });
    }
  });

  app.post("/api/monitoring/alerts/:id/resolve", async (req, res) => {
    try {
      const alertId = req.params.id;
      // In a real implementation, this would update the alert status
      res.json({ success: true, message: "Alert resolved" });
    } catch (error) {
      res.status(500).json({ message: "Failed to resolve alert" });
    }
  });

  // Payment system endpoints
  
  // Check available payment methods
  app.get("/api/payment-methods", async (req, res) => {
    try {
      const methods = {
        square: squareService.isAvailable(),
        paypal: paypalService.isAvailable(),
        cashapp: true // Always available as manual payment
      };
      res.json(methods);
    } catch (error) {
      res.status(500).json({ message: "Failed to check payment methods" });
    }
  });

  // Create subscription
  app.post("/api/subscribe", async (req, res) => {
    try {
      const { planId, paymentMethod, amount } = req.body;
      
      // Validate request
      if (!planId || !paymentMethod || !amount) {
        return res.status(400).json({ message: "Missing required fields" });
      }

      // For demonstration, we'll create a subscription record
      const subscriptionData = {
        userId: "demo-user", // In real app, get from authentication
        paymentProvider: paymentMethod,
        subscriptionId: randomUUID(),
        status: "active",
        planType: planId,
        amount: parseInt(amount),
        currency: "USD",
        startDate: new Date(),
        expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000) // 30 days
      };

      const subscription = await storage.createSubscription(subscriptionData);

      // Handle different payment methods
      let paymentResult = null;
      
      if (paymentMethod === 'square' && squareService.isAvailable()) {
        // Square payment processing would go here
        paymentResult = { method: 'square', status: 'pending' };
      } else if (paymentMethod === 'paypal' && paypalService.isAvailable()) {
        // PayPal payment processing would go here
        try {
          const order = await paypalService.createOrder(amount.toString(), 'USD');
          paymentResult = { 
            method: 'paypal', 
            status: 'pending',
            orderId: order.data.id,
            approvalUrl: order.data.links?.find((link: any) => link.rel === 'approve')?.href
          };
        } catch (error) {
          console.error('PayPal order creation failed:', error);
          paymentResult = { method: 'paypal', status: 'failed', error: 'PayPal unavailable' };
        }
      } else if (paymentMethod === 'cashapp') {
        paymentResult = { 
          method: 'cashapp', 
          status: 'pending_manual_verification',
          instructions: `Send $${amount} to $TMHGlobal with your email as the note`
        };
      }

      res.json({
        success: true,
        subscription,
        payment: paymentResult,
        message: "Subscription initiated successfully"
      });

    } catch (error) {
      console.error('Subscription creation error:', error);
      res.status(500).json({ 
        success: false, 
        message: "Failed to create subscription" 
      });
    }
  });

  // PayPal specific endpoints
  app.get("/api/paypal/setup", async (req, res) => {
    if (!paypalService.isAvailable()) {
      return res.status(503).json({ error: "PayPal service unavailable" });
    }

    try {
      const clientToken = await paypalService.getClientToken();
      res.json({ clientToken });
    } catch (error) {
      res.status(500).json({ error: "Failed to get PayPal client token" });
    }
  });

  app.post("/api/paypal/order", async (req, res) => {
    if (!paypalService.isAvailable()) {
      return res.status(503).json({ error: "PayPal service unavailable" });
    }

    try {
      const { amount, currency = 'USD', intent = 'CAPTURE' } = req.body;
      const result = await paypalService.createOrder(amount, currency, intent);
      res.status(result.statusCode).json(result.data);
    } catch (error: any) {
      res.status(500).json({ error: error.message });
    }
  });

  app.post("/api/paypal/order/:orderID/capture", async (req, res) => {
    if (!paypalService.isAvailable()) {
      return res.status(503).json({ error: "PayPal service unavailable" });
    }

    try {
      const { orderID } = req.params;
      const result = await paypalService.captureOrder(orderID);
      res.status(result.statusCode).json(result.data);
    } catch (error: any) {
      res.status(500).json({ error: error.message });
    }
  });

  // SMS integration endpoint with RingCentral
  app.post("/api/sms", async (req, res) => {
    try {
      const { to, message } = req.body;
      
      if (!ringCentralService.isConfigured()) {
        return res.status(400).json({ 
          success: false, 
          message: "SMS service not configured. Please contact administrator to set up RingCentral credentials." 
        });
      }
      
      // Send actual SMS using RingCentral
      const result = await ringCentralService.sendSMS({
        to: [to],
        text: message
      });
      
      console.log(`SMS sent successfully to ${to}: ${message}`);
      
      res.json({ 
        success: true, 
        message: "SMS sent successfully",
        smsId: result.id
      });
    } catch (error: any) {
      console.error('SMS sending error:', error);
      res.status(500).json({ 
        success: false, 
        message: "Failed to send SMS", 
        error: error.message 
      });
    }
  });

  // Live Training Sessions with Zoom Integration
  app.get("/api/live-training", async (req, res) => {
    try {
      const category = req.query.category as string;
      const isPremiumOnly = req.query.premium === 'true';
      
      const sessions = await storage.getLiveTrainingSessions(category, isPremiumOnly);
      res.json(sessions);
    } catch (error) {
      console.error("Error fetching live training sessions:", error);
      res.status(500).json({ message: "Failed to fetch training sessions" });
    }
  });

  app.get("/api/live-training/upcoming", async (req, res) => {
    try {
      const sessions = await storage.getUpcomingTrainingSessions();
      res.json(sessions);
    } catch (error) {
      console.error("Error fetching upcoming training sessions:", error);
      res.status(500).json({ message: "Failed to fetch upcoming sessions" });
    }
  });

  app.get("/api/live-training/:id", async (req, res) => {
    try {
      const session = await storage.getLiveTrainingSession(req.params.id);
      if (!session) {
        return res.status(404).json({ message: "Training session not found" });
      }
      res.json(session);
    } catch (error) {
      console.error("Error fetching training session:", error);
      res.status(500).json({ message: "Failed to fetch training session" });
    }
  });

  app.post("/api/live-training", async (req, res) => {
    try {
      const sessionData = req.body;
      
      // Create Zoom meeting
      let zoomMeeting = null;
      if (sessionData.createZoomMeeting) {
        try {
          if (sessionData.sessionType === 'webinar') {
            zoomMeeting = await zoomService.createWebinar({
              topic: sessionData.title,
              agenda: sessionData.agenda || sessionData.description,
              startTime: sessionData.scheduledStartTime,
              duration: sessionData.duration,
              timezone: sessionData.timezone,
              registrationRequired: true,
              practiceSession: true
            });
          } else {
            zoomMeeting = await zoomService.createTrainingSession({
              topic: sessionData.title,
              agenda: sessionData.agenda || sessionData.description,
              startTime: sessionData.scheduledStartTime,
              duration: sessionData.duration,
              timezone: sessionData.timezone,
              waitingRoom: true,
              allowRecording: sessionData.recordingEnabled
            });
          }

          // Add Zoom details to session data
          sessionData.zoomMeetingId = zoomMeeting.id.toString();
          sessionData.zoomMeetingUuid = zoomMeeting.uuid;
          sessionData.zoomJoinUrl = zoomMeeting.join_url;
          sessionData.zoomStartUrl = zoomMeeting.start_url;
          sessionData.zoomPassword = zoomMeeting.password;
        } catch (zoomError) {
          console.error("Error creating Zoom meeting:", zoomError);
          return res.status(500).json({ message: "Failed to create Zoom meeting" });
        }
      }

      const session = await storage.createLiveTrainingSession(sessionData);
      res.json(session);
    } catch (error) {
      console.error("Error creating training session:", error);
      res.status(500).json({ message: "Failed to create training session" });
    }
  });

  app.put("/api/live-training/:id", async (req, res) => {
    try {
      const sessionId = req.params.id;
      const updates = req.body;
      
      // Update Zoom meeting if needed
      if (updates.updateZoomMeeting && updates.zoomMeetingId) {
        try {
          await zoomService.updateMeeting(updates.zoomMeetingId, {
            topic: updates.title,
            agenda: updates.agenda,
            start_time: updates.scheduledStartTime,
            duration: updates.duration
          });
        } catch (zoomError) {
          console.error("Error updating Zoom meeting:", zoomError);
        }
      }

      const session = await storage.updateLiveTrainingSession(sessionId, updates);
      if (!session) {
        return res.status(404).json({ message: "Training session not found" });
      }
      res.json(session);
    } catch (error) {
      console.error("Error updating training session:", error);
      res.status(500).json({ message: "Failed to update training session" });
    }
  });

  // Training session registration
  app.post("/api/live-training/:id/register", async (req, res) => {
    try {
      const sessionId = req.params.id;
      const userId = req.body.userId;
      
      // Check if already registered
      const existingRegistration = await storage.getTrainingRegistration(sessionId, userId);
      if (existingRegistration) {
        return res.status(400).json({ message: "Already registered for this session" });
      }

      const session = await storage.getLiveTrainingSession(sessionId);
      if (!session) {
        return res.status(404).json({ message: "Training session not found" });
      }

      // Register for webinar if needed
      let zoomRegistration = null;
      if (session.sessionType === 'webinar' && session.zoomMeetingId) {
        try {
          const user = await storage.getUser(userId);
          if (user) {
            zoomRegistration = await zoomService.addWebinarRegistrant(session.zoomMeetingId, {
              email: user.email || '',
              first_name: user.firstName || '',
              last_name: user.lastName || '',
              organization: user.company || ''
            });
          }
        } catch (zoomError) {
          console.error("Error registering for Zoom webinar:", zoomError);
        }
      }

      const registration = await storage.createTrainingRegistration({
        sessionId,
        userId,
        zoomRegistrantId: zoomRegistration?.registrant_id || null,
        zoomJoinUrl: zoomRegistration?.join_url || session.zoomJoinUrl
      });

      res.json(registration);
    } catch (error) {
      console.error("Error registering for training session:", error);
      res.status(500).json({ message: "Failed to register for training session" });
    }
  });

  app.get("/api/live-training/:id/registrations", async (req, res) => {
    try {
      const registrations = await storage.getTrainingRegistrations(req.params.id);
      res.json(registrations);
    } catch (error) {
      console.error("Error fetching registrations:", error);
      res.status(500).json({ message: "Failed to fetch registrations" });
    }
  });

  // Training session resources
  app.get("/api/live-training/:id/resources", async (req, res) => {
    try {
      const resources = await storage.getTrainingResources(req.params.id);
      res.json(resources);
    } catch (error) {
      console.error("Error fetching training resources:", error);
      res.status(500).json({ message: "Failed to fetch resources" });
    }
  });

  app.post("/api/live-training/:id/resources", upload.single('file'), async (req, res) => {
    try {
      const sessionId = req.params.id;
      const { title, description, resourceType } = req.body;
      
      let fileUrl = null;
      let fileSize = null;
      let mimeType = null;

      if (req.file) {
        fileUrl = `/uploads/${req.file.filename}`;
        fileSize = req.file.size;
        mimeType = req.file.mimetype;
      }

      const resource = await storage.createTrainingResource({
        sessionId,
        title,
        description,
        resourceType: resourceType || 'document',
        fileUrl,
        fileSize,
        mimeType,
        uploadedBy: req.body.uploadedBy || 'instructor'
      });

      res.json(resource);
    } catch (error) {
      console.error("Error creating training resource:", error);
      res.status(500).json({ message: "Failed to create resource" });
    }
  });

  // Create instant meeting for quick training
  app.post("/api/live-training/instant", async (req, res) => {
    try {
      const { topic, agenda, instructorId, instructorName } = req.body;
      
      // Get user's PMI settings from database
      const adminUsers = Array.from((storage as any).users.values()).filter((u: any) => u.username === "tmh_admin");
      const userPmi = adminUsers.length > 0 && adminUsers[0].zoomPmi ? {
        pmi: adminUsers[0].zoomPmi,
        password: adminUsers[0].zoomPmiPassword || ''
      } : undefined;
      
      // Create instant Zoom meeting using user's PMI
      const zoomMeeting = await zoomService.createInstantMeeting(topic, agenda, userPmi);
      
      const session = await storage.createLiveTrainingSession({
        title: topic,
        description: agenda,
        category: 'instant',
        sessionType: 'meeting',
        instructorId,
        instructorName,
        zoomMeetingId: zoomMeeting.id.toString(),
        zoomMeetingUuid: zoomMeeting.uuid,
        zoomJoinUrl: zoomMeeting.join_url,
        zoomStartUrl: zoomMeeting.start_url,
        zoomPassword: zoomMeeting.password,
        scheduledStartTime: new Date(),
        scheduledEndTime: new Date(Date.now() + 60 * 60 * 1000), // 1 hour
        duration: 60,
        status: 'live'
      });

      res.json(session);
    } catch (error) {
      console.error("Error creating instant training session:", error);
      res.status(500).json({ message: "Failed to create instant session" });
    }
  });

  // Section 508 Compliance Monitoring API Routes
  app.get("/api/compliance/scans", async (req, res) => {
    try {
      const scans = await storage.getComplianceScans();
      res.json(scans);
    } catch (error) {
      console.error("Error fetching compliance scans:", error);
      res.status(500).json({ message: "Failed to fetch compliance scans" });
    }
  });

  app.get("/api/compliance/scans/:id", async (req, res) => {
    try {
      const scan = await storage.getComplianceScan(req.params.id);
      if (!scan) {
        return res.status(404).json({ message: "Compliance scan not found" });
      }
      res.json(scan);
    } catch (error) {
      console.error("Error fetching compliance scan:", error);
      res.status(500).json({ message: "Failed to fetch compliance scan" });
    }
  });

  app.post("/api/compliance/scan", async (req, res) => {
    try {
      const { url } = req.body;
      
      if (!url) {
        return res.status(400).json({ message: "URL is required" });
      }

      // Validate URL format
      try {
        new URL(url);
      } catch {
        return res.status(400).json({ message: "Invalid URL format" });
      }

      // Start scan using compliance scanner
      const scanId = await complianceScanner.scanUrl(url, 'manual');
      
      res.json({ 
        success: true, 
        scanId, 
        message: "Accessibility scan started successfully" 
      });
    } catch (error) {
      console.error("Error starting compliance scan:", error);
      res.status(500).json({ message: "Failed to start compliance scan" });
    }
  });

  app.get("/api/compliance/issues", async (req, res) => {
    try {
      const severity = req.query.severity as string;
      const issues = await storage.getComplianceIssues(severity);
      res.json(issues);
    } catch (error) {
      console.error("Error fetching compliance issues:", error);
      res.status(500).json({ message: "Failed to fetch compliance issues" });
    }
  });

  app.get("/api/compliance/alerts", async (req, res) => {
    try {
      const alerts = await storage.getComplianceAlerts();
      res.json(alerts);
    } catch (error) {
      console.error("Error fetching compliance alerts:", error);
      res.status(500).json({ message: "Failed to fetch compliance alerts" });
    }
  });

  app.put("/api/compliance/alerts/:id/read", async (req, res) => {
    try {
      const alertId = req.params.id;
      const alert = await storage.markComplianceAlertAsRead(alertId);
      
      if (!alert) {
        return res.status(404).json({ message: "Alert not found" });
      }
      
      res.json(alert);
    } catch (error) {
      console.error("Error marking alert as read:", error);
      res.status(500).json({ message: "Failed to mark alert as read" });
    }
  });

  app.get("/api/compliance/reports/:scanId", async (req, res) => {
    try {
      const scanId = req.params.scanId;
      const report = await storage.getComplianceReport(scanId);
      
      if (!report) {
        return res.status(404).json({ message: "Compliance report not found" });
      }
      
      res.json(report);
    } catch (error) {
      console.error("Error fetching compliance report:", error);
      res.status(500).json({ message: "Failed to fetch compliance report" });
    }
  });

  // Interactive Training Features API Routes
  app.get("/api/training/polls/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const polls = await storage.getTrainingPolls(sessionId);
      res.json(polls);
    } catch (error) {
      console.error("Error fetching training polls:", error);
      res.status(500).json({ message: "Failed to fetch training polls" });
    }
  });

  app.post("/api/training/polls/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const { question, options, createdBy } = req.body;
      
      const poll = await storage.createTrainingPoll({
        sessionId,
        question,
        options,
        createdBy,
        isActive: true
      });
      
      res.json(poll);
    } catch (error) {
      console.error("Error creating training poll:", error);
      res.status(500).json({ message: "Failed to create training poll" });
    }
  });

  app.post("/api/training/polls/:pollId/vote", async (req, res) => {
    try {
      const pollId = req.params.pollId;
      const { optionIndex, userId } = req.body;
      
      const result = await storage.voteOnTrainingPoll(pollId, optionIndex, userId);
      res.json(result);
    } catch (error) {
      console.error("Error voting on poll:", error);
      res.status(500).json({ message: "Failed to vote on poll" });
    }
  });

  app.get("/api/training/qa/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const questions = await storage.getTrainingQAQuestions(sessionId);
      res.json(questions);
    } catch (error) {
      console.error("Error fetching Q&A questions:", error);
      res.status(500).json({ message: "Failed to fetch Q&A questions" });
    }
  });

  app.post("/api/training/qa/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const { question, askedBy } = req.body;
      
      const qaQuestion = await storage.createTrainingQAQuestion({
        sessionId,
        question,
        askedBy,
        status: 'pending'
      });
      
      res.json(qaQuestion);
    } catch (error) {
      console.error("Error creating Q&A question:", error);
      res.status(500).json({ message: "Failed to create Q&A question" });
    }
  });

  app.put("/api/training/qa/:questionId/answer", async (req, res) => {
    try {
      const questionId = req.params.questionId;
      const { answer, answeredBy } = req.body;
      
      const question = await storage.answerTrainingQAQuestion(questionId, answer, answeredBy);
      res.json(question);
    } catch (error) {
      console.error("Error answering Q&A question:", error);
      res.status(500).json({ message: "Failed to answer Q&A question" });
    }
  });

  app.get("/api/training/breakouts/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const breakoutRooms = await storage.getTrainingBreakoutRooms(sessionId);
      res.json(breakoutRooms);
    } catch (error) {
      console.error("Error fetching breakout rooms:", error);
      res.status(500).json({ message: "Failed to fetch breakout rooms" });
    }
  });

  app.post("/api/training/breakouts/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const { name, maxParticipants, topic } = req.body;
      
      const breakoutRoom = await storage.createTrainingBreakoutRoom({
        sessionId,
        name,
        maxParticipants,
        topic,
        status: 'active',
        participants: []
      });
      
      res.json(breakoutRoom);
    } catch (error) {
      console.error("Error creating breakout room:", error);
      res.status(500).json({ message: "Failed to create breakout room" });
    }
  });

  // One-on-One Consultation API Routes (removed duplicate - using the one at line 130)

  app.get("/api/consultations/availability", async (req, res) => {
    try {
      const date = req.query.date as string;
      const availableSlots = await storage.getConsultationAvailability(date);
      res.json(availableSlots);
    } catch (error) {
      console.error("Error fetching availability:", error);
      res.status(500).json({ message: "Failed to fetch availability" });
    }
  });

  app.post("/api/consultations", async (req, res) => {
    try {
      const validatedData = insertConsultationSchema.parse(req.body);
      console.log('📝 Processing consultation booking:', validatedData.firstName, validatedData.lastName);
      
      // Initialize conflict warning variable
      let conflictWarning = null;
      
      // Check calendar availability if tokens are set
      if (req.session?.googleTokens && validatedData.preferredDate) {
        console.log('🔗 Google tokens available, checking calendar availability...');
        googleCalendarService.setCredentials(req.session.googleTokens);
        
        const startTime = new Date(validatedData.preferredDate);
        const endTime = new Date(startTime.getTime() + 60 * 60 * 1000); // 1 hour duration
        
        const isAvailable = await googleCalendarService.checkAvailability(
          startTime.toISOString(),
          endTime.toISOString(),
          'primary'
        );
        
        // Instead of blocking, allow booking but warn about conflicts
        if (!isAvailable) {
          conflictWarning = "Note: There may be a scheduling conflict. We'll coordinate with you to resolve any timing issues.";
          console.log('⚠️ Calendar conflict detected, but allowing booking to proceed');
        }
        
        // Create calendar event
        console.log('🗓️ Creating Google Calendar event for consultation...');
        const calendarEvent = await googleCalendarService.createEvent({
          summary: `Consultation: ${validatedData.firstName} ${validatedData.lastName}`,
          description: `Business consultation meeting.\nService: ${validatedData.serviceType}\nMessage: ${validatedData.message}`,
          startDateTime: startTime.toISOString(),
          endDateTime: endTime.toISOString(),
          attendeeEmail: validatedData.email,
          timeZone: 'America/New_York'
        });
        
        console.log('✅ Google Calendar event created successfully:', calendarEvent.id);
        // Add calendar event ID to consultation data
        validatedData.calendarEventId = calendarEvent.id;
      } else {
        console.log('ℹ️ Google tokens not available - calendar event creation skipped');
      }
      
      const consultation = await storage.createConsultation(validatedData);
      
      // Send email notification if Google tokens are available
      if (req.session?.googleTokens) {
        try {
          gmailService.setCredentials(req.session.googleTokens);
          await gmailService.sendConsultationNotification(consultation);
          console.log('✅ Email notification sent for new consultation:', consultation.firstName, consultation.lastName);
        } catch (error) {
          console.error('❌ Failed to send consultation email notification:', error);
        }
      } else {
        console.log('ℹ️ No Google tokens available - email notification skipped');
      }

      // Send SMS notification if RingCentral is configured - consultation booking always sends admin alerts
      console.log('📱 Checking RingCentral SMS configuration...');
      if (ringCentralService.isConfigured()) {
        // Load tokens from session if available
        if (req.session?.ringCentralTokens) {
          console.log('📱 Loading RingCentral tokens from session...');
          ringCentralService.setTokens(req.session.ringCentralTokens);
        }
        
        if (ringCentralService.hasValidTokens()) {
          console.log('📱 RingCentral tokens valid, sending SMS alert...');
          try {
            const dateStr = consultation.preferredDate ? new Date(consultation.preferredDate).toLocaleDateString() : 'TBD';
            // Send SMS to customer if they provided phone
            if (consultation.phone) {
              const formattedPhone = ringCentralService.formatPhoneNumber(consultation.phone);
              await ringCentralService.sendSMS({
                to: [formattedPhone],
                text: `TMH Global LLC: Consultation confirmed! We'll contact you soon about ${consultation.serviceType} services. Support: support@tmhglobal.com or 703-829-7277. Reply STOP to opt out.`
              });
              console.log(`✅ SMS sent to customer: ${formattedPhone}`);
            }
            
            // Also send SMS alert to business phone
            await ringCentralService.sendSMS({
              to: ['+17038297277'], // Your business phone number
              text: `📅 NEW CONSULTATION: ${consultation.firstName} ${consultation.lastName} scheduled for ${dateStr}. Service: ${consultation.serviceType}. Email: ${consultation.email}${consultation.phone ? `, Phone: ${consultation.phone}` : ''}`
            });
            console.log('✅ SMS alert sent for new consultation');
          } catch (error) {
            console.error('❌ Failed to send consultation SMS alert:', error);
          }
        } else {
          console.log('ℹ️ RingCentral OAuth tokens not available - SMS notification skipped. Please authorize RingCentral SMS.');
        }
      } else {
        console.log('ℹ️ RingCentral not configured - SMS notification skipped');
      }
      
      console.log('New consultation request:', consultation);
      
      res.json({ 
        success: true, 
        message: conflictWarning || "Consultation scheduled successfully and added to calendar", 
        id: consultation.id,
        calendarIntegrated: !!req.session?.googleTokens,
        hasConflict: !!conflictWarning
      });
    } catch (error: any) {
      console.error('Consultation scheduling error:', error);
      res.status(400).json({ 
        success: false, 
        message: error.message || "Invalid consultation data", 
        errors: error.errors || error.message 
      });
    }
  });

  app.put("/api/consultations/:id/cancel", async (req, res) => {
    try {
      const consultationId = req.params.id;
      const consultation = await storage.updateConsultationStatus(consultationId, 'cancelled');
      res.json(consultation);
    } catch (error) {
      console.error("Error cancelling consultation:", error);
      res.status(500).json({ message: "Failed to cancel consultation" });
    }
  });

  app.put("/api/consultations/:id/reschedule", async (req, res) => {
    try {
      const consultationId = req.params.id;
      const { scheduledDate, scheduledTime } = req.body;
      
      const consultation = await storage.rescheduleConsultation(consultationId, scheduledDate, scheduledTime || "10:00");
      res.json(consultation);
    } catch (error) {
      console.error("Error rescheduling consultation:", error);
      res.status(500).json({ message: "Failed to reschedule consultation" });
    }
  });

  app.put("/api/consultations/:id/complete", async (req, res) => {
    try {
      const consultationId = req.params.id;
      const { notes } = req.body;
      
      const consultation = await storage.completeConsultation(consultationId, notes);
      
      res.json(consultation);
    } catch (error) {
      console.error("Error completing consultation:", error);
      res.status(500).json({ message: "Failed to complete consultation" });
    }
  });

  // ===== SMS CAMPAIGN MANAGEMENT API ROUTES =====
  
  // Get all SMS campaigns
  app.get("/api/sms-campaigns", async (req, res) => {
    try {
      const campaigns = await storage.getSMSCampaigns();
      res.json(campaigns);
    } catch (error) {
      console.error("Error fetching SMS campaigns:", error);
      res.status(500).json({ message: "Failed to fetch SMS campaigns" });
    }
  });

  // Get SMS campaign statistics
  app.get("/api/sms-campaigns/stats", async (req, res) => {
    try {
      const stats = await storage.getSMSCampaignStats();
      res.json(stats);
    } catch (error) {
      console.error("Error fetching SMS campaign stats:", error);
      res.status(500).json({ message: "Failed to fetch SMS campaign statistics" });
    }
  });

  // Create new SMS campaign
  app.post("/api/sms-campaigns", async (req, res) => {
    try {
      const { name, message, campaignType, targetAudience, tierTarget, scheduledAt } = req.body;
      
      if (!name || !message || !campaignType || !targetAudience) {
        return res.status(400).json({ message: "Missing required fields" });
      }

      if (message.length > 320) {
        return res.status(400).json({ message: "Message too long. Maximum 320 characters." });
      }

      const campaign = await storage.createSMSCampaign({
        name,
        message,
        campaignType,
        targetAudience,
        tierTarget,
        scheduledAt: scheduledAt ? new Date(scheduledAt) : null,
        status: 'draft'
      });
      
      res.json(campaign);
    } catch (error) {
      console.error("Error creating SMS campaign:", error);
      res.status(500).json({ message: "Failed to create SMS campaign" });
    }
  });

  // Send SMS campaign
  app.post("/api/sms-campaigns/:id/send", async (req, res) => {
    try {
      const campaignId = req.params.id;
      
      if (!ringCentralService.isConfigured()) {
        return res.status(400).json({ 
          message: "RingCentral is not configured. Please check your API credentials." 
        });
      }

      const campaign = await storage.getSMSCampaign(campaignId);
      if (!campaign) {
        return res.status(404).json({ message: "Campaign not found" });
      }

      if (campaign.status !== 'draft') {
        return res.status(400).json({ message: "Campaign has already been sent" });
      }

      // Get recipient list based on target audience
      let recipients: Array<{ phone: string; type: string; id: string }> = [];

      if (campaign.targetAudience === 'all' || campaign.targetAudience === 'members') {
        const members = await storage.getUsersWithSMSOptIn(campaign.tierTarget);
        recipients.push(...members.map(m => ({ 
          phone: m.phone!, 
          type: 'member', 
          id: m.id 
        })));
      }

      if (campaign.targetAudience === 'all' || campaign.targetAudience === 'non-members') {
        const subscribers = await storage.getActiveSMSSubscribers();
        recipients.push(...subscribers.map(s => ({ 
          phone: s.phone, 
          type: 'subscriber', 
          id: s.id 
        })));
      }

      if (recipients.length === 0) {
        return res.status(400).json({ message: "No recipients found for this campaign" });
      }

      // Update campaign status
      await storage.updateSMSCampaignStatus(campaignId, 'sending', recipients.length);

      let successCount = 0;
      let failureCount = 0;

      // Process recipients in batches for high volume sending
      const batchSize = 100;
      for (let i = 0; i < recipients.length; i += batchSize) {
        const batch = recipients.slice(i, i + batchSize);
        
        try {
          // Format phone numbers for RingCentral
          const formattedPhones = batch
            .map(r => ringCentralService.formatPhoneNumber(r.phone))
            .filter(phone => ringCentralService.isValidPhoneNumber(phone));

          if (formattedPhones.length > 0) {
            if (formattedPhones.length === 1) {
              // Single SMS
              await ringCentralService.sendSMS({
                to: formattedPhones,
                text: campaign.message
              });
            } else {
              // Batch SMS
              await ringCentralService.sendBatchSMS({
                from: process.env.RINGCENTRAL_USERNAME || '',
                messages: formattedPhones.map(phone => ({ to: [phone] })),
                text: campaign.message
              });
            }
            
            successCount += formattedPhones.length;
            
            // Log successful sends
            for (let j = 0; j < batch.length; j++) {
              const recipient = batch[j];
              await storage.createSMSLog({
                campaignId,
                recipientPhone: recipient.phone,
                recipientType: recipient.type,
                recipientId: recipient.id,
                message: campaign.message,
                status: 'sent'
              });
            }
          }
        } catch (error) {
          console.error(`Failed to send batch ${i}-${i + batch.length}:`, error);
          failureCount += batch.length;
          
          // Log failed sends
          for (const recipient of batch) {
            await storage.createSMSLog({
              campaignId,
              recipientPhone: recipient.phone,
              recipientType: recipient.type,
              recipientId: recipient.id,
              message: campaign.message,
              status: 'failed',
              errorMessage: error instanceof Error ? error.message : 'Unknown error'
            });
          }
        }

        // Small delay between batches to avoid rate limiting
        if (i + batchSize < recipients.length) {
          await new Promise(resolve => setTimeout(resolve, 1000));
        }
      }

      // Update campaign with final results
      await storage.completeSMSCampaign(campaignId, successCount, failureCount);

      res.json({ 
        success: true, 
        totalRecipients: recipients.length,
        successful: successCount,
        failed: failureCount
      });
    } catch (error) {
      console.error("Error sending SMS campaign:", error);
      res.status(500).json({ message: "Failed to send SMS campaign" });
    }
  });

  // SMS subscriber management
  app.get("/api/sms-subscribers", async (req, res) => {
    try {
      const subscribers = await storage.getAllSMSSubscribers();
      res.json(subscribers);
    } catch (error) {
      console.error("Error fetching SMS subscribers:", error);
      res.status(500).json({ message: "Failed to fetch SMS subscribers" });
    }
  });

  // Add SMS subscriber (public endpoint for opt-ins)
  app.post("/api/sms-subscribers", async (req, res) => {
    try {
      const { phone, firstName, lastName, email, source, preferences } = req.body;
      
      if (!phone) {
        return res.status(400).json({ message: "Phone number is required" });
      }

      if (!ringCentralService.isValidPhoneNumber(phone)) {
        return res.status(400).json({ message: "Invalid phone number format" });
      }

      const formattedPhone = ringCentralService.formatPhoneNumber(phone);

      const subscriber = await storage.createSMSSubscriber({
        phone: formattedPhone,
        firstName,
        lastName,
        email,
        source: source || 'website',
        preferences: preferences || 'all',
        isActive: true
      });
      
      res.json(subscriber);
    } catch (error: any) {
      if (error.code === '23505') { // Unique constraint violation
        return res.status(400).json({ message: "Phone number is already subscribed" });
      }
      console.error("Error creating SMS subscriber:", error);
      res.status(500).json({ message: "Failed to create SMS subscriber" });
    }
  });

  // SMS opt-out endpoint
  app.post("/api/sms-subscribers/:phone/opt-out", async (req, res) => {
    try {
      const phone = req.params.phone;
      const formattedPhone = ringCentralService.formatPhoneNumber(phone);
      
      await storage.optOutSMSSubscriber(formattedPhone);
      res.json({ success: true, message: "Successfully opted out of SMS notifications" });
    } catch (error) {
      console.error("Error opting out SMS subscriber:", error);
      res.status(500).json({ message: "Failed to opt out of SMS notifications" });
    }
  });

  const httpServer = createServer(app);
  return httpServer;
}
