import axios from 'axios';

// Zoom API configuration
const ZOOM_API_BASE = 'https://api.zoom.us/v2';
const ZOOM_OAUTH_BASE = 'https://zoom.us/oauth';

interface ZoomAccessToken {
  access_token: string;
  token_type: string;
  expires_in: number;
  scope: string;
}

interface ZoomMeeting {
  id: number;
  uuid: string;
  host_id: string;
  topic: string;
  type: number;
  status: string;
  start_time: string;
  duration: number;
  timezone: string;
  agenda: string;
  created_at: string;
  start_url: string;
  join_url: string;
  password: string;
  h323_password: string;
  pstn_password: string;
  encrypted_password: string;
  settings: {
    host_video: boolean;
    participant_video: boolean;
    cn_meeting: boolean;
    in_meeting: boolean;
    join_before_host: boolean;
    jbh_time: number;
    mute_upon_entry: boolean;
    watermark: boolean;
    use_pmi: boolean;
    approval_type: number;
    audio: string;
    auto_recording: string;
    enforce_login: boolean;
    enforce_login_domains: string;
    alternative_hosts: string;
    close_registration: boolean;
    show_share_button: boolean;
    allow_multiple_devices: boolean;
    registrants_confirmation_email: boolean;
    waiting_room: boolean;
    request_permission_to_unmute_participants: boolean;
    global_dial_in_countries: string[];
    global_dial_in_numbers: any[];
    registrants_email_notification: boolean;
    meeting_authentication: boolean;
    encryption_type: string;
    approved_or_denied_countries_or_regions: {
      enable: boolean;
    };
  };
}

interface ZoomWebinar {
  uuid: string;
  id: number;
  host_id: string;
  topic: string;
  type: number;
  start_time: string;
  duration: number;
  timezone: string;
  agenda: string;
  created_at: string;
  start_url: string;
  join_url: string;
  password: string;
  settings: {
    host_video: boolean;
    panelists_video: boolean;
    practice_session: boolean;
    hd_video: boolean;
    approval_type: number;
    registration_type: number;
    audio: string;
    auto_recording: string;
    enforce_login: boolean;
    enforce_login_domains: string;
    alternative_hosts: string;
    close_registration: boolean;
    show_share_button: boolean;
    allow_multiple_devices: boolean;
    on_demand: boolean;
    global_dial_in_countries: string[];
    global_dial_in_numbers: any[];
    contact_name: string;
    contact_email: string;
    registrants_confirmation_email: boolean;
    registrants_restrict_number: number;
    notify_registrants: boolean;
    post_webinar_survey: boolean;
    survey_url: string;
    registrants_email_notification: boolean;
    meeting_authentication: boolean;
    authentication_option: string;
    authentication_domains: string;
    authentication_name: string;
  };
}

export class ZoomService {
  private accessToken: string | null = null;
  private tokenExpiry: number = 0;

  constructor() {}

  /**
   * Get or refresh access token for Zoom API
   */
  private async getAccessToken(): Promise<string> {
    // Check if current token is still valid
    if (this.accessToken && Date.now() < this.tokenExpiry) {
      return this.accessToken;
    }

    try {
      // Debug: Check if all required environment variables are present
      if (!process.env.ZOOM_API_KEY || !process.env.ZOOM_API_SECRET || !process.env.ZOOM_ACCOUNT_ID) {
        throw new Error('Missing required Zoom credentials: ZOOM_API_KEY, ZOOM_API_SECRET, or ZOOM_ACCOUNT_ID');
      }

      console.log('Zoom Auth Debug:', {
        hasApiKey: !!process.env.ZOOM_API_KEY,
        hasApiSecret: !!process.env.ZOOM_API_SECRET,
        hasAccountId: !!process.env.ZOOM_ACCOUNT_ID,
        accountIdLength: process.env.ZOOM_ACCOUNT_ID?.length,
        apiKeyLength: process.env.ZOOM_API_KEY?.length
      });

      const credentials = Buffer.from(
        `${process.env.ZOOM_API_KEY}:${process.env.ZOOM_API_SECRET}`
      ).toString('base64');

      const response = await axios.post(
        `${ZOOM_OAUTH_BASE}/token?grant_type=account_credentials&account_id=${process.env.ZOOM_ACCOUNT_ID}`,
        {},
        {
          headers: {
            'Authorization': `Basic ${credentials}`,
            'Content-Type': 'application/x-www-form-urlencoded',
          },
        }
      );

      const tokenData: ZoomAccessToken = response.data;
      this.accessToken = tokenData.access_token;
      this.tokenExpiry = Date.now() + (tokenData.expires_in - 300) * 1000; // Refresh 5 minutes early

      return this.accessToken;
    } catch (error) {
      console.error('Failed to get Zoom access token:', error);
      throw new Error('Failed to authenticate with Zoom API');
    }
  }

  /**
   * Create a scheduled training session
   */
  async createTrainingSession(data: {
    topic: string;
    agenda: string;
    startTime: string;
    duration: number;
    timezone?: string;
    password?: string;
    waitingRoom?: boolean;
    allowRecording?: boolean;
  }): Promise<ZoomMeeting> {
    const token = await this.getAccessToken();

    const meetingData = {
      topic: data.topic,
      type: 2, // Scheduled meeting
      start_time: data.startTime,
      duration: data.duration,
      timezone: data.timezone || 'America/New_York',
      agenda: data.agenda,
      password: data.password,
      settings: {
        host_video: true,
        participant_video: false,
        cn_meeting: false,
        in_meeting: false,
        join_before_host: true,
        jbh_time: 5,
        mute_upon_entry: true,
        watermark: false,
        use_pmi: false,
        approval_type: 0, // Automatically approve
        audio: 'both',
        auto_recording: data.allowRecording ? 'cloud' : 'none',
        enforce_login: false,
        waiting_room: data.waitingRoom ?? true,
        allow_multiple_devices: true,
        request_permission_to_unmute_participants: true,
        meeting_authentication: false,
      },
    };

    try {
      const response = await axios.post(
        `${ZOOM_API_BASE}/users/me/meetings`,
        meetingData,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        }
      );

      return response.data;
    } catch (error) {
      console.error('Failed to create Zoom meeting:', error);
      throw new Error('Failed to create training session');
    }
  }

  /**
   * Create a webinar for larger training events
   */
  async createWebinar(data: {
    topic: string;
    agenda: string;
    startTime: string;
    duration: number;
    timezone?: string;
    password?: string;
    registrationRequired?: boolean;
    practiceSession?: boolean;
  }): Promise<ZoomWebinar> {
    const token = await this.getAccessToken();

    const webinarData = {
      topic: data.topic,
      type: 5, // Webinar
      start_time: data.startTime,
      duration: data.duration,
      timezone: data.timezone || 'America/New_York',
      agenda: data.agenda,
      password: data.password,
      settings: {
        host_video: true,
        panelists_video: true,
        practice_session: data.practiceSession ?? true,
        hd_video: true,
        approval_type: data.registrationRequired ? 1 : 0,
        registration_type: data.registrationRequired ? 1 : 3,
        audio: 'both',
        auto_recording: 'cloud',
        enforce_login: false,
        allow_multiple_devices: true,
        on_demand: false,
        contact_name: 'TMH Global Training',
        contact_email: 'training@tmhglobal.com',
        registrants_confirmation_email: true,
        notify_registrants: true,
        post_webinar_survey: false,
        registrants_email_notification: true,
        meeting_authentication: false,
      },
    };

    try {
      const response = await axios.post(
        `${ZOOM_API_BASE}/users/me/webinars`,
        webinarData,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        }
      );

      return response.data;
    } catch (error) {
      console.error('Failed to create Zoom webinar:', error);
      throw new Error('Failed to create webinar');
    }
  }

  /**
   * Get meeting details
   */
  async getMeeting(meetingId: string): Promise<ZoomMeeting> {
    const token = await this.getAccessToken();

    try {
      const response = await axios.get(
        `${ZOOM_API_BASE}/meetings/${meetingId}`,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
          },
        }
      );

      return response.data;
    } catch (error) {
      console.error('Failed to get Zoom meeting:', error);
      throw new Error('Failed to retrieve meeting details');
    }
  }

  /**
   * Update meeting details
   */
  async updateMeeting(meetingId: string, updates: Partial<{
    topic: string;
    agenda: string;
    start_time: string;
    duration: number;
    password: string;
  }>): Promise<void> {
    const token = await this.getAccessToken();

    try {
      await axios.patch(
        `${ZOOM_API_BASE}/meetings/${meetingId}`,
        updates,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        }
      );
    } catch (error) {
      console.error('Failed to update Zoom meeting:', error);
      throw new Error('Failed to update meeting');
    }
  }

  /**
   * Delete/cancel a meeting
   */
  async deleteMeeting(meetingId: string): Promise<void> {
    const token = await this.getAccessToken();

    try {
      await axios.delete(
        `${ZOOM_API_BASE}/meetings/${meetingId}`,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
          },
        }
      );
    } catch (error) {
      console.error('Failed to delete Zoom meeting:', error);
      throw new Error('Failed to cancel meeting');
    }
  }

  /**
   * Get list of user's meetings
   */
  async getMeetings(type: 'scheduled' | 'live' | 'upcoming' = 'scheduled'): Promise<ZoomMeeting[]> {
    const token = await this.getAccessToken();

    try {
      const response = await axios.get(
        `${ZOOM_API_BASE}/users/me/meetings?type=${type}`,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
          },
        }
      );

      return response.data.meetings || [];
    } catch (error) {
      console.error('Failed to get Zoom meetings:', error);
      throw new Error('Failed to retrieve meetings');
    }
  }

  /**
   * Create instant meeting using PMI (Personal Meeting ID)
   * Since Zoom no longer allows meeting creation for most apps, we use PMI as a workaround
   */
  async createInstantMeeting(topic: string, agenda?: string, userPmi?: { pmi: string; password?: string }): Promise<ZoomMeeting> {
    // Use user's configured PMI or fallback to environment variables
    const pmiMeetingId = userPmi?.pmi || process.env.ZOOM_PMI || '1234567890';
    const password = userPmi?.password || process.env.ZOOM_PMI_PASSWORD || '';
    const joinUrl = password ? 
      `https://zoom.us/j/${pmiMeetingId}?pwd=${password}` : 
      `https://zoom.us/j/${pmiMeetingId}`;
    
    // Return a meeting-like object for compatibility
    const pmiMeeting: ZoomMeeting = {
      id: parseInt(pmiMeetingId),
      uuid: `pmi-${Date.now()}`,
      host_id: 'host',
      topic: topic,
      type: 4, // Personal Meeting Room
      status: 'waiting',
      start_time: new Date().toISOString(),
      duration: 60,
      timezone: 'America/New_York',
      agenda: agenda || 'Instant training session',
      created_at: new Date().toISOString(),
      start_url: joinUrl, // Same as join for PMI
      join_url: joinUrl,
      password: password,
      h323_password: '',
      pstn_password: '',
      encrypted_password: '',
      settings: {
        host_video: true,
        participant_video: false,
        cn_meeting: false,
        in_meeting: false,
        join_before_host: true,
        jbh_time: 0,
        mute_upon_entry: true,
        watermark: false,
        use_pmi: true,
        approval_type: 0,
        audio: 'both',
        auto_recording: 'none',
        enforce_login: false,
        enforce_login_domains: '',
        alternative_hosts: '',
        close_registration: false,
        show_share_button: true,
        allow_multiple_devices: true,
        registrants_confirmation_email: true,
        waiting_room: true,
        request_permission_to_unmute_participants: false,
        global_dial_in_countries: [],
        global_dial_in_numbers: [],
        registrants_email_notification: true,
        meeting_authentication: false,
        encryption_type: 'enhanced_encryption',
        approved_or_denied_countries_or_regions: {
          enable: false
        }
      }
    };

    console.log(`Using PMI-based meeting: ${joinUrl}`);
    return pmiMeeting;
  }

  /**
   * Add registrant to webinar
   */
  async addWebinarRegistrant(webinarId: string, registrant: {
    email: string;
    first_name: string;
    last_name: string;
    organization?: string;
    job_title?: string;
  }): Promise<{ id: string; join_url: string; registrant_id: string }> {
    const token = await this.getAccessToken();

    try {
      const response = await axios.post(
        `${ZOOM_API_BASE}/webinars/${webinarId}/registrants`,
        registrant,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        }
      );

      return response.data;
    } catch (error) {
      console.error('Failed to add webinar registrant:', error);
      throw new Error('Failed to register for webinar');
    }
  }
}

export const zoomService = new ZoomService();