import fetch from 'node-fetch';
import { storage } from '../storage.js';

interface RingCentralConfig {
  clientId: string;
  clientSecret: string;
  server: string; // production or sandbox
  redirectUri: string;
  username?: string;
  extension?: string;
  password?: string;
}

interface SMSMessage {
  to: string[];
  text: string;
  from?: string;
}

interface BatchSMSMessage {
  to: string[];
  text?: string;
}

interface BatchSMSRequest {
  from: string;
  text?: string;
  messages: BatchSMSMessage[];
}

interface RingCentralResponse {
  access_token: string;
  token_type: string;
  expires_in: number;
  refresh_token?: string;
  scope: string;
  owner_id?: string;
  endpoint_id?: string;
}

interface RingCentralTokens {
  accessToken: string;
  refreshToken?: string;
  expiresAt: Date;
}

interface SMSResponse {
  id: number;
  uri: string;
  batchId?: string;
  from: { phoneNumber: string };
  to: Array<{ phoneNumber: string; messageStatus: string }>;
  text: string;
  subject?: string;
  creationTime: string;
  lastModifiedTime: string;
  messageStatus: string;
  conversationId: number;
  direction: string;
  availability: string;
  readStatus: string;
}

export class RingCentralService {
  private config: RingCentralConfig;
  private accessToken: string | null = null;
  private refreshToken: string | null = null;
  private tokenExpiry: Date | null = null;
  private defaultFromNumber: string | null = null;

  constructor() {
    this.config = {
      clientId: process.env.RINGCENTRAL_CLIENT_ID || '',
      clientSecret: process.env.RINGCENTRAL_CLIENT_SECRET || '',
      server: process.env.RINGCENTRAL_SERVER === 'default' ? 'https://platform.ringcentral.com' : (process.env.RINGCENTRAL_SERVER || 'https://platform.ringcentral.com'),
      redirectUri: process.env.RINGCENTRAL_REDIRECT_URI || `https://203e87de-1cfe-419c-b966-9bcc47fea614-00-1pr4mm00uvzid.kirk.replit.dev/api/auth/ringcentral/callback`,
      username: process.env.RINGCENTRAL_USERNAME || '',
      extension: process.env.RINGCENTRAL_EXTENSION || '',
      password: process.env.RINGCENTRAL_PASSWORD || ''
    };

    if (!this.config.clientId || !this.config.clientSecret) {
      console.warn('RingCentral credentials not configured. SMS functionality will not work.');
    }
  }

  /**
   * Update configuration dynamically
   */
  updateConfig(clientId: string, clientSecret: string): void {
    console.log(`🔧 Updating RingCentral config from ${this.config.clientId} to ${clientId}`);
    this.config.clientId = clientId;
    this.config.clientSecret = clientSecret;
    console.log('🔧 RingCentral configuration updated with new credentials');
  }

  /**
   * Update server environment
   */
  updateServer(serverInput: 'production' | 'sandbox' | string): void {
    if (serverInput === 'production' || serverInput === 'default') {
      this.config.server = 'https://platform.ringcentral.com';
    } else if (serverInput === 'sandbox') {
      this.config.server = 'https://platform.devtest.ringcentral.com';
    } else {
      this.config.server = serverInput;
    }
    console.log(`🔧 RingCentral server updated to: ${this.config.server}`);
  }

  /**
   * Update redirect URI
   */
  updateRedirectUri(uri?: string): void {
    if (uri) {
      this.config.redirectUri = uri;
      console.log(`🔧 RingCentral redirect URI updated to: ${this.config.redirectUri}`);
    }
  }

  /**
   * Get current configuration (for debugging)
   */
  getConfig(): RingCentralConfig & { authUrl?: string } {
    try {
      const authUrl = this.getAuthorizationUrl();
      return { ...this.config, authUrl };
    } catch {
      return this.config;
    }
  }

  /**
   * Generate OAuth authorization URL
   */
  getAuthorizationUrl(): string {
    const params = new URLSearchParams({
      response_type: 'code',
      client_id: this.config.clientId,
      redirect_uri: this.config.redirectUri,
      scope: 'SMS'
    });
    
    return `${this.config.server}/restapi/oauth/authorize?${params.toString()}`;
  }

  /**
   * Exchange authorization code for access token
   */
  async exchangeCodeForToken(code: string): Promise<RingCentralTokens> {
    try {
      console.log('🔧 Exchanging code for RingCentral tokens...');
      
      const response = await fetch(`${this.config.server}/restapi/oauth/token`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
          'Authorization': `Basic ${Buffer.from(`${this.config.clientId}:${this.config.clientSecret}`).toString('base64')}`
        },
        body: new URLSearchParams({
          grant_type: 'authorization_code',
          code: code,
          redirect_uri: this.config.redirectUri
        }).toString()
      });

      if (!response.ok) {
        const errorText = await response.text();
        console.error('🔧 RingCentral OAuth Error:', {
          status: response.status,
          statusText: response.statusText,
          body: errorText
        });
        throw new Error(`Token exchange failed: ${response.status} ${response.statusText} - ${errorText}`);
      }

      const data = await response.json() as RingCentralResponse;
      console.log('🔧 RingCentral token exchange successful');
      
      // Store tokens
      this.accessToken = data.access_token;
      this.refreshToken = data.refresh_token || null;
      this.tokenExpiry = new Date(Date.now() + (data.expires_in * 1000) - 300000); // Expire 5 min early

      return {
        accessToken: data.access_token,
        refreshToken: data.refresh_token,
        expiresAt: this.tokenExpiry
      };
    } catch (error) {
      console.error('RingCentral token exchange error:', error);
      throw error;
    }
  }

  /**
   * Set tokens manually (for persistence)
   */
  setTokens(tokens: RingCentralTokens): void {
    this.accessToken = tokens.accessToken;
    this.refreshToken = tokens.refreshToken || null;
    // Handle both Date objects and strings (from session serialization)
    this.tokenExpiry = tokens.expiresAt instanceof Date ? tokens.expiresAt : new Date(tokens.expiresAt);
    
    // Auto-discover SMS sender number when tokens are set
    this.discoverDefaultSmsNumber().catch(error => {
      console.warn('Could not discover default SMS number:', error.message);
    });
  }

  /**
   * Check if we have valid tokens
   */
  hasValidTokens(): boolean {
    if (!this.accessToken || !this.tokenExpiry) {
      return false;
    }
    // Convert to timestamp for reliable comparison (handles both Date and string)
    const expiryTime = this.tokenExpiry instanceof Date ? this.tokenExpiry.getTime() : new Date(this.tokenExpiry).getTime();
    return expiryTime > (Date.now() + 5000); // 5 second buffer
  }

  /**
   * Refresh access token using refresh token
   */
  private async refreshAccessToken(): Promise<void> {
    if (!this.refreshToken) {
      throw new Error('No refresh token available. Re-authorization required.');
    }

    try {
      const response = await fetch(`${this.config.server}/restapi/oauth/token`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
          'Authorization': `Basic ${Buffer.from(`${this.config.clientId}:${this.config.clientSecret}`).toString('base64')}`
        },
        body: new URLSearchParams({
          grant_type: 'refresh_token',
          refresh_token: this.refreshToken
        })
      });

      if (!response.ok) {
        const errorText = await response.text();
        console.error('🔧 RingCentral Token Refresh Error:', errorText);
        throw new Error(`Token refresh failed: ${response.status} ${response.statusText} - ${errorText}`);
      }

      const data = await response.json() as RingCentralResponse;
      this.accessToken = data.access_token;
      if (data.refresh_token) {
        this.refreshToken = data.refresh_token;
      }
      this.tokenExpiry = new Date(Date.now() + (data.expires_in * 1000) - 300000); // Expire 5 min early
      
      console.log('RingCentral token refreshed successfully');

      // Persist refreshed tokens to database
      try {
        const adminUser = await storage.getUserByUsername("tmh_admin");
        if (adminUser) {
          await storage.upsertOAuthTokens(adminUser.id, "ringcentral", {
            accessToken: this.accessToken,
            refreshToken: this.refreshToken || undefined,
            expiresAt: this.tokenExpiry,
            scope: null,
            accountId: null,
            metadata: null
          });
          console.log('✅ Refreshed RingCentral tokens saved to database');
        }
      } catch (dbError) {
        console.error('❌ Failed to save refreshed RingCentral tokens to database:', dbError);
        // Don't throw here - token refresh was successful, database save failure shouldn't break SMS functionality
      }
    } catch (error) {
      console.error('RingCentral token refresh error:', error);
      throw error;
    }
  }

  /**
   * Check if token is valid and refresh if needed
   */
  private async ensureAuthenticated(): Promise<void> {
    if (!this.accessToken) {
      throw new Error('No access token available. OAuth authorization required.');
    }
    
    if (!this.tokenExpiry || new Date() >= this.tokenExpiry) {
      await this.refreshAccessToken();
    }
  }

  /**
   * Auto-discover SMS-enabled phone numbers
   */
  private async discoverDefaultSmsNumber(): Promise<void> {
    if (!this.accessToken) return;
    
    try {
      const response = await fetch(`${this.config.server}/restapi/v1.0/account/~/extension/~/phone-number?usageType=SmsSender`, {
        headers: {
          'Authorization': `Bearer ${this.accessToken}`
        }
      });
      
      if (response.ok) {
        const data = await response.json();
        const smsNumbers = data.records || [];
        
        // Prefer DirectNumber type, fallback to any SMS-enabled number
        let selectedNumber = smsNumbers.find((num: any) => num.type === 'DirectNumber');
        if (!selectedNumber && smsNumbers.length > 0) {
          selectedNumber = smsNumbers[0];
        }
        
        if (selectedNumber) {
          this.defaultFromNumber = selectedNumber.phoneNumber;
          console.log(`📱 SMS sender number discovered: ${this.defaultFromNumber}`);
        } else {
          console.warn('⚠️ No SMS-enabled numbers found on this RingCentral account');
          // Use known SMS-enabled number as fallback
          this.defaultFromNumber = '+17038297277';
          console.log(`📱 Using configured SMS number: ${this.defaultFromNumber}`);
        }
      } else {
        const errorText = await response.text();
        console.error(`❌ Failed to discover SMS numbers: ${response.status} - ${errorText}`);
        // Use known SMS-enabled number as fallback
        this.defaultFromNumber = '+17038297277';
        console.log(`📱 Using configured SMS number as fallback: ${this.defaultFromNumber}`);
      }
    } catch (error) {
      console.error('Error discovering SMS sender numbers:', error);
    }
  }

  /**
   * Send a single SMS message
   */
  async sendSMS(message: SMSMessage): Promise<SMSResponse> {
    await this.ensureAuthenticated();
    
    // Ensure we have a default SMS number
    if (!this.defaultFromNumber) {
      await this.discoverDefaultSmsNumber();
      if (!this.defaultFromNumber) {
        throw new Error('No SMS-enabled phone number found on RingCentral account. Please assign a direct number with SMS feature.');
      }
    }

    try {
      const fromNumber = message.from || this.defaultFromNumber;
      console.log(`📱 Sending SMS from ${fromNumber} to ${message.to.join(', ')}`);
      
      const response = await fetch(`${this.config.server}/restapi/v1.0/account/~/extension/~/sms`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${this.accessToken}`
        },
        body: JSON.stringify({
          from: { phoneNumber: fromNumber },
          to: message.to.map(phone => ({ phoneNumber: phone })),
          text: message.text
        })
      });

      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`SMS send failed: ${response.status} ${errorText}`);
      }

      const result = await response.json() as SMSResponse;
      console.log(`SMS sent successfully to ${message.to.join(', ')}`);
      return result;
    } catch (error) {
      console.error('SMS send error:', error);
      throw error;
    }
  }

  /**
   * Send batch/bulk SMS messages (High Volume SMS API)
   */
  async sendBatchSMS(request: BatchSMSRequest): Promise<any> {
    await this.ensureAuthenticated();

    try {
      const response = await fetch(`${this.config.server}/restapi/v1.0/account/~/a2p-sms/batches`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${this.accessToken}`
        },
        body: JSON.stringify(request)
      });

      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`Batch SMS send failed: ${response.status} ${errorText}`);
      }

      const result = await response.json();
      console.log(`Batch SMS initiated for ${request.messages.length} recipients`);
      return result;
    } catch (error) {
      console.error('Batch SMS send error:', error);
      throw error;
    }
  }

  /**
   * Format phone number for RingCentral (must be in E.164 format)
   */
  formatPhoneNumber(phone: string): string {
    // Remove all non-digit characters
    const digits = phone.replace(/\D/g, '');
    
    // If it's 10 digits, assume US number and add +1
    if (digits.length === 10) {
      return `+1${digits}`;
    }
    
    // If it's 11 digits and starts with 1, add +
    if (digits.length === 11 && digits.startsWith('1')) {
      return `+${digits}`;
    }
    
    // If it already starts with +, return as is
    if (phone.startsWith('+')) {
      return phone;
    }
    
    // Otherwise, add + prefix
    return `+${digits}`;
  }

  /**
   * Validate phone number format
   */
  isValidPhoneNumber(phone: string): boolean {
    const formatted = this.formatPhoneNumber(phone);
    // Basic E.164 validation: + followed by 7-15 digits
    return /^\+[1-9]\d{6,14}$/.test(formatted);
  }

  /**
   * Get SMS delivery status
   */
  async getSMSStatus(messageId: string): Promise<any> {
    await this.ensureAuthenticated();

    try {
      const response = await fetch(`${this.config.server}/restapi/v1.0/account/~/extension/~/message-store/${messageId}`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${this.accessToken}`
        }
      });

      if (!response.ok) {
        throw new Error(`Failed to get SMS status: ${response.status}`);
      }

      return await response.json();
    } catch (error) {
      console.error('SMS status check error:', error);
      throw error;
    }
  }

  /**
   * Check if the service is properly configured
   */
  isConfigured(): boolean {
    return !!(
      this.config.clientId &&
      this.config.clientSecret
    );
  }
}

// Singleton instance
export const ringCentralService = new RingCentralService();