import { google } from 'googleapis';
import { OAuth2Client } from 'google-auth-library';

interface EmailMessage {
  to: string;
  subject: string;
  text: string;
  html?: string;
}

export class GmailService {
  private oauth2Client: OAuth2Client;
  private gmail: any;

  constructor() {
    if (!process.env.GOOGLE_CLIENT_ID || !process.env.GOOGLE_CLIENT_SECRET) {
      throw new Error('Gmail credentials not configured');
    }

    this.oauth2Client = new google.auth.OAuth2(
      process.env.GOOGLE_CLIENT_ID,
      process.env.GOOGLE_CLIENT_SECRET,
      process.env.REPLIT_DOMAINS ? `https://${process.env.REPLIT_DOMAINS.split(',')[0]}/api/auth/google/callback` : 'http://localhost:5000/api/auth/google/callback'
    );

    this.gmail = google.gmail({ version: 'v1', auth: this.oauth2Client });
  }

  // Set credentials (use same tokens as calendar)
  setCredentials(tokens: any) {
    this.oauth2Client.setCredentials(tokens);
  }

  // Create raw email message
  private createRawMessage(message: EmailMessage): string {
    const messageParts = [
      `To: ${message.to}`,
      `Subject: ${message.subject}`,
      'Content-Type: text/html; charset=utf-8',
      '',
      message.html || message.text
    ];
    
    return Buffer.from(messageParts.join('\r\n')).toString('base64')
      .replace(/\+/g, '-')
      .replace(/\//g, '_')
      .replace(/=+$/, '');
  }

  // Send email notification
  async sendEmail(message: EmailMessage): Promise<boolean> {
    try {
      const rawMessage = this.createRawMessage(message);
      
      await this.gmail.users.messages.send({
        userId: 'me',
        requestBody: {
          raw: rawMessage
        }
      });

      console.log(`Email sent successfully to ${message.to}`);
      return true;
    } catch (error) {
      console.error('Gmail send error:', error);
      return false;
    }
  }

  // Send new contact notification to admin
  async sendContactNotification(contact: any): Promise<boolean> {
    const message: EmailMessage = {
      to: 'tmhglobal@gmail.com', // Your business email
      subject: `New Contact Form Submission - ${contact.firstName} ${contact.lastName}`,
      text: `New Contact: ${contact.firstName} ${contact.lastName} (${contact.email}) - ${contact.serviceInterest}`,
      html: `
        <h2>New Contact Form Submission</h2>
        <div style="font-family: Arial, sans-serif; max-width: 600px;">
          <h3>Contact Information:</h3>
          <ul>
            <li><strong>Name:</strong> ${contact.firstName} ${contact.lastName}</li>
            <li><strong>Email:</strong> ${contact.email}</li>
            <li><strong>Phone:</strong> ${contact.phone || 'Not provided'}</li>
            <li><strong>Service Interest:</strong> ${contact.serviceInterest}</li>
            <li><strong>Urgent:</strong> ${contact.isUrgent ? 'YES' : 'No'}</li>
            <li><strong>SMS Opt-in:</strong> ${contact.smsOptIn ? 'YES' : 'No'}</li>
          </ul>
          
          <h3>Message:</h3>
          <p style="background: #f5f5f5; padding: 15px; border-radius: 5px;">
            ${contact.message}
          </p>
          
          <p style="margin-top: 20px;">
            <strong>Submitted:</strong> ${new Date(contact.createdAt).toLocaleString()}
          </p>
          
          ${contact.smsOptIn ? '<p style="color: #007cba;"><strong>Note:</strong> This client opted in for SMS updates - add them to your SMS campaigns!</p>' : ''}
        </div>
      `
    };

    return await this.sendEmail(message);
  }

  // Send consultation notification email
  async sendConsultationNotification(consultation: any): Promise<boolean> {
    const message: EmailMessage = {
      to: 'tmhglobal@gmail.com', // Your business email
      subject: `New Consultation Scheduled - ${consultation.firstName} ${consultation.lastName}`,
      text: `New Consultation: ${consultation.firstName} ${consultation.lastName} (${consultation.email}) - ${consultation.serviceType} scheduled for ${new Date(consultation.preferredDate).toLocaleDateString()}`,
      html: `
        <h2>New Consultation Scheduled</h2>
        <div style="font-family: Arial, sans-serif; max-width: 600px;">
          <h3>Client Information:</h3>
          <ul>
            <li><strong>Name:</strong> ${consultation.firstName} ${consultation.lastName}</li>
            <li><strong>Email:</strong> ${consultation.email}</li>
            <li><strong>Phone:</strong> ${consultation.phone || 'Not provided'}</li>
            <li><strong>Service Type:</strong> ${consultation.serviceType}</li>
            <li><strong>Preferred Date:</strong> ${new Date(consultation.preferredDate).toLocaleString()}</li>
            <li><strong>SMS Opt-in:</strong> ${consultation.smsOptIn ? 'YES' : 'No'}</li>
          </ul>
          
          <h3>Message:</h3>
          <p style="background: #f5f5f5; padding: 15px; border-radius: 5px;">
            ${consultation.message}
          </p>
          
          <p style="margin-top: 20px;">
            <strong>Scheduled:</strong> ${new Date(consultation.createdAt).toLocaleString()}
          </p>
          
          ${consultation.calendarEventId ? '<p style="color: #28a745;"><strong>✅ Google Calendar Event Created</strong></p>' : '<p style="color: #dc3545;">❌ Google Calendar Event NOT Created</p>'}
          ${consultation.smsOptIn ? '<p style="color: #007cba;"><strong>Note:</strong> This client opted in for SMS updates - add them to your SMS campaigns!</p>' : ''}
        </div>
      `
    };

    return await this.sendEmail(message);
  }
}

// Export singleton instance
export const gmailService = new GmailService();