import express, { type Request, Response, NextFunction } from "express";
import session from "express-session";
import connectPgSimple from "connect-pg-simple";
import { pool } from "./db";
import { registerRoutes } from "./routes";
import { setupVite, serveStatic, log } from "./vite";
import { storage } from "./storage";
import { googleCalendarService } from "./integrations/google-calendar";
import { ringCentralService } from "./integrations/ringcentral";

// Add global error handling to prevent uncaught exceptions from crashing the server
process.on('uncaughtException', (error) => {
  console.error('Uncaught Exception:', error);
  console.error('Stack:', error.stack);
  // Don't exit the process, just log the error
});

process.on('unhandledRejection', (reason, promise) => {
  console.error('Unhandled Rejection at:', promise, 'reason:', reason);
  // Don't exit the process, just log the error
});

const app = express();

// Configure session store with PostgreSQL persistence
const PgSession = connectPgSimple(session);

// Ensure session secret is configured
if (!process.env.SESSION_SECRET) {
  console.warn('⚠️  SESSION_SECRET not set. Using fallback for development only.');
  if (process.env.NODE_ENV === 'production') {
    throw new Error('SESSION_SECRET must be set in production environment');
  }
}

// Session configuration with PostgreSQL persistence for Google Calendar tokens
app.use(session({
  store: new PgSession({
    pool: pool,
    tableName: 'user_sessions',
    createTableIfMissing: true
  }),
  secret: process.env.SESSION_SECRET || 'tmh-global-dev-fallback-secret',
  resave: false,
  saveUninitialized: false,
  cookie: {
    secure: process.env.NODE_ENV === 'production', // Secure in production
    httpOnly: true,
    maxAge: 24 * 60 * 60 * 1000, // 24 hours
    sameSite: 'lax' // CSRF protection
  }
}));

app.use(express.json());
app.use(express.urlencoded({ extended: false }));

// Token hydration middleware - Load OAuth tokens from database into session
app.use(async (req, res, next) => {
  try {
    // Only hydrate for API requests to avoid unnecessary database calls
    if (req.path.startsWith('/api')) {
      // Get admin user for this single-user system
      const adminUser = await storage.getUserByUsername("tmh_admin");
      if (adminUser) {
        // Initialize session if needed
        if (!req.session) {
          req.session = {};
        }

        // Hydrate Google tokens if not in session
        if (!req.session.googleTokens) {
          const googleTokens = await storage.getOAuthTokens(adminUser.id, "google");
          if (googleTokens) {
            req.session.googleTokens = {
              access_token: googleTokens.accessToken,
              refresh_token: googleTokens.refreshToken || undefined,
              scope: googleTokens.scope || '',
              token_type: 'Bearer',
              expiry_date: googleTokens.expiresAt ? googleTokens.expiresAt.getTime() : undefined
            };
            // Set tokens in the service
            googleCalendarService.setCredentials(req.session.googleTokens);
            console.log('🔄 Google tokens hydrated from database');
          }
        }

        // Hydrate RingCentral tokens if not in session
        if (!req.session.ringCentralTokens) {
          const ringCentralTokens = await storage.getOAuthTokens(adminUser.id, "ringcentral");
          if (ringCentralTokens) {
            req.session.ringCentralTokens = {
              accessToken: ringCentralTokens.accessToken,
              refreshToken: ringCentralTokens.refreshToken || undefined,
              expiresAt: ringCentralTokens.expiresAt || new Date()
            };
            // Set tokens in the service
            ringCentralService.setTokens(req.session.ringCentralTokens);
            console.log('🔄 RingCentral tokens hydrated from database');
          }
        }
      }
    }
  } catch (error) {
    console.error('❌ Token hydration error:', error);
    // Don't block the request if token hydration fails
  }
  next();
});

app.use((req, res, next) => {
  const start = Date.now();
  const path = req.path;
  let capturedJsonResponse: Record<string, any> | undefined = undefined;

  const originalResJson = res.json;
  res.json = function (bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };

  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path.startsWith("/api")) {
      let logLine = `${req.method} ${path} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }

      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "…";
      }

      log(logLine);
    }
  });

  next();
});

(async () => {
  const server = await registerRoutes(app);

  app.use((err: any, _req: Request, res: Response, _next: NextFunction) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";
    
    console.error('Express error handler:', err);
    res.status(status).json({ message });
    // Don't rethrow to avoid uncaught exception handler
  });

  // importantly only setup vite in development and after
  // setting up all the other routes so the catch-all route
  // doesn't interfere with the other routes
  if (app.get("env") === "development") {
    await setupVite(app, server);
  } else {
    serveStatic(app);
  }

  // ALWAYS serve the app on the port specified in the environment variable PORT
  // Other ports are firewalled. Default to 5000 if not specified.
  // this serves both the API and the client.
  // It is the only port that is not firewalled.
  const port = parseInt(process.env.PORT || '5000', 10);
  server.listen({
    port,
    host: "0.0.0.0",
    reusePort: true,
  }, () => {
    log(`serving on port ${port}`);
  });
})();
