var __defProp = Object.defineProperty;
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};

// server/index.ts
import express2 from "express";
import session from "express-session";
import connectPgSimple from "connect-pg-simple";

// server/db.ts
import { Pool, neonConfig } from "@neondatabase/serverless";
import { drizzle } from "drizzle-orm/neon-serverless";
import ws from "ws";

// shared/schema.ts
var schema_exports = {};
__export(schema_exports, {
  comments: () => comments,
  complianceAlerts: () => complianceAlerts,
  complianceIssues: () => complianceIssues,
  complianceReports: () => complianceReports,
  complianceScans: () => complianceScans,
  consultations: () => consultations,
  contacts: () => contacts,
  fileUploads: () => fileUploads,
  follows: () => follows,
  insertCommentSchema: () => insertCommentSchema,
  insertComplianceAlertSchema: () => insertComplianceAlertSchema,
  insertComplianceIssueSchema: () => insertComplianceIssueSchema,
  insertComplianceReportSchema: () => insertComplianceReportSchema,
  insertComplianceScanSchema: () => insertComplianceScanSchema,
  insertConsultationSchema: () => insertConsultationSchema,
  insertContactSchema: () => insertContactSchema,
  insertOAuthTokenSchema: () => insertOAuthTokenSchema,
  insertPostSchema: () => insertPostSchema,
  insertSettingsSchema: () => insertSettingsSchema,
  insertSubscriptionSchema: () => insertSubscriptionSchema,
  insertTrainingVideoSchema: () => insertTrainingVideoSchema,
  insertUserSchema: () => insertUserSchema,
  likes: () => likes,
  liveTrainingSessions: () => liveTrainingSessions,
  oauthTokens: () => oauthTokens,
  posts: () => posts,
  settings: () => settings,
  smsCampaigns: () => smsCampaigns,
  smsLogs: () => smsLogs,
  smsSubscribers: () => smsSubscribers,
  subscriptions: () => subscriptions,
  systemMetrics: () => systemMetrics,
  trainingRegistrations: () => trainingRegistrations,
  trainingResources: () => trainingResources,
  trainingVideos: () => trainingVideos,
  usageAlerts: () => usageAlerts,
  users: () => users
});
import { sql } from "drizzle-orm";
import { pgTable, text, varchar, timestamp, boolean, integer, jsonb } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";
var users = pgTable("users", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  username: text("username").notNull().unique(),
  password: text("password").notNull(),
  email: text("email").unique(),
  firstName: text("first_name"),
  lastName: text("last_name"),
  profileImageUrl: text("profile_image_url"),
  bio: text("bio"),
  company: text("company"),
  isPremium: boolean("is_premium").default(false),
  premiumExpiresAt: timestamp("premium_expires_at"),
  squareCustomerId: text("square_customer_id"),
  paypalCustomerId: text("paypal_customer_id"),
  phone: text("phone"),
  smsOptIn: boolean("sms_opt_in").default(false),
  smsPreferences: text("sms_preferences").default("all"),
  // all, training, promotions, reminders
  zoomPmi: text("zoom_pmi"),
  zoomPmiPassword: text("zoom_pmi_password"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});
var posts = pgTable("posts", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  title: text("title").notNull(),
  content: text("content").notNull(),
  imageUrl: text("image_url"),
  category: text("category").default("general"),
  isPublic: boolean("is_public").default(true),
  isPremiumOnly: boolean("is_premium_only").default(false),
  requiredTier: text("required_tier").default("free"),
  // free, starter, professional, premium
  likesCount: integer("likes_count").default(0),
  commentsCount: integer("comments_count").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});
var comments = pgTable("comments", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  postId: varchar("post_id").notNull().references(() => posts.id, { onDelete: "cascade" }),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  content: text("content").notNull(),
  createdAt: timestamp("created_at").defaultNow()
});
var likes = pgTable("likes", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  postId: varchar("post_id").notNull().references(() => posts.id, { onDelete: "cascade" }),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  createdAt: timestamp("created_at").defaultNow()
});
var follows = pgTable("follows", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  followerId: varchar("follower_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  followingId: varchar("following_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  createdAt: timestamp("created_at").defaultNow()
});
var subscriptions = pgTable("subscriptions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  paymentProvider: text("payment_provider").notNull(),
  // square, paypal, venmo, cashapp
  subscriptionId: text("subscription_id"),
  status: text("status").default("active"),
  // active, cancelled, expired
  planType: text("plan_type").default("starter"),
  // starter, professional, premium, enterprise
  amount: integer("amount").notNull(),
  currency: text("currency").default("USD"),
  startDate: timestamp("start_date").defaultNow(),
  expiresAt: timestamp("expires_at"),
  createdAt: timestamp("created_at").defaultNow()
});
var systemMetrics = pgTable("system_metrics", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  metricType: text("metric_type").notNull(),
  // api_requests, storage_usage, user_activity, errors
  metricName: text("metric_name").notNull(),
  // specific metric name
  value: integer("value").notNull(),
  metadata: text("metadata"),
  // JSON string for additional data
  timestamp: timestamp("timestamp").defaultNow()
});
var usageAlerts = pgTable("usage_alerts", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  alertType: text("alert_type").notNull(),
  // limit_warning, limit_exceeded, high_usage
  resource: text("resource").notNull(),
  // database, storage, api_requests, users
  currentValue: integer("current_value").notNull(),
  limitValue: integer("limit_value").notNull(),
  threshold: integer("threshold").notNull(),
  // percentage threshold that triggered alert
  message: text("message").notNull(),
  isResolved: boolean("is_resolved").default(false),
  createdAt: timestamp("created_at").defaultNow()
});
var contacts = pgTable("contacts", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  firstName: text("first_name").notNull(),
  lastName: text("last_name").notNull(),
  email: text("email").notNull(),
  phone: text("phone"),
  serviceInterest: text("service_interest"),
  message: text("message").notNull(),
  isUrgent: boolean("is_urgent").default(false),
  smsOptIn: boolean("sms_opt_in").default(false),
  createdAt: timestamp("created_at").defaultNow()
});
var smsCampaigns = pgTable("sms_campaigns", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  message: text("message").notNull(),
  campaignType: text("campaign_type").notNull(),
  // training, promotion, reminder, announcement
  targetAudience: text("target_audience").notNull(),
  // all, members, non-members, tier-specific
  tierTarget: text("tier_target"),
  // free, starter, professional, premium (for tier-specific campaigns)
  status: text("status").default("draft"),
  // draft, scheduled, sending, completed, cancelled
  scheduledAt: timestamp("scheduled_at"),
  sentAt: timestamp("sent_at"),
  totalRecipients: integer("total_recipients").default(0),
  successfulSends: integer("successful_sends").default(0),
  failedSends: integer("failed_sends").default(0),
  createdBy: varchar("created_by").references(() => users.id, { onDelete: "set null" }),
  createdAt: timestamp("created_at").defaultNow()
});
var smsSubscribers = pgTable("sms_subscribers", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  phone: text("phone").notNull().unique(),
  firstName: text("first_name"),
  lastName: text("last_name"),
  email: text("email"),
  source: text("source").default("website"),
  // website, consultation, contact-form
  preferences: text("preferences").default("all"),
  // all, training, promotions, reminders
  isActive: boolean("is_active").default(true),
  optInDate: timestamp("opt_in_date").defaultNow(),
  optOutDate: timestamp("opt_out_date"),
  createdAt: timestamp("created_at").defaultNow()
});
var smsLogs = pgTable("sms_logs", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  campaignId: varchar("campaign_id").references(() => smsCampaigns.id, { onDelete: "cascade" }),
  recipientPhone: text("recipient_phone").notNull(),
  recipientType: text("recipient_type").notNull(),
  // member, subscriber
  recipientId: varchar("recipient_id"),
  // user_id or subscriber_id
  message: text("message").notNull(),
  status: text("status").notNull(),
  // sent, delivered, failed, bounced
  ringcentralMessageId: text("ringcentral_message_id"),
  errorMessage: text("error_message"),
  deliveredAt: timestamp("delivered_at"),
  sentAt: timestamp("sent_at").defaultNow()
});
var consultations = pgTable("consultations", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  firstName: text("first_name").notNull(),
  lastName: text("last_name").notNull(),
  email: text("email").notNull(),
  phone: text("phone"),
  companyName: text("company_name"),
  serviceType: text("service_type").notNull(),
  preferredDate: text("preferred_date"),
  preferredTime: text("preferred_time"),
  message: text("message"),
  calendarEventId: text("calendar_event_id"),
  // Google Calendar event ID
  createdAt: timestamp("created_at").defaultNow()
});
var trainingVideos = pgTable("training_videos", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  title: text("title").notNull(),
  description: text("description"),
  videoUrl: text("video_url").notNull(),
  captionsUrl: text("captions_url"),
  transcriptUrl: text("transcript_url"),
  category: text("category").notNull(),
  isPublic: boolean("is_public").default(false),
  requiredTier: text("required_tier").default("free"),
  // free, starter, professional, premium
  createdAt: timestamp("created_at").defaultNow()
});
var fileUploads = pgTable("file_uploads", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  originalName: text("original_name").notNull(),
  fileName: text("file_name").notNull(),
  fileSize: text("file_size").notNull(),
  mimeType: text("mime_type").notNull(),
  uploadedBy: text("uploaded_by"),
  uploadedAt: timestamp("uploaded_at").defaultNow()
});
var complianceScans = pgTable("compliance_scans", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  url: text("url").notNull(),
  scanType: text("scan_type").notNull(),
  // 'manual', 'automated', 'scheduled'
  status: text("status").notNull(),
  // 'running', 'completed', 'failed'
  overallScore: integer("overall_score"),
  // 0-100 compliance score
  totalIssues: integer("total_issues").default(0),
  criticalIssues: integer("critical_issues").default(0),
  warningIssues: integer("warning_issues").default(0),
  infoIssues: integer("info_issues").default(0),
  scanDuration: integer("scan_duration"),
  // milliseconds
  userAgent: text("user_agent"),
  createdAt: timestamp("created_at").defaultNow(),
  completedAt: timestamp("completed_at")
});
var complianceIssues = pgTable("compliance_issues", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  scanId: varchar("scan_id").notNull().references(() => complianceScans.id, { onDelete: "cascade" }),
  ruleId: text("rule_id").notNull(),
  // WCAG rule identifier
  severity: text("severity").notNull(),
  // 'critical', 'warning', 'info'
  wcagLevel: text("wcag_level").notNull(),
  // 'A', 'AA', 'AAA'
  section508Rule: text("section508_rule"),
  // Section 508 rule reference
  element: text("element"),
  // CSS selector of problematic element
  message: text("message").notNull(),
  helpUrl: text("help_url"),
  xpath: text("xpath"),
  html: text("html"),
  status: text("status").default("open"),
  // 'open', 'resolved', 'false_positive'
  resolvedAt: timestamp("resolved_at"),
  resolvedBy: text("resolved_by"),
  createdAt: timestamp("created_at").defaultNow()
});
var complianceReports = pgTable("compliance_reports", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  title: text("title").notNull(),
  description: text("description"),
  reportType: text("report_type").notNull(),
  // 'vpat', 'section508', 'wcag', 'audit'
  status: text("status").notNull().default("draft"),
  // 'draft', 'published', 'archived'
  complianceLevel: text("compliance_level"),
  // 'A', 'AA', 'AAA', 'partial', 'non_compliant'
  dateRange: jsonb("date_range"),
  // { start: date, end: date }
  includedPages: jsonb("included_pages"),
  // array of URLs
  scanIds: jsonb("scan_ids"),
  // array of scan IDs included in report
  generatedBy: text("generated_by"),
  publishedAt: timestamp("published_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});
var complianceAlerts = pgTable("compliance_alerts", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  alertType: text("alert_type").notNull(),
  // 'critical_issue', 'compliance_drop', 'new_violations'
  severity: text("severity").notNull(),
  // 'high', 'medium', 'low'
  title: text("title").notNull(),
  message: text("message").notNull(),
  scanId: varchar("scan_id").references(() => complianceScans.id),
  url: text("url"),
  isRead: boolean("is_read").default(false),
  isResolved: boolean("is_resolved").default(false),
  resolvedAt: timestamp("resolved_at"),
  resolvedBy: text("resolved_by"),
  createdAt: timestamp("created_at").defaultNow()
});
var insertUserSchema = createInsertSchema(users).omit({
  id: true,
  createdAt: true,
  updatedAt: true
}).extend({
  username: z.string().min(3, "Username must be at least 3 characters"),
  password: z.string().min(6, "Password must be at least 6 characters"),
  email: z.string().email("Please enter a valid email address").optional()
});
var insertPostSchema = createInsertSchema(posts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  likesCount: true,
  commentsCount: true
}).extend({
  title: z.string().min(1, "Title is required"),
  content: z.string().min(10, "Content must be at least 10 characters"),
  category: z.string().optional()
});
var insertCommentSchema = createInsertSchema(comments).omit({
  id: true,
  createdAt: true
}).extend({
  content: z.string().min(1, "Comment cannot be empty")
});
var insertSubscriptionSchema = createInsertSchema(subscriptions).omit({
  id: true,
  createdAt: true
}).extend({
  paymentProvider: z.enum(["square", "paypal", "venmo", "cashapp"]),
  amount: z.number().positive("Amount must be positive")
});
var insertContactSchema = createInsertSchema(contacts).omit({
  id: true,
  createdAt: true
}).extend({
  firstName: z.string().min(1, "First name is required"),
  lastName: z.string().min(1, "Last name is required"),
  email: z.string().email("Please enter a valid email address"),
  phone: z.string().optional(),
  message: z.string().min(10, "Message must be at least 10 characters")
});
var insertConsultationSchema = createInsertSchema(consultations).omit({
  id: true,
  createdAt: true
}).extend({
  firstName: z.string().min(1, "First name is required"),
  lastName: z.string().min(1, "Last name is required"),
  email: z.string().email("Please enter a valid email address"),
  serviceType: z.string().min(1, "Please select a consultation type"),
  message: z.string().min(10, "Please provide details about your consultation needs (minimum 10 characters)")
});
var insertTrainingVideoSchema = createInsertSchema(trainingVideos).omit({
  id: true,
  createdAt: true
}).extend({
  title: z.string().min(1, "Title is required"),
  videoUrl: z.string().url("Please enter a valid video URL"),
  category: z.string().min(1, "Category is required")
});
var insertComplianceScanSchema = createInsertSchema(complianceScans).omit({
  id: true,
  createdAt: true,
  completedAt: true
}).extend({
  url: z.string().url("Please enter a valid URL"),
  scanType: z.enum(["manual", "automated", "scheduled"]),
  status: z.enum(["running", "completed", "failed"])
});
var insertComplianceIssueSchema = createInsertSchema(complianceIssues).omit({
  id: true,
  createdAt: true,
  resolvedAt: true
}).extend({
  ruleId: z.string().min(1, "Rule ID is required"),
  severity: z.enum(["critical", "warning", "info"]),
  wcagLevel: z.enum(["A", "AA", "AAA"]),
  message: z.string().min(1, "Message is required")
});
var insertComplianceReportSchema = createInsertSchema(complianceReports).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  publishedAt: true
}).extend({
  title: z.string().min(1, "Title is required"),
  reportType: z.enum(["vpat", "section508", "wcag", "audit"]),
  status: z.enum(["draft", "published", "archived"])
});
var insertComplianceAlertSchema = createInsertSchema(complianceAlerts).omit({
  id: true,
  createdAt: true,
  resolvedAt: true
}).extend({
  alertType: z.enum(["critical_issue", "compliance_drop", "new_violations"]),
  severity: z.enum(["high", "medium", "low"]),
  title: z.string().min(1, "Title is required"),
  message: z.string().min(1, "Message is required")
});
var settings = pgTable("settings", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  key: text("key").notNull().unique(),
  value: text("value"),
  encrypted: boolean("encrypted").default(false),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});
var insertSettingsSchema = createInsertSchema(settings);
var liveTrainingSessions = pgTable("live_training_sessions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  title: varchar("title", { length: 255 }).notNull(),
  description: text("description"),
  category: varchar("category", { length: 100 }).notNull(),
  // 'compliance', 'tax', 'hr', 'accounting'
  sessionType: varchar("session_type", { length: 50 }).notNull(),
  // 'meeting', 'webinar', 'workshop'
  // Zoom integration
  zoomMeetingId: varchar("zoom_meeting_id", { length: 50 }),
  // Zoom meeting ID
  zoomMeetingUuid: varchar("zoom_meeting_uuid", { length: 100 }),
  // Zoom meeting UUID
  zoomJoinUrl: varchar("zoom_join_url", { length: 500 }),
  // Join URL for participants
  zoomStartUrl: varchar("zoom_start_url", { length: 500 }),
  // Start URL for host
  zoomPassword: varchar("zoom_password", { length: 50 }),
  // Meeting password
  // Schedule and duration
  scheduledStartTime: timestamp("scheduled_start_time").notNull(),
  scheduledEndTime: timestamp("scheduled_end_time").notNull(),
  duration: integer("duration").notNull(),
  // Minutes
  timezone: varchar("timezone", { length: 50 }).notNull().default("America/New_York"),
  // Instructor and access
  instructorId: varchar("instructor_id", { length: 255 }).notNull(),
  instructorName: varchar("instructor_name", { length: 255 }).notNull(),
  isPremiumOnly: boolean("is_premium_only").notNull().default(false),
  maxParticipants: integer("max_participants").default(100),
  // Session details
  agenda: text("agenda"),
  learningObjectives: text("learning_objectives").array(),
  // Array of objectives
  materials: text("materials").array(),
  // URLs to resources
  recordingEnabled: boolean("recording_enabled").notNull().default(true),
  recordingUrl: varchar("recording_url", { length: 500 }),
  // Post-session recording
  // Status and management
  status: varchar("status", { length: 50 }).notNull().default("scheduled"),
  // 'scheduled', 'live', 'completed', 'cancelled'
  actualStartTime: timestamp("actual_start_time"),
  actualEndTime: timestamp("actual_end_time"),
  // Metadata
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});
var trainingRegistrations = pgTable("training_registrations", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  sessionId: varchar("session_id", { length: 255 }).notNull().references(() => liveTrainingSessions.id),
  userId: varchar("user_id", { length: 255 }).notNull().references(() => users.id),
  // Registration details
  registeredAt: timestamp("registered_at").defaultNow(),
  status: varchar("status", { length: 50 }).notNull().default("registered"),
  // 'registered', 'attended', 'no-show', 'cancelled'
  // Zoom integration
  zoomRegistrantId: varchar("zoom_registrant_id", { length: 100 }),
  // For webinars
  zoomJoinUrl: varchar("zoom_join_url", { length: 500 }),
  // Personal join URL
  // Attendance tracking
  joinedAt: timestamp("joined_at"),
  leftAt: timestamp("left_at"),
  attendanceDuration: integer("attendance_duration"),
  // Minutes attended
  // Feedback and completion
  completed: boolean("completed").notNull().default(false),
  feedbackRating: integer("feedback_rating"),
  // 1-5 stars
  feedbackComments: text("feedback_comments"),
  certificateIssued: boolean("certificate_issued").notNull().default(false),
  // Metadata
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});
var trainingResources = pgTable("training_resources", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  sessionId: varchar("session_id", { length: 255 }).notNull().references(() => liveTrainingSessions.id),
  // Resource details
  title: varchar("title", { length: 255 }).notNull(),
  description: text("description"),
  resourceType: varchar("resource_type", { length: 50 }).notNull(),
  // 'slide', 'document', 'video', 'link'
  fileUrl: varchar("file_url", { length: 500 }),
  // Cloud storage URL
  fileSize: integer("file_size"),
  // File size in bytes
  mimeType: varchar("mime_type", { length: 100 }),
  // Access control
  availableFrom: timestamp("available_from").defaultNow(),
  availableUntil: timestamp("available_until"),
  // Optional expiry
  downloadAllowed: boolean("download_allowed").notNull().default(true),
  // Metadata
  uploadedBy: varchar("uploaded_by", { length: 255 }).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});
var oauthTokens = pgTable("oauth_tokens", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  provider: text("provider").notNull(),
  // 'google', 'ringcentral'
  accessToken: text("access_token").notNull(),
  refreshToken: text("refresh_token"),
  expiresAt: timestamp("expires_at"),
  scope: text("scope"),
  accountId: text("account_id"),
  // Google email, RingCentral extension/owner
  metadata: jsonb("metadata"),
  // Provider-specific data like defaultFromNumber
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
}, (table) => ({
  userProviderUnique: sql`UNIQUE(${table.userId}, ${table.provider})`
}));
var insertOAuthTokenSchema = createInsertSchema(oauthTokens).omit({
  id: true,
  createdAt: true,
  updatedAt: true
}).extend({
  provider: z.enum(["google", "ringcentral"]),
  accessToken: z.string().min(1, "Access token is required"),
  refreshToken: z.string().optional()
});
var insertLiveTrainingSessionSchema = createInsertSchema(liveTrainingSessions);
var insertTrainingRegistrationSchema = createInsertSchema(trainingRegistrations);
var insertTrainingResourceSchema = createInsertSchema(trainingResources);

// server/db.ts
neonConfig.webSocketConstructor = ws;
if (!process.env.DATABASE_URL) {
  throw new Error(
    "DATABASE_URL must be set. Did you forget to provision a database?"
  );
}
var pool = new Pool({
  connectionString: process.env.DATABASE_URL,
  max: 10,
  // Maximum number of connections
  idleTimeoutMillis: 3e4,
  // Close idle connections after 30s
  connectionTimeoutMillis: 1e4
  // Connection timeout
});
pool.on("error", (err, client2) => {
  console.error("Unexpected database pool error:", err);
});
var db = drizzle({ client: pool, schema: schema_exports });
async function withRetry(operation, maxRetries = 3) {
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      return await operation();
    } catch (error) {
      console.error(`Database operation failed (attempt ${attempt}/${maxRetries}):`, error.message);
      if (attempt === maxRetries) {
        throw error;
      }
      await new Promise((resolve) => setTimeout(resolve, Math.pow(2, attempt) * 1e3));
    }
  }
  throw new Error("Maximum retry attempts exceeded");
}

// server/routes.ts
import { createServer } from "http";

// server/storage.ts
import { randomUUID } from "crypto";

// server/user-storage.ts
import { eq } from "drizzle-orm";
var DatabaseUserStorage = class {
  // Get user from database
  async getUser(id) {
    try {
      return await withRetry(async () => {
        return await db.query.users.findFirst({
          where: eq(users.id, id)
        });
      });
    } catch (error) {
      console.error("Error fetching user after retries:", error);
      return void 0;
    }
  }
  // Get user by username
  async getUserByUsername(username) {
    try {
      return await withRetry(async () => {
        return await db.query.users.findFirst({
          where: eq(users.username, username)
        });
      });
    } catch (error) {
      console.error("Error fetching user by username after retries:", error);
      return void 0;
    }
  }
  // Create user in database
  async createUser(userData) {
    try {
      return await withRetry(async () => {
        const [newUser] = await db.insert(users).values({
          ...userData,
          createdAt: /* @__PURE__ */ new Date(),
          updatedAt: /* @__PURE__ */ new Date()
        }).returning();
        return newUser;
      });
    } catch (error) {
      console.error("Error creating user after retries:", error);
      throw new Error("Failed to create user");
    }
  }
  // Update user settings (PMI, etc.)
  async updateUserSettings(id, settings2) {
    try {
      return await withRetry(async () => {
        const [updatedUser] = await db.update(users).set({
          zoomPmi: settings2.zoomPmi,
          zoomPmiPassword: settings2.zoomPmiPassword,
          updatedAt: /* @__PURE__ */ new Date()
        }).where(eq(users.id, id)).returning();
        if (!updatedUser) {
          throw new Error("User not found");
        }
        return updatedUser;
      });
    } catch (error) {
      console.error("Error updating user settings after retries:", error);
      throw new Error("Failed to update user settings");
    }
  }
  // Ensure admin user exists (migration helper)
  async ensureAdminUser() {
    try {
      let adminUser = await this.getUserByUsername("tmh_admin");
      if (!adminUser) {
        adminUser = await this.createUser({
          username: "tmh_admin",
          password: "admin123",
          email: "admin@tmhglobal.com",
          firstName: "TMH",
          lastName: "Global",
          company: "TMH Global LLC",
          bio: "Professional accounting and business services for government contractors",
          isPremium: true,
          smsOptIn: false,
          smsPreferences: "all"
        });
        console.log("\u2705 Admin user created in database");
      }
      return adminUser;
    } catch (error) {
      console.error("Error ensuring admin user:", error);
      throw error;
    }
  }
};
var databaseUserStorage = new DatabaseUserStorage();

// server/storage.ts
import { eq as eq2, and } from "drizzle-orm";
var MemStorage = class {
  users;
  contacts;
  consultations;
  trainingVideos;
  fileUploads;
  posts;
  comments;
  likes;
  follows;
  subscriptions;
  systemMetrics;
  usageAlerts;
  complianceScans;
  complianceIssues;
  complianceAlerts;
  liveTrainingSessions;
  trainingRegistrations;
  trainingResources;
  requestCount;
  constructor() {
    this.users = /* @__PURE__ */ new Map();
    this.contacts = /* @__PURE__ */ new Map();
    this.consultations = /* @__PURE__ */ new Map();
    this.trainingVideos = /* @__PURE__ */ new Map();
    this.fileUploads = /* @__PURE__ */ new Map();
    this.posts = /* @__PURE__ */ new Map();
    this.comments = /* @__PURE__ */ new Map();
    this.likes = /* @__PURE__ */ new Map();
    this.follows = /* @__PURE__ */ new Map();
    this.subscriptions = /* @__PURE__ */ new Map();
    this.systemMetrics = /* @__PURE__ */ new Map();
    this.usageAlerts = /* @__PURE__ */ new Map();
    this.complianceScans = [];
    this.complianceIssues = [];
    this.complianceAlerts = [];
    this.liveTrainingSessions = [];
    this.trainingRegistrations = [];
    this.trainingResources = [];
    this.requestCount = 0;
    this.initializeSampleData();
    this.initializeSampleMonitoring();
    this.ensureDatabaseAdmin();
  }
  initializeSampleData() {
    const adminUser = {
      id: randomUUID(),
      username: "tmh_admin",
      password: "admin123",
      email: "admin@tmhglobal.com",
      firstName: "TMH",
      lastName: "Global",
      profileImageUrl: null,
      bio: "Professional accounting and business services for government contractors",
      company: "TMH Global LLC",
      isPremium: true,
      premiumExpiresAt: null,
      squareCustomerId: null,
      paypalCustomerId: null,
      phone: null,
      smsOptIn: false,
      smsPreferences: "all",
      zoomPmi: null,
      zoomPmiPassword: null,
      createdAt: /* @__PURE__ */ new Date(),
      updatedAt: /* @__PURE__ */ new Date()
    };
    this.users.set(adminUser.id, adminUser);
    const samplePosts = [
      {
        id: randomUUID(),
        userId: adminUser.id,
        title: "Welcome to TMH Global Community",
        content: "We're excited to launch our exclusive community platform for government contractors. Here you'll find expert insights, compliance updates, and connect with fellow professionals in the industry.",
        imageUrl: null,
        category: "general",
        isPublic: true,
        isPremiumOnly: false,
        requiredTier: "free",
        likesCount: 12,
        commentsCount: 5,
        createdAt: new Date(Date.now() - 2 * 24 * 60 * 60 * 1e3),
        // 2 days ago
        updatedAt: /* @__PURE__ */ new Date()
      },
      {
        id: randomUUID(),
        userId: adminUser.id,
        title: "SAM Registration Updates - Important Changes",
        content: "Recent updates to SAM.gov have changed some key requirements for registration. Make sure your business information is current and complete to avoid any delays in contract opportunities.",
        imageUrl: null,
        category: "sam-registration",
        isPublic: true,
        isPremiumOnly: false,
        requiredTier: "free",
        likesCount: 8,
        commentsCount: 3,
        createdAt: new Date(Date.now() - 1 * 24 * 60 * 60 * 1e3),
        // 1 day ago
        updatedAt: /* @__PURE__ */ new Date()
      },
      {
        id: randomUUID(),
        userId: adminUser.id,
        title: "Advanced Compliance Strategies",
        content: "For our premium members: Advanced techniques for maintaining compliance across multiple contract vehicles and agencies. This includes documentation best practices and audit preparation.",
        imageUrl: null,
        category: "compliance",
        isPublic: true,
        isPremiumOnly: true,
        requiredTier: "premium",
        likesCount: 15,
        commentsCount: 8,
        createdAt: new Date(Date.now() - 3 * 60 * 60 * 1e3),
        // 3 hours ago
        updatedAt: /* @__PURE__ */ new Date()
      }
    ];
    samplePosts.forEach((post) => {
      this.posts.set(post.id, post);
    });
    const sampleVideos = [
      {
        id: randomUUID(),
        title: "Introduction to Government Contracting",
        description: "Learn the basics of federal contracting opportunities and requirements",
        videoUrl: "https://example.com/video1",
        captionsUrl: "https://example.com/captions1.vtt",
        transcriptUrl: "https://example.com/transcript1.txt",
        category: "Government Contracting",
        isPublic: true,
        requiredTier: "free",
        createdAt: /* @__PURE__ */ new Date()
      },
      {
        id: randomUUID(),
        title: "SAM.gov Registration Walkthrough",
        description: "Step-by-step guide to registering your business in SAM.gov",
        videoUrl: "https://example.com/video2",
        captionsUrl: "https://example.com/captions2.vtt",
        transcriptUrl: "https://example.com/transcript2.txt",
        category: "SAM Registration",
        isPublic: true,
        requiredTier: "free",
        createdAt: /* @__PURE__ */ new Date()
      },
      {
        id: randomUUID(),
        title: "Section 508 Compliance Basics",
        description: "Understanding accessibility requirements for government contracts",
        videoUrl: "https://example.com/video3",
        captionsUrl: "https://example.com/captions3.vtt",
        transcriptUrl: "https://example.com/transcript3.txt",
        category: "Compliance",
        isPublic: true,
        requiredTier: "premium",
        createdAt: /* @__PURE__ */ new Date()
      }
    ];
    sampleVideos.forEach((video) => {
      this.trainingVideos.set(video.id, video);
    });
    const sampleSessions = [
      {
        id: randomUUID(),
        title: "Section 508 Compliance Workshop",
        description: "Interactive workshop covering digital accessibility requirements for federal contractors",
        category: "compliance",
        sessionType: "webinar",
        zoomMeetingId: "123456789",
        zoomMeetingUuid: "abc123",
        zoomJoinUrl: "https://zoom.us/j/123456789",
        zoomStartUrl: "https://zoom.us/s/123456789",
        zoomPassword: "workshop2025",
        scheduledStartTime: new Date(Date.now() + 24 * 60 * 60 * 1e3),
        // Tomorrow
        scheduledEndTime: new Date(Date.now() + 24 * 60 * 60 * 1e3 + 2 * 60 * 60 * 1e3),
        // Tomorrow + 2 hours
        duration: 120,
        timezone: "America/New_York",
        instructorId: "instructor-1",
        instructorName: "Sarah Johnson",
        isPremiumOnly: false,
        maxParticipants: 100,
        agenda: "1. Understanding Section 508\n2. WCAG 2.1 AA Standards\n3. Testing Tools and Techniques\n4. Common Compliance Issues\n5. Q&A Session",
        learningObjectives: [
          "Understand Section 508 requirements for government contractors",
          "Learn WCAG 2.1 AA compliance standards",
          "Master accessibility testing tools and techniques",
          "Identify and fix common accessibility issues"
        ],
        materials: null,
        recordingEnabled: true,
        recordingUrl: null,
        status: "scheduled",
        actualStartTime: null,
        actualEndTime: null,
        createdAt: /* @__PURE__ */ new Date(),
        updatedAt: /* @__PURE__ */ new Date()
      },
      {
        id: randomUUID(),
        title: "Government Contracting Basics",
        description: "Essential knowledge for small businesses entering government contracting",
        category: "business",
        sessionType: "meeting",
        zoomMeetingId: "987654321",
        zoomMeetingUuid: "xyz789",
        zoomJoinUrl: "https://zoom.us/j/987654321",
        zoomStartUrl: "https://zoom.us/s/987654321",
        zoomPassword: "govcon101",
        scheduledStartTime: new Date(Date.now() + 3 * 24 * 60 * 60 * 1e3),
        // 3 days from now
        scheduledEndTime: new Date(Date.now() + 3 * 24 * 60 * 60 * 1e3 + 90 * 60 * 1e3),
        // 3 days + 90 minutes
        duration: 90,
        timezone: "America/New_York",
        instructorId: "instructor-2",
        instructorName: "Michael Chen",
        isPremiumOnly: true,
        maxParticipants: 50,
        agenda: "1. Introduction to Government Contracting\n2. S.A.M. Registration Process\n3. NAICS Codes and Set-Asides\n4. Proposal Writing Basics\n5. Compliance Requirements",
        learningObjectives: [
          "Navigate the government contracting landscape",
          "Complete S.A.M. registration successfully",
          "Understand NAICS codes and set-aside programs",
          "Write competitive proposals"
        ],
        materials: null,
        recordingEnabled: true,
        recordingUrl: null,
        status: "scheduled",
        actualStartTime: null,
        actualEndTime: null,
        createdAt: /* @__PURE__ */ new Date(),
        updatedAt: /* @__PURE__ */ new Date()
      },
      {
        id: randomUUID(),
        title: "Tax Preparation for Contractors - COMPLETED",
        description: "Year-end tax strategies and deductions for government contractors",
        category: "tax",
        sessionType: "webinar",
        zoomMeetingId: "555666777",
        zoomMeetingUuid: "completed123",
        zoomJoinUrl: "https://zoom.us/j/555666777",
        zoomStartUrl: "https://zoom.us/s/555666777",
        zoomPassword: "tax2024",
        scheduledStartTime: new Date(Date.now() - 7 * 24 * 60 * 60 * 1e3),
        // 7 days ago
        scheduledEndTime: new Date(Date.now() - 7 * 24 * 60 * 60 * 1e3 + 2 * 60 * 60 * 1e3),
        // 7 days ago + 2 hours
        duration: 120,
        timezone: "America/New_York",
        instructorId: "instructor-3",
        instructorName: "Jennifer Martinez",
        isPremiumOnly: false,
        maxParticipants: 200,
        agenda: "1. Contractor Tax Obligations\n2. Deductible Business Expenses\n3. Quarterly Payment Strategies\n4. Record Keeping Best Practices",
        learningObjectives: [
          "Understand contractor tax obligations",
          "Maximize deductible business expenses",
          "Plan quarterly tax payments effectively",
          "Implement proper record keeping"
        ],
        materials: null,
        recordingEnabled: true,
        recordingUrl: "https://zoom.us/rec/share/tax-recording-2024",
        status: "completed",
        actualStartTime: new Date(Date.now() - 7 * 24 * 60 * 60 * 1e3),
        actualEndTime: new Date(Date.now() - 7 * 24 * 60 * 60 * 1e3 + 2 * 60 * 60 * 1e3),
        createdAt: new Date(Date.now() - 14 * 24 * 60 * 60 * 1e3),
        updatedAt: new Date(Date.now() - 7 * 24 * 60 * 60 * 1e3)
      }
    ];
    this.liveTrainingSessions = sampleSessions;
  }
  // Ensure admin user exists in database for PMI persistence
  async ensureDatabaseAdmin() {
    try {
      await databaseUserStorage.ensureAdminUser();
      console.log("\u2705 Admin user verified in database for PMI persistence");
    } catch (error) {
      console.error("\u274C Failed to ensure admin user in database:", error);
    }
  }
  // User methods - Now using database storage for persistence
  async getUser(id) {
    return await databaseUserStorage.getUser(id);
  }
  async getUserByUsername(username) {
    return await databaseUserStorage.getUserByUsername(username);
  }
  async createUser(insertUser) {
    return await databaseUserStorage.createUser(insertUser);
  }
  async updateUserSettings(id, settings2) {
    return await databaseUserStorage.updateUserSettings(id, settings2);
  }
  // Social features - Posts
  async createPost(insertPost) {
    const id = randomUUID();
    const post = {
      ...insertPost,
      id,
      imageUrl: insertPost.imageUrl || null,
      category: insertPost.category || "general",
      isPublic: insertPost.isPublic ?? true,
      isPremiumOnly: insertPost.isPremiumOnly ?? false,
      likesCount: 0,
      commentsCount: 0,
      createdAt: /* @__PURE__ */ new Date(),
      updatedAt: /* @__PURE__ */ new Date()
    };
    this.posts.set(id, post);
    return post;
  }
  async getAllPosts() {
    return Array.from(this.posts.values()).sort(
      (a, b) => (b.createdAt?.getTime() || 0) - (a.createdAt?.getTime() || 0)
    );
  }
  async getPost(id) {
    return this.posts.get(id);
  }
  async updatePostCounts(postId, likesCount, commentsCount) {
    const post = this.posts.get(postId);
    if (post) {
      post.likesCount = likesCount;
      post.commentsCount = commentsCount;
      post.updatedAt = /* @__PURE__ */ new Date();
      this.posts.set(postId, post);
    }
  }
  // Social features - Comments
  async createComment(insertComment) {
    const id = randomUUID();
    const comment = {
      ...insertComment,
      id,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.comments.set(id, comment);
    const comments2 = await this.getCommentsByPost(insertComment.postId);
    const post = await this.getPost(insertComment.postId);
    if (post) {
      await this.updatePostCounts(insertComment.postId, post.likesCount || 0, comments2.length);
    }
    return comment;
  }
  async getCommentsByPost(postId) {
    return Array.from(this.comments.values()).filter((comment) => comment.postId === postId).sort((a, b) => (a.createdAt?.getTime() || 0) - (b.createdAt?.getTime() || 0));
  }
  // Social features - Likes
  async createLike(postId, userId) {
    const id = randomUUID();
    const like = {
      id,
      postId,
      userId,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.likes.set(id, like);
    const likes2 = Array.from(this.likes.values()).filter((l) => l.postId === postId);
    const post = await this.getPost(postId);
    if (post) {
      await this.updatePostCounts(postId, likes2.length, post.commentsCount || 0);
    }
    return like;
  }
  async deleteLike(postId, userId) {
    const like = Array.from(this.likes.values()).find(
      (l) => l.postId === postId && l.userId === userId
    );
    if (like) {
      this.likes.delete(like.id);
      const likes2 = Array.from(this.likes.values()).filter((l) => l.postId === postId);
      const post = await this.getPost(postId);
      if (post) {
        await this.updatePostCounts(postId, likes2.length, post.commentsCount || 0);
      }
    }
  }
  async getLike(postId, userId) {
    return Array.from(this.likes.values()).find(
      (like) => like.postId === postId && like.userId === userId
    );
  }
  // Social features - Follows
  async createFollow(followerId, followingId) {
    const id = randomUUID();
    const follow = {
      id,
      followerId,
      followingId,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.follows.set(id, follow);
    return follow;
  }
  async deleteFollow(followerId, followingId) {
    const follow = Array.from(this.follows.values()).find(
      (f) => f.followerId === followerId && f.followingId === followingId
    );
    if (follow) {
      this.follows.delete(follow.id);
    }
  }
  async getFollows(userId) {
    return Array.from(this.follows.values()).filter(
      (follow) => follow.followerId === userId
    );
  }
  // Premium subscriptions
  async createSubscription(insertSubscription) {
    const id = randomUUID();
    const subscription = {
      ...insertSubscription,
      id,
      subscriptionId: insertSubscription.subscriptionId || null,
      status: insertSubscription.status || "active",
      planType: insertSubscription.planType || "premium",
      currency: insertSubscription.currency || "USD",
      startDate: /* @__PURE__ */ new Date(),
      expiresAt: insertSubscription.expiresAt || null,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.subscriptions.set(id, subscription);
    return subscription;
  }
  async getSubscription(userId) {
    return Array.from(this.subscriptions.values()).find(
      (sub) => sub.userId === userId
    );
  }
  async updateSubscription(userId, status) {
    const subscription = await this.getSubscription(userId);
    if (subscription) {
      subscription.status = status;
      this.subscriptions.set(subscription.id, subscription);
    }
  }
  // Contact methods
  async createContact(insertContact) {
    const id = randomUUID();
    const contact = {
      ...insertContact,
      serviceInterest: insertContact.serviceInterest || null,
      phone: insertContact.phone || null,
      id,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.contacts.set(id, contact);
    return contact;
  }
  async getAllContacts() {
    return Array.from(this.contacts.values()).sort(
      (a, b) => (b.createdAt?.getTime() || 0) - (a.createdAt?.getTime() || 0)
    );
  }
  async getContact(id) {
    return this.contacts.get(id);
  }
  // Consultation methods
  async createConsultation(insertConsultation) {
    const id = randomUUID();
    const consultation = {
      ...insertConsultation,
      phone: insertConsultation.phone || null,
      message: insertConsultation.message || null,
      companyName: insertConsultation.companyName || null,
      preferredDate: insertConsultation.preferredDate || null,
      preferredTime: insertConsultation.preferredTime || null,
      id,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.consultations.set(id, consultation);
    return consultation;
  }
  async getAllConsultations() {
    return Array.from(this.consultations.values()).sort(
      (a, b) => (b.createdAt?.getTime() || 0) - (a.createdAt?.getTime() || 0)
    );
  }
  async getConsultation(id) {
    return this.consultations.get(id);
  }
  // Training video methods
  async createTrainingVideo(insertVideo) {
    const id = randomUUID();
    const video = {
      ...insertVideo,
      description: insertVideo.description || null,
      captionsUrl: insertVideo.captionsUrl || null,
      transcriptUrl: insertVideo.transcriptUrl || null,
      isPublic: insertVideo.isPublic ?? false,
      id,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.trainingVideos.set(id, video);
    return video;
  }
  async getAllTrainingVideos() {
    return Array.from(this.trainingVideos.values()).sort(
      (a, b) => (b.createdAt?.getTime() || 0) - (a.createdAt?.getTime() || 0)
    );
  }
  async getPublicTrainingVideos() {
    return Array.from(this.trainingVideos.values()).filter((video) => video.isPublic).sort((a, b) => (b.createdAt?.getTime() || 0) - (a.createdAt?.getTime() || 0));
  }
  async getTrainingVideo(id) {
    return this.trainingVideos.get(id);
  }
  // Monitoring methods
  async recordMetric(metricType, metricName, value, metadata) {
    const id = randomUUID();
    const metric = {
      id,
      metricType,
      metricName,
      value,
      metadata: metadata || null,
      timestamp: /* @__PURE__ */ new Date()
    };
    this.systemMetrics.set(id, metric);
    return metric;
  }
  async getMetrics(type, hours) {
    let metrics = Array.from(this.systemMetrics.values());
    if (type) {
      metrics = metrics.filter((m) => m.metricType === type);
    }
    if (hours) {
      const cutoff = new Date(Date.now() - hours * 60 * 60 * 1e3);
      metrics = metrics.filter((m) => m.timestamp && m.timestamp > cutoff);
    }
    return metrics.sort((a, b) => (b.timestamp ? b.timestamp.getTime() : 0) - (a.timestamp ? a.timestamp.getTime() : 0));
  }
  async createAlert(alertType, resource, currentValue, limitValue, threshold, message) {
    const id = randomUUID();
    const alert = {
      id,
      alertType,
      resource,
      currentValue,
      limitValue,
      threshold,
      message,
      isResolved: false,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.usageAlerts.set(id, alert);
    return alert;
  }
  async getAlerts(activeOnly = false) {
    let alerts = Array.from(this.usageAlerts.values());
    if (activeOnly) {
      alerts = alerts.filter((a) => !a.isResolved);
    }
    return alerts.sort((a, b) => (b.createdAt ? b.createdAt.getTime() : 0) - (a.createdAt ? a.createdAt.getTime() : 0));
  }
  async incrementRequestCount() {
    this.requestCount++;
    if (this.requestCount % 100 === 0) {
      await this.recordMetric("api_requests", "total_requests", this.requestCount);
    }
  }
  async getRequestCount() {
    return this.requestCount;
  }
  async getSummaryStats() {
    const totalUsers = this.users.size;
    const totalPosts = this.posts.size;
    const totalComments = this.comments.size;
    const premiumUsers = Array.from(this.users.values()).filter((u) => u.isPremium).length;
    const storageUsed = totalUsers * 1024 + totalPosts * 2048 + totalComments * 512;
    return {
      totalUsers,
      totalPosts,
      totalComments,
      apiRequestsToday: this.requestCount,
      storageUsed,
      premiumUsers
    };
  }
  async initializeSampleMonitoring() {
    await this.recordMetric("api_requests", "total_requests", 1250, '{"endpoint": "/api/posts", "method": "GET"}');
    await this.recordMetric("user_activity", "new_registrations", 23, '{"source": "organic"}');
    await this.recordMetric("user_activity", "active_users", 187, '{"timeframe": "daily"}');
    await this.recordMetric("storage_usage", "database_size", 2.1 * 1024 * 1024, '{"table": "posts"}');
    await this.createAlert(
      "high_usage",
      "api_requests",
      1250,
      1500,
      83,
      "API request volume is approaching daily limits. Consider optimizing client-side caching."
    );
    this.requestCount = 1250;
  }
  // File upload methods
  async createFileUpload(fileData) {
    const id = randomUUID();
    const fileUpload = {
      ...fileData,
      id,
      uploadedAt: /* @__PURE__ */ new Date()
    };
    this.fileUploads.set(id, fileUpload);
    return fileUpload;
  }
  async getAllFileUploads() {
    return Array.from(this.fileUploads.values()).sort(
      (a, b) => (b.uploadedAt?.getTime() || 0) - (a.uploadedAt?.getTime() || 0)
    );
  }
  async getFileUpload(id) {
    return this.fileUploads.get(id);
  }
  // Section 508 Compliance operations
  async createComplianceScan(scanData) {
    const scan = {
      id: randomUUID(),
      url: scanData.url || "",
      scanType: scanData.scanType || "manual",
      status: scanData.status || "running",
      overallScore: scanData.overallScore || null,
      totalIssues: scanData.totalIssues || 0,
      criticalIssues: scanData.criticalIssues || 0,
      warningIssues: scanData.warningIssues || 0,
      infoIssues: scanData.infoIssues || 0,
      scanDuration: scanData.scanDuration || null,
      userAgent: scanData.userAgent || null,
      createdAt: /* @__PURE__ */ new Date(),
      completedAt: scanData.completedAt || null
    };
    this.complianceScans.push(scan);
    return scan;
  }
  async updateComplianceScan(id, updates) {
    const scanIndex = this.complianceScans.findIndex((s) => s.id === id);
    if (scanIndex === -1) return null;
    this.complianceScans[scanIndex] = { ...this.complianceScans[scanIndex], ...updates };
    return this.complianceScans[scanIndex];
  }
  async getComplianceScan(id) {
    return this.complianceScans.find((s) => s.id === id) || null;
  }
  async getComplianceScans(url, limit = 50) {
    let scans = [...this.complianceScans];
    if (url) {
      scans = scans.filter((s) => s.url === url);
    }
    return scans.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime()).slice(0, limit);
  }
  async getLatestComplianceScan(url) {
    const scans = this.complianceScans.filter((s) => s.url === url && s.status === "completed").sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
    return scans[0] || null;
  }
  async createComplianceIssue(issueData) {
    const issue = {
      id: randomUUID(),
      scanId: issueData.scanId || "",
      ruleId: issueData.ruleId || "",
      severity: issueData.severity || "info",
      wcagLevel: issueData.wcagLevel || "A",
      section508Rule: issueData.section508Rule || null,
      element: issueData.element || null,
      message: issueData.message || "",
      helpUrl: issueData.helpUrl || null,
      xpath: issueData.xpath || null,
      html: issueData.html || null,
      status: issueData.status || "open",
      resolvedAt: issueData.resolvedAt || null,
      resolvedBy: issueData.resolvedBy || null,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.complianceIssues.push(issue);
    return issue;
  }
  async getComplianceIssues(scanId, status) {
    let issues = [...this.complianceIssues];
    if (scanId) {
      issues = issues.filter((i) => i.scanId === scanId);
    }
    if (status) {
      issues = issues.filter((i) => i.status === status);
    }
    return issues.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }
  async createComplianceAlert(alertData) {
    const alert = {
      id: randomUUID(),
      alertType: alertData.alertType || "new_violations",
      severity: alertData.severity || "medium",
      title: alertData.title || "",
      message: alertData.message || "",
      scanId: alertData.scanId || null,
      url: alertData.url || null,
      isRead: alertData.isRead || false,
      isResolved: alertData.isResolved || false,
      resolvedAt: alertData.resolvedAt || null,
      resolvedBy: alertData.resolvedBy || null,
      createdAt: /* @__PURE__ */ new Date()
    };
    this.complianceAlerts.push(alert);
    return alert;
  }
  async getComplianceAlerts(onlyUnread = false) {
    let alerts = [...this.complianceAlerts];
    if (onlyUnread) {
      alerts = alerts.filter((a) => !a.isRead);
    }
    return alerts.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }
  async markComplianceAlertAsRead(id) {
    const alertIndex = this.complianceAlerts.findIndex((a) => a.id === id);
    if (alertIndex === -1) return null;
    this.complianceAlerts[alertIndex].isRead = true;
    return this.complianceAlerts[alertIndex];
  }
  // Live Training Sessions with Zoom integration
  async createLiveTrainingSession(sessionData) {
    const session2 = {
      id: randomUUID(),
      title: sessionData.title || "",
      description: sessionData.description || null,
      category: sessionData.category || "compliance",
      sessionType: sessionData.sessionType || "meeting",
      // Zoom integration fields
      zoomMeetingId: sessionData.zoomMeetingId || null,
      zoomMeetingUuid: sessionData.zoomMeetingUuid || null,
      zoomJoinUrl: sessionData.zoomJoinUrl || null,
      zoomStartUrl: sessionData.zoomStartUrl || null,
      zoomPassword: sessionData.zoomPassword || null,
      // Schedule and duration
      scheduledStartTime: sessionData.scheduledStartTime || /* @__PURE__ */ new Date(),
      scheduledEndTime: sessionData.scheduledEndTime || /* @__PURE__ */ new Date(),
      duration: sessionData.duration || 60,
      timezone: sessionData.timezone || "America/New_York",
      // Instructor and access
      instructorId: sessionData.instructorId || "",
      instructorName: sessionData.instructorName || "",
      isPremiumOnly: sessionData.isPremiumOnly || false,
      maxParticipants: sessionData.maxParticipants || 100,
      // Session details
      agenda: sessionData.agenda || null,
      learningObjectives: sessionData.learningObjectives || null,
      materials: sessionData.materials || null,
      recordingEnabled: sessionData.recordingEnabled || true,
      recordingUrl: sessionData.recordingUrl || null,
      // Status and management
      status: sessionData.status || "scheduled",
      actualStartTime: sessionData.actualStartTime || null,
      actualEndTime: sessionData.actualEndTime || null,
      // Metadata
      createdAt: /* @__PURE__ */ new Date(),
      updatedAt: /* @__PURE__ */ new Date()
    };
    this.liveTrainingSessions.push(session2);
    return session2;
  }
  async updateLiveTrainingSession(id, updates) {
    const sessionIndex = this.liveTrainingSessions.findIndex((s) => s.id === id);
    if (sessionIndex === -1) return null;
    this.liveTrainingSessions[sessionIndex] = {
      ...this.liveTrainingSessions[sessionIndex],
      ...updates,
      updatedAt: /* @__PURE__ */ new Date()
    };
    return this.liveTrainingSessions[sessionIndex];
  }
  async getLiveTrainingSession(id) {
    return this.liveTrainingSessions.find((s) => s.id === id) || null;
  }
  async getLiveTrainingSessions(category, isPremiumOnly) {
    let sessions = [...this.liveTrainingSessions];
    if (category) {
      sessions = sessions.filter((s) => s.category === category);
    }
    if (isPremiumOnly !== void 0) {
      sessions = sessions.filter((s) => s.isPremiumOnly === isPremiumOnly);
    }
    return sessions.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }
  async getUpcomingTrainingSessions() {
    const now = /* @__PURE__ */ new Date();
    return this.liveTrainingSessions.filter((s) => s.scheduledStartTime > now && s.status === "scheduled").sort((a, b) => a.scheduledStartTime.getTime() - b.scheduledStartTime.getTime());
  }
  async createTrainingRegistration(registrationData) {
    const registration = {
      id: randomUUID(),
      sessionId: registrationData.sessionId || "",
      userId: registrationData.userId || "",
      // Registration details
      registeredAt: /* @__PURE__ */ new Date(),
      status: registrationData.status || "registered",
      // Zoom integration
      zoomRegistrantId: registrationData.zoomRegistrantId || null,
      zoomJoinUrl: registrationData.zoomJoinUrl || null,
      // Attendance tracking
      joinedAt: registrationData.joinedAt || null,
      leftAt: registrationData.leftAt || null,
      attendanceDuration: registrationData.attendanceDuration || null,
      // Feedback and completion
      completed: registrationData.completed || false,
      feedbackRating: registrationData.feedbackRating || null,
      feedbackComments: registrationData.feedbackComments || null,
      certificateIssued: registrationData.certificateIssued || false,
      // Metadata
      createdAt: /* @__PURE__ */ new Date(),
      updatedAt: /* @__PURE__ */ new Date()
    };
    this.trainingRegistrations.push(registration);
    return registration;
  }
  async getTrainingRegistration(sessionId, userId) {
    return this.trainingRegistrations.find((r) => r.sessionId === sessionId && r.userId === userId) || null;
  }
  async getTrainingRegistrations(sessionId, userId) {
    let registrations = [...this.trainingRegistrations];
    if (sessionId) {
      registrations = registrations.filter((r) => r.sessionId === sessionId);
    }
    if (userId) {
      registrations = registrations.filter((r) => r.userId === userId);
    }
    return registrations.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }
  async updateTrainingRegistration(id, updates) {
    const registrationIndex = this.trainingRegistrations.findIndex((r) => r.id === id);
    if (registrationIndex === -1) return null;
    this.trainingRegistrations[registrationIndex] = {
      ...this.trainingRegistrations[registrationIndex],
      ...updates,
      updatedAt: /* @__PURE__ */ new Date()
    };
    return this.trainingRegistrations[registrationIndex];
  }
  async createTrainingResource(resourceData) {
    const resource = {
      id: randomUUID(),
      sessionId: resourceData.sessionId || "",
      // Resource details
      title: resourceData.title || "",
      description: resourceData.description || null,
      resourceType: resourceData.resourceType || "document",
      fileUrl: resourceData.fileUrl || null,
      fileSize: resourceData.fileSize || null,
      mimeType: resourceData.mimeType || null,
      // Access control
      availableFrom: resourceData.availableFrom || /* @__PURE__ */ new Date(),
      availableUntil: resourceData.availableUntil || null,
      downloadAllowed: resourceData.downloadAllowed || true,
      // Metadata
      uploadedBy: resourceData.uploadedBy || "",
      createdAt: /* @__PURE__ */ new Date(),
      updatedAt: /* @__PURE__ */ new Date()
    };
    this.trainingResources.push(resource);
    return resource;
  }
  async getTrainingResources(sessionId) {
    return this.trainingResources.filter((r) => r.sessionId === sessionId).sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
  }
  // ===== SMS CAMPAIGN MANAGEMENT =====
  async getSMSCampaigns() {
    return [];
  }
  async getSMSCampaignStats() {
    return {
      totalCampaigns: 0,
      totalSent: 0,
      activeSubscribers: 0,
      avgDeliveryRate: 0
    };
  }
  async createSMSCampaign(campaign) {
    const newCampaign = {
      id: randomUUID(),
      ...campaign,
      createdAt: /* @__PURE__ */ new Date(),
      updatedAt: /* @__PURE__ */ new Date(),
      totalRecipients: 0,
      successfulSends: 0,
      failedSends: 0
    };
    return newCampaign;
  }
  async getSMSCampaign(id) {
    return null;
  }
  async updateSMSCampaignStatus(id, status, totalRecipients) {
  }
  async completeSMSCampaign(id, successCount, failureCount) {
  }
  async getUsersWithSMSOptIn(tierTarget) {
    const users2 = Array.from(this.users.values());
    return users2.filter(
      (user) => user.phone && user.smsOptIn && (!tierTarget || this.getUserTier(user) === tierTarget)
    );
  }
  async getActiveSMSSubscribers() {
    return [];
  }
  async createSMSLog(log2) {
    const newLog = {
      id: randomUUID(),
      ...log2,
      sentAt: /* @__PURE__ */ new Date()
    };
    return newLog;
  }
  async getAllSMSSubscribers() {
    return [];
  }
  async createSMSSubscriber(subscriber) {
    const newSubscriber = {
      id: randomUUID(),
      ...subscriber,
      optInDate: /* @__PURE__ */ new Date(),
      createdAt: /* @__PURE__ */ new Date()
    };
    return newSubscriber;
  }
  async optOutSMSSubscriber(phone) {
  }
  // ===== TRAINING INTERACTIVE FEATURES =====
  async getTrainingPolls(sessionId) {
    return [];
  }
  async createTrainingPoll(pollData) {
    const newPoll = {
      id: randomUUID(),
      ...pollData,
      createdAt: /* @__PURE__ */ new Date(),
      votes: []
    };
    return newPoll;
  }
  async voteOnTrainingPoll(pollId, optionIndex, userId) {
    return { success: true, pollId, optionIndex, userId };
  }
  async getTrainingQAQuestions(sessionId) {
    return [];
  }
  async createTrainingQAQuestion(questionData) {
    const newQuestion = {
      id: randomUUID(),
      ...questionData,
      createdAt: /* @__PURE__ */ new Date(),
      status: "pending"
    };
    return newQuestion;
  }
  async answerTrainingQAQuestion(questionId, answer, answeredBy) {
    return {
      id: questionId,
      answer,
      answeredBy,
      answeredAt: /* @__PURE__ */ new Date(),
      status: "answered"
    };
  }
  async getTrainingBreakoutRooms(sessionId) {
    return [];
  }
  async createTrainingBreakoutRoom(roomData) {
    const newRoom = {
      id: randomUUID(),
      ...roomData,
      createdAt: /* @__PURE__ */ new Date(),
      participants: []
    };
    return newRoom;
  }
  // ===== ENHANCED CONSULTATION MANAGEMENT =====
  async getConsultationAvailability(date) {
    const requestedDate = new Date(date);
    const today = /* @__PURE__ */ new Date();
    today.setHours(0, 0, 0, 0);
    const minDate = new Date(today);
    minDate.setDate(today.getDate() + 1);
    if (requestedDate < minDate) {
      return [];
    }
    const dayOfWeek = requestedDate.getDay();
    if (dayOfWeek === 0 || dayOfWeek === 2 || dayOfWeek === 6) {
      return [];
    }
    const availableSlots = [
      { time: "09:00", available: true },
      { time: "10:00", available: true },
      { time: "11:00", available: true },
      { time: "13:00", available: true },
      { time: "14:00", available: true },
      { time: "15:00", available: true },
      { time: "16:00", available: true }
    ];
    return availableSlots;
  }
  async updateConsultationStatus(id, status) {
    const consultation = this.consultations.get(id);
    if (consultation) {
      const updated = { ...consultation, status, updatedAt: /* @__PURE__ */ new Date() };
      this.consultations.set(id, updated);
      return updated;
    }
    return null;
  }
  async rescheduleConsultation(id, newDate, newTime) {
    const consultation = this.consultations.get(id);
    if (consultation) {
      const updated = {
        ...consultation,
        preferredDate: newDate,
        preferredTime: newTime,
        updatedAt: /* @__PURE__ */ new Date()
      };
      this.consultations.set(id, updated);
      return updated;
    }
    return null;
  }
  async completeConsultation(id, notes) {
    const consultation = this.consultations.get(id);
    if (consultation) {
      const updated = {
        ...consultation,
        status: "completed",
        notes,
        completedAt: /* @__PURE__ */ new Date(),
        updatedAt: /* @__PURE__ */ new Date()
      };
      this.consultations.set(id, updated);
      return updated;
    }
    return null;
  }
  // ===== COMPLIANCE REPORTS =====
  async getComplianceReport(id) {
    return null;
  }
  // ===== OAUTH TOKEN MANAGEMENT =====
  async upsertOAuthTokens(userId, provider, tokens) {
    const tokenData = {
      userId,
      provider,
      ...tokens
    };
    const [result] = await db.insert(oauthTokens).values(tokenData).onConflictDoUpdate({
      target: [oauthTokens.userId, oauthTokens.provider],
      set: {
        accessToken: tokenData.accessToken,
        refreshToken: tokenData.refreshToken,
        expiresAt: tokenData.expiresAt,
        scope: tokenData.scope,
        accountId: tokenData.accountId,
        metadata: tokenData.metadata,
        updatedAt: /* @__PURE__ */ new Date()
      }
    }).returning();
    return result;
  }
  async getOAuthTokens(userId, provider) {
    const [result] = await db.select().from(oauthTokens).where(and(
      eq2(oauthTokens.userId, userId),
      eq2(oauthTokens.provider, provider)
    )).limit(1);
    return result;
  }
  async deleteOAuthTokens(userId, provider) {
    await db.delete(oauthTokens).where(and(
      eq2(oauthTokens.userId, userId),
      eq2(oauthTokens.provider, provider)
    ));
  }
  getUserTier(user) {
    if (user.isPremium) {
      return "premium";
    }
    return "free";
  }
};
var storage = new MemStorage();

// server/integrations/ringcentral.ts
import fetch from "node-fetch";
var RingCentralService = class {
  config;
  accessToken = null;
  refreshToken = null;
  tokenExpiry = null;
  defaultFromNumber = null;
  constructor() {
    this.config = {
      clientId: process.env.RINGCENTRAL_CLIENT_ID || "",
      clientSecret: process.env.RINGCENTRAL_CLIENT_SECRET || "",
      server: process.env.RINGCENTRAL_SERVER === "default" ? "https://platform.ringcentral.com" : process.env.RINGCENTRAL_SERVER || "https://platform.ringcentral.com",
      redirectUri: process.env.RINGCENTRAL_REDIRECT_URI || `https://203e87de-1cfe-419c-b966-9bcc47fea614-00-1pr4mm00uvzid.kirk.replit.dev/api/auth/ringcentral/callback`,
      username: process.env.RINGCENTRAL_USERNAME || "",
      extension: process.env.RINGCENTRAL_EXTENSION || "",
      password: process.env.RINGCENTRAL_PASSWORD || ""
    };
    if (!this.config.clientId || !this.config.clientSecret) {
      console.warn("RingCentral credentials not configured. SMS functionality will not work.");
    }
  }
  /**
   * Update configuration dynamically
   */
  updateConfig(clientId, clientSecret) {
    console.log(`\u{1F527} Updating RingCentral config from ${this.config.clientId} to ${clientId}`);
    this.config.clientId = clientId;
    this.config.clientSecret = clientSecret;
    console.log("\u{1F527} RingCentral configuration updated with new credentials");
  }
  /**
   * Update server environment
   */
  updateServer(serverInput) {
    if (serverInput === "production" || serverInput === "default") {
      this.config.server = "https://platform.ringcentral.com";
    } else if (serverInput === "sandbox") {
      this.config.server = "https://platform.devtest.ringcentral.com";
    } else {
      this.config.server = serverInput;
    }
    console.log(`\u{1F527} RingCentral server updated to: ${this.config.server}`);
  }
  /**
   * Update redirect URI
   */
  updateRedirectUri(uri) {
    if (uri) {
      this.config.redirectUri = uri;
      console.log(`\u{1F527} RingCentral redirect URI updated to: ${this.config.redirectUri}`);
    }
  }
  /**
   * Get current configuration (for debugging)
   */
  getConfig() {
    try {
      const authUrl = this.getAuthorizationUrl();
      return { ...this.config, authUrl };
    } catch {
      return this.config;
    }
  }
  /**
   * Generate OAuth authorization URL
   */
  getAuthorizationUrl() {
    const params = new URLSearchParams({
      response_type: "code",
      client_id: this.config.clientId,
      redirect_uri: this.config.redirectUri,
      scope: "SMS"
    });
    return `${this.config.server}/restapi/oauth/authorize?${params.toString()}`;
  }
  /**
   * Exchange authorization code for access token
   */
  async exchangeCodeForToken(code) {
    try {
      console.log("\u{1F527} Exchanging code for RingCentral tokens...");
      const response = await fetch(`${this.config.server}/restapi/oauth/token`, {
        method: "POST",
        headers: {
          "Content-Type": "application/x-www-form-urlencoded",
          "Authorization": `Basic ${Buffer.from(`${this.config.clientId}:${this.config.clientSecret}`).toString("base64")}`
        },
        body: new URLSearchParams({
          grant_type: "authorization_code",
          code,
          redirect_uri: this.config.redirectUri
        }).toString()
      });
      if (!response.ok) {
        const errorText = await response.text();
        console.error("\u{1F527} RingCentral OAuth Error:", {
          status: response.status,
          statusText: response.statusText,
          body: errorText
        });
        throw new Error(`Token exchange failed: ${response.status} ${response.statusText} - ${errorText}`);
      }
      const data = await response.json();
      console.log("\u{1F527} RingCentral token exchange successful");
      this.accessToken = data.access_token;
      this.refreshToken = data.refresh_token || null;
      this.tokenExpiry = new Date(Date.now() + data.expires_in * 1e3 - 3e5);
      return {
        accessToken: data.access_token,
        refreshToken: data.refresh_token,
        expiresAt: this.tokenExpiry
      };
    } catch (error) {
      console.error("RingCentral token exchange error:", error);
      throw error;
    }
  }
  /**
   * Set tokens manually (for persistence)
   */
  setTokens(tokens) {
    this.accessToken = tokens.accessToken;
    this.refreshToken = tokens.refreshToken || null;
    this.tokenExpiry = tokens.expiresAt instanceof Date ? tokens.expiresAt : new Date(tokens.expiresAt);
    this.discoverDefaultSmsNumber().catch((error) => {
      console.warn("Could not discover default SMS number:", error.message);
    });
  }
  /**
   * Check if we have valid tokens
   */
  hasValidTokens() {
    if (!this.accessToken || !this.tokenExpiry) {
      return false;
    }
    const expiryTime = this.tokenExpiry instanceof Date ? this.tokenExpiry.getTime() : new Date(this.tokenExpiry).getTime();
    return expiryTime > Date.now() + 5e3;
  }
  /**
   * Refresh access token using refresh token
   */
  async refreshAccessToken() {
    if (!this.refreshToken) {
      throw new Error("No refresh token available. Re-authorization required.");
    }
    try {
      const response = await fetch(`${this.config.server}/restapi/oauth/token`, {
        method: "POST",
        headers: {
          "Content-Type": "application/x-www-form-urlencoded",
          "Authorization": `Basic ${Buffer.from(`${this.config.clientId}:${this.config.clientSecret}`).toString("base64")}`
        },
        body: new URLSearchParams({
          grant_type: "refresh_token",
          refresh_token: this.refreshToken
        })
      });
      if (!response.ok) {
        const errorText = await response.text();
        console.error("\u{1F527} RingCentral Token Refresh Error:", errorText);
        throw new Error(`Token refresh failed: ${response.status} ${response.statusText} - ${errorText}`);
      }
      const data = await response.json();
      this.accessToken = data.access_token;
      if (data.refresh_token) {
        this.refreshToken = data.refresh_token;
      }
      this.tokenExpiry = new Date(Date.now() + data.expires_in * 1e3 - 3e5);
      console.log("RingCentral token refreshed successfully");
      try {
        const adminUser = await storage.getUserByUsername("tmh_admin");
        if (adminUser) {
          await storage.upsertOAuthTokens(adminUser.id, "ringcentral", {
            accessToken: this.accessToken,
            refreshToken: this.refreshToken || void 0,
            expiresAt: this.tokenExpiry,
            scope: null,
            accountId: null,
            metadata: null
          });
          console.log("\u2705 Refreshed RingCentral tokens saved to database");
        }
      } catch (dbError) {
        console.error("\u274C Failed to save refreshed RingCentral tokens to database:", dbError);
      }
    } catch (error) {
      console.error("RingCentral token refresh error:", error);
      throw error;
    }
  }
  /**
   * Check if token is valid and refresh if needed
   */
  async ensureAuthenticated() {
    if (!this.accessToken) {
      throw new Error("No access token available. OAuth authorization required.");
    }
    if (!this.tokenExpiry || /* @__PURE__ */ new Date() >= this.tokenExpiry) {
      await this.refreshAccessToken();
    }
  }
  /**
   * Auto-discover SMS-enabled phone numbers
   */
  async discoverDefaultSmsNumber() {
    if (!this.accessToken) return;
    try {
      const response = await fetch(`${this.config.server}/restapi/v1.0/account/~/extension/~/phone-number?usageType=SmsSender`, {
        headers: {
          "Authorization": `Bearer ${this.accessToken}`
        }
      });
      if (response.ok) {
        const data = await response.json();
        const smsNumbers = data.records || [];
        let selectedNumber = smsNumbers.find((num) => num.type === "DirectNumber");
        if (!selectedNumber && smsNumbers.length > 0) {
          selectedNumber = smsNumbers[0];
        }
        if (selectedNumber) {
          this.defaultFromNumber = selectedNumber.phoneNumber;
          console.log(`\u{1F4F1} SMS sender number discovered: ${this.defaultFromNumber}`);
        } else {
          console.warn("\u26A0\uFE0F No SMS-enabled numbers found on this RingCentral account");
          this.defaultFromNumber = "+17038297277";
          console.log(`\u{1F4F1} Using configured SMS number: ${this.defaultFromNumber}`);
        }
      } else {
        const errorText = await response.text();
        console.error(`\u274C Failed to discover SMS numbers: ${response.status} - ${errorText}`);
        this.defaultFromNumber = "+17038297277";
        console.log(`\u{1F4F1} Using configured SMS number as fallback: ${this.defaultFromNumber}`);
      }
    } catch (error) {
      console.error("Error discovering SMS sender numbers:", error);
    }
  }
  /**
   * Send a single SMS message
   */
  async sendSMS(message) {
    await this.ensureAuthenticated();
    if (!this.defaultFromNumber) {
      await this.discoverDefaultSmsNumber();
      if (!this.defaultFromNumber) {
        throw new Error("No SMS-enabled phone number found on RingCentral account. Please assign a direct number with SMS feature.");
      }
    }
    try {
      const fromNumber = message.from || this.defaultFromNumber;
      console.log(`\u{1F4F1} Sending SMS from ${fromNumber} to ${message.to.join(", ")}`);
      const response = await fetch(`${this.config.server}/restapi/v1.0/account/~/extension/~/sms`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "Authorization": `Bearer ${this.accessToken}`
        },
        body: JSON.stringify({
          from: { phoneNumber: fromNumber },
          to: message.to.map((phone) => ({ phoneNumber: phone })),
          text: message.text
        })
      });
      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`SMS send failed: ${response.status} ${errorText}`);
      }
      const result = await response.json();
      console.log(`SMS sent successfully to ${message.to.join(", ")}`);
      return result;
    } catch (error) {
      console.error("SMS send error:", error);
      throw error;
    }
  }
  /**
   * Send batch/bulk SMS messages (High Volume SMS API)
   */
  async sendBatchSMS(request) {
    await this.ensureAuthenticated();
    try {
      const response = await fetch(`${this.config.server}/restapi/v1.0/account/~/a2p-sms/batches`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "Authorization": `Bearer ${this.accessToken}`
        },
        body: JSON.stringify(request)
      });
      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`Batch SMS send failed: ${response.status} ${errorText}`);
      }
      const result = await response.json();
      console.log(`Batch SMS initiated for ${request.messages.length} recipients`);
      return result;
    } catch (error) {
      console.error("Batch SMS send error:", error);
      throw error;
    }
  }
  /**
   * Format phone number for RingCentral (must be in E.164 format)
   */
  formatPhoneNumber(phone) {
    const digits = phone.replace(/\D/g, "");
    if (digits.length === 10) {
      return `+1${digits}`;
    }
    if (digits.length === 11 && digits.startsWith("1")) {
      return `+${digits}`;
    }
    if (phone.startsWith("+")) {
      return phone;
    }
    return `+${digits}`;
  }
  /**
   * Validate phone number format
   */
  isValidPhoneNumber(phone) {
    const formatted = this.formatPhoneNumber(phone);
    return /^\+[1-9]\d{6,14}$/.test(formatted);
  }
  /**
   * Get SMS delivery status
   */
  async getSMSStatus(messageId) {
    await this.ensureAuthenticated();
    try {
      const response = await fetch(`${this.config.server}/restapi/v1.0/account/~/extension/~/message-store/${messageId}`, {
        method: "GET",
        headers: {
          "Authorization": `Bearer ${this.accessToken}`
        }
      });
      if (!response.ok) {
        throw new Error(`Failed to get SMS status: ${response.status}`);
      }
      return await response.json();
    } catch (error) {
      console.error("SMS status check error:", error);
      throw error;
    }
  }
  /**
   * Check if the service is properly configured
   */
  isConfigured() {
    return !!(this.config.clientId && this.config.clientSecret);
  }
};
var ringCentralService = new RingCentralService();

// server/payments/square.ts
import { randomUUID as randomUUID2 } from "crypto";
var SquarePaymentService = class {
  isConfigured;
  constructor() {
    const accessToken = process.env.SQUARE_ACCESS_TOKEN;
    const applicationId = process.env.SQUARE_APPLICATION_ID;
    if (!accessToken || !applicationId) {
      console.warn("Square credentials not found. Square payments will be unavailable.");
      this.isConfigured = false;
      return;
    }
    this.isConfigured = true;
    console.log("Square payment service initialized successfully");
  }
  async createPayment(amount, sourceId, customerId) {
    if (!this.isConfigured) {
      throw new Error("Square client not initialized. Please check your API credentials.");
    }
    try {
      const payment = {
        id: randomUUID2(),
        status: "COMPLETED",
        amount: Math.round(amount * 100),
        // Convert to cents
        currency: "USD",
        sourceId,
        customerId,
        createdAt: (/* @__PURE__ */ new Date()).toISOString()
      };
      console.log("Square payment created:", payment);
      return { payment };
    } catch (error) {
      console.error("Square API Error:", error);
      throw new Error(`Payment failed: ${error.message || "Unknown error"}`);
    }
  }
  async createCustomer(email, firstName, lastName) {
    if (!this.isConfigured) {
      throw new Error("Square client not initialized");
    }
    try {
      const customer = {
        id: randomUUID2(),
        emailAddress: email,
        givenName: firstName,
        familyName: lastName,
        companyName: "TMH Global LLC",
        createdAt: (/* @__PURE__ */ new Date()).toISOString()
      };
      console.log("Square customer created:", customer);
      return customer;
    } catch (error) {
      console.error("Square Customer Creation Error:", error);
      throw error;
    }
  }
  async createSubscription(customerId, planId) {
    if (!this.isConfigured) {
      throw new Error("Square client not initialized");
    }
    try {
      const subscription = {
        id: randomUUID2(),
        customerId,
        planId,
        status: "ACTIVE",
        startDate: (/* @__PURE__ */ new Date()).toISOString().split("T")[0],
        createdAt: (/* @__PURE__ */ new Date()).toISOString()
      };
      console.log("Square subscription created:", subscription);
      return subscription;
    } catch (error) {
      console.error("Square Subscription Creation Error:", error);
      throw error;
    }
  }
  isAvailable() {
    return this.isConfigured && !!process.env.SQUARE_ACCESS_TOKEN && !!process.env.SQUARE_APPLICATION_ID;
  }
};
var squareService = new SquarePaymentService();

// server/payments/paypal.ts
import { randomUUID as randomUUID3 } from "crypto";
var PayPalPaymentService = class {
  isConfigured;
  constructor() {
    const { PAYPAL_CLIENT_ID: PAYPAL_CLIENT_ID2, PAYPAL_CLIENT_SECRET: PAYPAL_CLIENT_SECRET2 } = process.env;
    if (!PAYPAL_CLIENT_ID2 || !PAYPAL_CLIENT_SECRET2) {
      console.warn("PayPal credentials not found. PayPal payments will be unavailable.");
      this.isConfigured = false;
      return;
    }
    this.isConfigured = true;
    console.log("PayPal payment service initialized successfully");
  }
  async getClientToken() {
    if (!this.isConfigured) {
      throw new Error("PayPal client not initialized");
    }
    return `client_token_${randomUUID3()}`;
  }
  async createOrder(amount, currency = "USD", intent = "CAPTURE") {
    if (!this.isConfigured) {
      throw new Error("PayPal client not initialized");
    }
    try {
      if (!amount || isNaN(parseFloat(amount)) || parseFloat(amount) <= 0) {
        throw new Error("Invalid amount. Amount must be a positive number.");
      }
      const order = {
        id: randomUUID3(),
        intent,
        status: "CREATED",
        purchase_units: [
          {
            amount: {
              currency_code: currency,
              value: amount
            }
          }
        ],
        links: [
          {
            href: `https://sandbox.paypal.com/checkoutnow?token=${randomUUID3()}`,
            rel: "approve",
            method: "GET"
          }
        ],
        create_time: (/* @__PURE__ */ new Date()).toISOString()
      };
      console.log("PayPal order created:", order);
      return {
        statusCode: 201,
        data: order
      };
    } catch (error) {
      console.error("Failed to create PayPal order:", error);
      throw new Error("Failed to create order.");
    }
  }
  async captureOrder(orderID) {
    if (!this.isConfigured) {
      throw new Error("PayPal client not initialized");
    }
    try {
      const capture = {
        id: orderID,
        status: "COMPLETED",
        purchase_units: [
          {
            payments: {
              captures: [
                {
                  id: randomUUID3(),
                  status: "COMPLETED",
                  amount: {
                    currency_code: "USD",
                    value: "297.00"
                  }
                }
              ]
            }
          }
        ],
        update_time: (/* @__PURE__ */ new Date()).toISOString()
      };
      console.log("PayPal order captured:", capture);
      return {
        statusCode: 200,
        data: capture
      };
    } catch (error) {
      console.error("Failed to capture PayPal order:", error);
      throw new Error("Failed to capture order.");
    }
  }
  async createSubscription(planId, customerId) {
    if (!this.isConfigured) {
      throw new Error("PayPal client not initialized");
    }
    try {
      const subscription = {
        id: randomUUID3(),
        plan_id: planId,
        status: "ACTIVE",
        start_time: new Date(Date.now() + 6e4).toISOString(),
        subscriber: customerId ? {
          name: { given_name: "Premium", surname: "Member" },
          email_address: customerId
        } : void 0,
        create_time: (/* @__PURE__ */ new Date()).toISOString()
      };
      console.log("PayPal subscription created:", subscription);
      return {
        statusCode: 201,
        data: subscription
      };
    } catch (error) {
      console.error("Failed to create PayPal subscription:", error);
      throw new Error("Failed to create subscription.");
    }
  }
  isAvailable() {
    return this.isConfigured && !!process.env.PAYPAL_CLIENT_ID && !!process.env.PAYPAL_CLIENT_SECRET;
  }
};
var paypalService = new PayPalPaymentService();

// server/integrations/zoom.ts
import axios from "axios";
var ZOOM_API_BASE = "https://api.zoom.us/v2";
var ZOOM_OAUTH_BASE = "https://zoom.us/oauth";
var ZoomService = class {
  accessToken = null;
  tokenExpiry = 0;
  constructor() {
  }
  /**
   * Get or refresh access token for Zoom API
   */
  async getAccessToken() {
    if (this.accessToken && Date.now() < this.tokenExpiry) {
      return this.accessToken;
    }
    try {
      if (!process.env.ZOOM_API_KEY || !process.env.ZOOM_API_SECRET || !process.env.ZOOM_ACCOUNT_ID) {
        throw new Error("Missing required Zoom credentials: ZOOM_API_KEY, ZOOM_API_SECRET, or ZOOM_ACCOUNT_ID");
      }
      console.log("Zoom Auth Debug:", {
        hasApiKey: !!process.env.ZOOM_API_KEY,
        hasApiSecret: !!process.env.ZOOM_API_SECRET,
        hasAccountId: !!process.env.ZOOM_ACCOUNT_ID,
        accountIdLength: process.env.ZOOM_ACCOUNT_ID?.length,
        apiKeyLength: process.env.ZOOM_API_KEY?.length
      });
      const credentials = Buffer.from(
        `${process.env.ZOOM_API_KEY}:${process.env.ZOOM_API_SECRET}`
      ).toString("base64");
      const response = await axios.post(
        `${ZOOM_OAUTH_BASE}/token?grant_type=account_credentials&account_id=${process.env.ZOOM_ACCOUNT_ID}`,
        {},
        {
          headers: {
            "Authorization": `Basic ${credentials}`,
            "Content-Type": "application/x-www-form-urlencoded"
          }
        }
      );
      const tokenData = response.data;
      this.accessToken = tokenData.access_token;
      this.tokenExpiry = Date.now() + (tokenData.expires_in - 300) * 1e3;
      return this.accessToken;
    } catch (error) {
      console.error("Failed to get Zoom access token:", error);
      throw new Error("Failed to authenticate with Zoom API");
    }
  }
  /**
   * Create a scheduled training session
   */
  async createTrainingSession(data) {
    const token = await this.getAccessToken();
    const meetingData = {
      topic: data.topic,
      type: 2,
      // Scheduled meeting
      start_time: data.startTime,
      duration: data.duration,
      timezone: data.timezone || "America/New_York",
      agenda: data.agenda,
      password: data.password,
      settings: {
        host_video: true,
        participant_video: false,
        cn_meeting: false,
        in_meeting: false,
        join_before_host: true,
        jbh_time: 5,
        mute_upon_entry: true,
        watermark: false,
        use_pmi: false,
        approval_type: 0,
        // Automatically approve
        audio: "both",
        auto_recording: data.allowRecording ? "cloud" : "none",
        enforce_login: false,
        waiting_room: data.waitingRoom ?? true,
        allow_multiple_devices: true,
        request_permission_to_unmute_participants: true,
        meeting_authentication: false
      }
    };
    try {
      const response = await axios.post(
        `${ZOOM_API_BASE}/users/me/meetings`,
        meetingData,
        {
          headers: {
            "Authorization": `Bearer ${token}`,
            "Content-Type": "application/json"
          }
        }
      );
      return response.data;
    } catch (error) {
      console.error("Failed to create Zoom meeting:", error);
      throw new Error("Failed to create training session");
    }
  }
  /**
   * Create a webinar for larger training events
   */
  async createWebinar(data) {
    const token = await this.getAccessToken();
    const webinarData = {
      topic: data.topic,
      type: 5,
      // Webinar
      start_time: data.startTime,
      duration: data.duration,
      timezone: data.timezone || "America/New_York",
      agenda: data.agenda,
      password: data.password,
      settings: {
        host_video: true,
        panelists_video: true,
        practice_session: data.practiceSession ?? true,
        hd_video: true,
        approval_type: data.registrationRequired ? 1 : 0,
        registration_type: data.registrationRequired ? 1 : 3,
        audio: "both",
        auto_recording: "cloud",
        enforce_login: false,
        allow_multiple_devices: true,
        on_demand: false,
        contact_name: "TMH Global Training",
        contact_email: "training@tmhglobal.com",
        registrants_confirmation_email: true,
        notify_registrants: true,
        post_webinar_survey: false,
        registrants_email_notification: true,
        meeting_authentication: false
      }
    };
    try {
      const response = await axios.post(
        `${ZOOM_API_BASE}/users/me/webinars`,
        webinarData,
        {
          headers: {
            "Authorization": `Bearer ${token}`,
            "Content-Type": "application/json"
          }
        }
      );
      return response.data;
    } catch (error) {
      console.error("Failed to create Zoom webinar:", error);
      throw new Error("Failed to create webinar");
    }
  }
  /**
   * Get meeting details
   */
  async getMeeting(meetingId) {
    const token = await this.getAccessToken();
    try {
      const response = await axios.get(
        `${ZOOM_API_BASE}/meetings/${meetingId}`,
        {
          headers: {
            "Authorization": `Bearer ${token}`
          }
        }
      );
      return response.data;
    } catch (error) {
      console.error("Failed to get Zoom meeting:", error);
      throw new Error("Failed to retrieve meeting details");
    }
  }
  /**
   * Update meeting details
   */
  async updateMeeting(meetingId, updates) {
    const token = await this.getAccessToken();
    try {
      await axios.patch(
        `${ZOOM_API_BASE}/meetings/${meetingId}`,
        updates,
        {
          headers: {
            "Authorization": `Bearer ${token}`,
            "Content-Type": "application/json"
          }
        }
      );
    } catch (error) {
      console.error("Failed to update Zoom meeting:", error);
      throw new Error("Failed to update meeting");
    }
  }
  /**
   * Delete/cancel a meeting
   */
  async deleteMeeting(meetingId) {
    const token = await this.getAccessToken();
    try {
      await axios.delete(
        `${ZOOM_API_BASE}/meetings/${meetingId}`,
        {
          headers: {
            "Authorization": `Bearer ${token}`
          }
        }
      );
    } catch (error) {
      console.error("Failed to delete Zoom meeting:", error);
      throw new Error("Failed to cancel meeting");
    }
  }
  /**
   * Get list of user's meetings
   */
  async getMeetings(type = "scheduled") {
    const token = await this.getAccessToken();
    try {
      const response = await axios.get(
        `${ZOOM_API_BASE}/users/me/meetings?type=${type}`,
        {
          headers: {
            "Authorization": `Bearer ${token}`
          }
        }
      );
      return response.data.meetings || [];
    } catch (error) {
      console.error("Failed to get Zoom meetings:", error);
      throw new Error("Failed to retrieve meetings");
    }
  }
  /**
   * Create instant meeting using PMI (Personal Meeting ID)
   * Since Zoom no longer allows meeting creation for most apps, we use PMI as a workaround
   */
  async createInstantMeeting(topic, agenda, userPmi) {
    const pmiMeetingId = userPmi?.pmi || process.env.ZOOM_PMI || "1234567890";
    const password = userPmi?.password || process.env.ZOOM_PMI_PASSWORD || "";
    const joinUrl = password ? `https://zoom.us/j/${pmiMeetingId}?pwd=${password}` : `https://zoom.us/j/${pmiMeetingId}`;
    const pmiMeeting = {
      id: parseInt(pmiMeetingId),
      uuid: `pmi-${Date.now()}`,
      host_id: "host",
      topic,
      type: 4,
      // Personal Meeting Room
      status: "waiting",
      start_time: (/* @__PURE__ */ new Date()).toISOString(),
      duration: 60,
      timezone: "America/New_York",
      agenda: agenda || "Instant training session",
      created_at: (/* @__PURE__ */ new Date()).toISOString(),
      start_url: joinUrl,
      // Same as join for PMI
      join_url: joinUrl,
      password,
      h323_password: "",
      pstn_password: "",
      encrypted_password: "",
      settings: {
        host_video: true,
        participant_video: false,
        cn_meeting: false,
        in_meeting: false,
        join_before_host: true,
        jbh_time: 0,
        mute_upon_entry: true,
        watermark: false,
        use_pmi: true,
        approval_type: 0,
        audio: "both",
        auto_recording: "none",
        enforce_login: false,
        enforce_login_domains: "",
        alternative_hosts: "",
        close_registration: false,
        show_share_button: true,
        allow_multiple_devices: true,
        registrants_confirmation_email: true,
        waiting_room: true,
        request_permission_to_unmute_participants: false,
        global_dial_in_countries: [],
        global_dial_in_numbers: [],
        registrants_email_notification: true,
        meeting_authentication: false,
        encryption_type: "enhanced_encryption",
        approved_or_denied_countries_or_regions: {
          enable: false
        }
      }
    };
    console.log(`Using PMI-based meeting: ${joinUrl}`);
    return pmiMeeting;
  }
  /**
   * Add registrant to webinar
   */
  async addWebinarRegistrant(webinarId, registrant) {
    const token = await this.getAccessToken();
    try {
      const response = await axios.post(
        `${ZOOM_API_BASE}/webinars/${webinarId}/registrants`,
        registrant,
        {
          headers: {
            "Authorization": `Bearer ${token}`,
            "Content-Type": "application/json"
          }
        }
      );
      return response.data;
    } catch (error) {
      console.error("Failed to add webinar registrant:", error);
      throw new Error("Failed to register for webinar");
    }
  }
};
var zoomService = new ZoomService();

// server/compliance/scanner.ts
import { randomUUID as randomUUID4 } from "crypto";
var ACCESSIBILITY_RULES = {
  // Image accessibility
  "img-alt": {
    id: "img-alt",
    wcagLevel: "A",
    section508Rule: "1194.22(a)",
    severity: "critical",
    description: "Images must have alternative text",
    helpUrl: "https://www.w3.org/WAI/WCAG21/Understanding/non-text-content.html"
  },
  // Form accessibility
  "label-content-name-mismatch": {
    id: "label-content-name-mismatch",
    wcagLevel: "AA",
    section508Rule: "1194.22(n)",
    severity: "warning",
    description: "Form elements must have proper labels",
    helpUrl: "https://www.w3.org/WAI/WCAG21/Understanding/label-in-name.html"
  },
  // Color contrast
  "color-contrast": {
    id: "color-contrast",
    wcagLevel: "AA",
    section508Rule: "1194.22(c)",
    severity: "critical",
    description: "Text must have sufficient color contrast",
    helpUrl: "https://www.w3.org/WAI/WCAG21/Understanding/contrast-minimum.html"
  },
  // Keyboard accessibility
  "keyboard": {
    id: "keyboard",
    wcagLevel: "A",
    section508Rule: "1194.22(a)",
    severity: "critical",
    description: "All interactive elements must be keyboard accessible",
    helpUrl: "https://www.w3.org/WAI/WCAG21/Understanding/keyboard.html"
  },
  // Heading structure
  "heading-order": {
    id: "heading-order",
    wcagLevel: "AA",
    section508Rule: "1194.22(d)",
    severity: "warning",
    description: "Heading elements should be in logical order",
    helpUrl: "https://www.w3.org/WAI/WCAG21/Understanding/headings-and-labels.html"
  },
  // Focus management
  "focus-order-semantics": {
    id: "focus-order-semantics",
    wcagLevel: "A",
    section508Rule: "1194.22(a)",
    severity: "warning",
    description: "Focus order must be logical and meaningful",
    helpUrl: "https://www.w3.org/WAI/WCAG21/Understanding/focus-order.html"
  },
  // ARIA usage
  "aria-valid-attr-value": {
    id: "aria-valid-attr-value",
    wcagLevel: "A",
    section508Rule: "1194.22(d)",
    severity: "critical",
    description: "ARIA attributes must have valid values",
    helpUrl: "https://www.w3.org/WAI/WCAG21/Understanding/name-role-value.html"
  }
};
var ComplianceScanner = class {
  constructor() {
  }
  /**
   * Perform accessibility scan of a URL
   */
  async scanUrl(url, scanType = "manual") {
    const scanId = randomUUID4();
    const startTime = Date.now();
    try {
      const scanData = {
        id: scanId,
        url,
        scanType,
        status: "running",
        userAgent: "TMH-Compliance-Scanner/1.0"
      };
      await storage.createComplianceScan(scanData);
      const result = await this.performScan(url);
      const scanDuration = Date.now() - startTime;
      await storage.updateComplianceScan(scanId, {
        status: "completed",
        overallScore: result.overallScore,
        totalIssues: result.totalIssues,
        criticalIssues: result.criticalIssues,
        warningIssues: result.warningIssues,
        infoIssues: result.infoIssues,
        scanDuration,
        completedAt: /* @__PURE__ */ new Date()
      });
      for (const issue of result.issues) {
        await storage.createComplianceIssue({
          scanId,
          ruleId: issue.ruleId,
          severity: issue.severity,
          wcagLevel: issue.wcagLevel,
          section508Rule: issue.section508Rule,
          element: issue.element,
          message: issue.message,
          helpUrl: issue.helpUrl,
          xpath: issue.xpath,
          html: issue.html
        });
      }
      await this.generateAlerts(scanId, result);
      return scanId;
    } catch (error) {
      await storage.updateComplianceScan(scanId, {
        status: "failed",
        scanDuration: Date.now() - startTime,
        completedAt: /* @__PURE__ */ new Date()
      });
      throw error;
    }
  }
  /**
   * Perform the actual accessibility scan
   * In production, this would use tools like axe-core, Pa11y, or Lighthouse
   */
  async performScan(url) {
    const startTime = Date.now();
    await new Promise((resolve) => setTimeout(resolve, 2e3));
    const issues = this.generateMockIssues(url);
    const criticalIssues = issues.filter((i) => i.severity === "critical").length;
    const warningIssues = issues.filter((i) => i.severity === "warning").length;
    const infoIssues = issues.filter((i) => i.severity === "info").length;
    const totalIssues = issues.length;
    const weightedScore = Math.max(0, 100 - criticalIssues * 10 - warningIssues * 5 - infoIssues * 2);
    const overallScore = Math.round(weightedScore);
    return {
      url,
      overallScore,
      issues,
      scanDuration: Date.now() - startTime,
      totalIssues,
      criticalIssues,
      warningIssues,
      infoIssues
    };
  }
  /**
   * Generate realistic mock accessibility issues for demonstration
   * In production, this would be replaced by actual accessibility testing
   */
  generateMockIssues(url) {
    const issues = [];
    if (url.includes("premium") || url.includes("training")) {
      issues.push({
        ruleId: "label-content-name-mismatch",
        severity: "warning",
        wcagLevel: "AA",
        section508Rule: "1194.22(n)",
        element: 'input[type="email"]',
        message: "Form input lacks properly associated label",
        helpUrl: ACCESSIBILITY_RULES["label-content-name-mismatch"].helpUrl,
        xpath: "/html/body/main/form/div[2]/input",
        html: '<input type="email" placeholder="Enter your email" />'
      });
    }
    if (url.includes("community")) {
      issues.push({
        ruleId: "color-contrast",
        severity: "critical",
        wcagLevel: "AA",
        section508Rule: "1194.22(c)",
        element: ".text-gray-500",
        message: "Text has insufficient color contrast ratio (3.2:1, requires 4.5:1)",
        helpUrl: ACCESSIBILITY_RULES["color-contrast"].helpUrl,
        xpath: "/html/body/main/div/p[1]",
        html: '<p class="text-gray-500">Join our exclusive community</p>'
      });
    }
    if (Math.random() > 0.3) {
      issues.push({
        ruleId: "img-alt",
        severity: "critical",
        wcagLevel: "A",
        section508Rule: "1194.22(a)",
        element: "img",
        message: "Image missing alternative text",
        helpUrl: ACCESSIBILITY_RULES["img-alt"].helpUrl,
        xpath: "/html/body/header/img[1]",
        html: '<img src="logo.jpg" />'
      });
    }
    if (Math.random() > 0.5) {
      issues.push({
        ruleId: "heading-order",
        severity: "warning",
        wcagLevel: "AA",
        section508Rule: "1194.22(d)",
        element: "h3",
        message: "Heading levels should not skip (h1 \u2192 h3)",
        helpUrl: ACCESSIBILITY_RULES["heading-order"].helpUrl,
        xpath: "/html/body/main/section[1]/h3",
        html: "<h3>Services Overview</h3>"
      });
    }
    if (Math.random() > 0.4) {
      issues.push({
        ruleId: "keyboard",
        severity: "critical",
        wcagLevel: "A",
        section508Rule: "1194.22(a)",
        element: "div[onclick]",
        message: "Interactive element not keyboard accessible",
        helpUrl: ACCESSIBILITY_RULES["keyboard"].helpUrl,
        xpath: "/html/body/main/div[3]/div",
        html: '<div onclick="handleClick()">Click me</div>'
      });
    }
    if (Math.random() > 0.6) {
      issues.push({
        ruleId: "focus-order-semantics",
        severity: "info",
        wcagLevel: "A",
        section508Rule: "1194.22(a)",
        element: "button",
        message: "Consider improving focus indicator visibility",
        helpUrl: ACCESSIBILITY_RULES["focus-order-semantics"].helpUrl,
        xpath: "/html/body/main/form/button",
        html: '<button type="submit">Submit</button>'
      });
    }
    return issues;
  }
  /**
   * Generate compliance alerts based on scan results
   */
  async generateAlerts(scanId, result) {
    const { criticalIssues, overallScore, url } = result;
    if (criticalIssues > 0) {
      await storage.createComplianceAlert({
        alertType: "critical_issue",
        severity: "high",
        title: `${criticalIssues} Critical Accessibility Issues Found`,
        message: `${url} has ${criticalIssues} critical accessibility violations that must be addressed immediately for Section 508 compliance.`,
        scanId,
        url
      });
    }
    if (overallScore < 70) {
      await storage.createComplianceAlert({
        alertType: "compliance_drop",
        severity: overallScore < 50 ? "high" : "medium",
        title: `Low Compliance Score: ${overallScore}%`,
        message: `${url} has a compliance score of ${overallScore}%, indicating significant accessibility barriers.`,
        scanId,
        url
      });
    }
    if (result.totalIssues > 5) {
      await storage.createComplianceAlert({
        alertType: "new_violations",
        severity: "medium",
        title: `${result.totalIssues} Accessibility Violations Detected`,
        message: `${url} has multiple accessibility issues that may prevent Section 508 compliance certification.`,
        scanId,
        url
      });
    }
  }
  /**
   * Get compliance status for multiple URLs
   */
  async getComplianceStatus(urls) {
    const results = [];
    for (const url of urls) {
      const latestScan = await storage.getLatestComplianceScan(url);
      results.push({
        url,
        score: latestScan?.overallScore || 0,
        lastScan: latestScan?.completedAt || /* @__PURE__ */ new Date(0),
        criticalIssues: latestScan?.criticalIssues || 0
      });
    }
    return results;
  }
  /**
   * Schedule automated scans
   */
  async scheduleAutomatedScan(url, interval) {
    console.log(`Scheduled ${interval} accessibility scan for ${url}`);
    setTimeout(() => {
      this.scanUrl(url, "scheduled").catch(console.error);
    }, 5e3);
  }
};
var complianceScanner = new ComplianceScanner();

// server/integrations/google-calendar.ts
import { google } from "googleapis";
var GoogleCalendarService = class {
  oauth2Client;
  calendar;
  constructor() {
    if (!process.env.GOOGLE_CLIENT_ID || !process.env.GOOGLE_CLIENT_SECRET) {
      throw new Error("Google Calendar credentials not configured");
    }
    const redirectUri = process.env.REPLIT_DOMAINS ? `https://${process.env.REPLIT_DOMAINS.split(",")[0]}/api/auth/google/callback` : "http://localhost:5000/api/auth/google/callback";
    this.oauth2Client = new google.auth.OAuth2(
      process.env.GOOGLE_CLIENT_ID,
      process.env.GOOGLE_CLIENT_SECRET,
      redirectUri
    );
    console.log("\u{1F527} Google OAuth configured with redirect URI:", redirectUri);
    this.calendar = google.calendar({ version: "v3", auth: this.oauth2Client });
  }
  // Generate OAuth URL for initial setup
  getAuthUrl() {
    const scopes = [
      "https://www.googleapis.com/auth/calendar",
      "https://www.googleapis.com/auth/calendar.events",
      "https://www.googleapis.com/auth/gmail.send"
      // Re-added for email notifications
    ];
    console.log("\u{1F517} Generating Google auth URL with scopes:", scopes);
    return this.oauth2Client.generateAuthUrl({
      access_type: "offline",
      scope: scopes,
      prompt: "consent"
    });
  }
  // Set credentials (called after OAuth flow)
  setCredentials(tokens) {
    this.oauth2Client.setCredentials(tokens);
  }
  // Exchange authorization code for tokens
  async getTokens(code) {
    try {
      const { tokens } = await this.oauth2Client.getToken(code);
      this.setCredentials(tokens);
      return tokens;
    } catch (error) {
      console.error("Error getting tokens:", error);
      throw new Error("Failed to exchange authorization code for tokens");
    }
  }
  // Check availability for a specific time slot
  async checkAvailability(startTime, endTime, calendarId = "primary") {
    try {
      const response = await this.calendar.freebusy.query({
        requestBody: {
          timeMin: startTime,
          timeMax: endTime,
          items: [{ id: calendarId }]
        }
      });
      const busyTimes = response.data.calendars[calendarId]?.busy || [];
      return busyTimes.length === 0;
    } catch (error) {
      console.error("Error checking availability:", error);
      throw new Error("Failed to check calendar availability");
    }
  }
  // Get existing events for a date range
  async getEvents(startDate, endDate, calendarId = "primary") {
    try {
      const response = await this.calendar.events.list({
        calendarId,
        timeMin: startDate,
        timeMax: endDate,
        singleEvents: true,
        orderBy: "startTime",
        maxResults: 250
      });
      return response.data.items || [];
    } catch (error) {
      console.error("Error fetching events:", error);
      throw new Error("Failed to fetch calendar events");
    }
  }
  // Create new calendar event
  async createEvent(eventData, calendarId = "primary") {
    try {
      const event = {
        summary: eventData.summary,
        description: eventData.description,
        start: {
          dateTime: eventData.startDateTime,
          timeZone: eventData.timeZone || "America/New_York"
        },
        end: {
          dateTime: eventData.endDateTime,
          timeZone: eventData.timeZone || "America/New_York"
        },
        attendees: eventData.attendeeEmail ? [{ email: eventData.attendeeEmail }] : [],
        reminders: {
          useDefault: false,
          overrides: [
            { method: "email", minutes: 1440 },
            // 24 hours
            { method: "popup", minutes: 30 }
          ]
        }
      };
      const response = await this.calendar.events.insert({
        calendarId,
        resource: event,
        sendUpdates: "all"
      });
      return response.data;
    } catch (error) {
      console.error("Error creating event:", error);
      throw new Error("Failed to create calendar event");
    }
  }
  // Update existing event
  async updateEvent(eventId, eventData, calendarId = "primary") {
    try {
      const updateData = {};
      if (eventData.summary) updateData.summary = eventData.summary;
      if (eventData.description) updateData.description = eventData.description;
      if (eventData.startDateTime) {
        updateData.start = {
          dateTime: eventData.startDateTime,
          timeZone: eventData.timeZone || "America/New_York"
        };
      }
      if (eventData.endDateTime) {
        updateData.end = {
          dateTime: eventData.endDateTime,
          timeZone: eventData.timeZone || "America/New_York"
        };
      }
      if (eventData.attendeeEmail) {
        updateData.attendees = [{ email: eventData.attendeeEmail }];
      }
      const response = await this.calendar.events.update({
        calendarId,
        eventId,
        resource: updateData,
        sendUpdates: "all"
      });
      return response.data;
    } catch (error) {
      console.error("Error updating event:", error);
      throw new Error("Failed to update calendar event");
    }
  }
  // Delete event
  async deleteEvent(eventId, calendarId = "primary") {
    try {
      await this.calendar.events.delete({
        calendarId,
        eventId,
        sendUpdates: "all"
      });
    } catch (error) {
      console.error("Error deleting event:", error);
      throw new Error("Failed to delete calendar event");
    }
  }
  // Get user's calendar list
  async getCalendars() {
    try {
      const response = await this.calendar.calendarList.list();
      return response.data.items || [];
    } catch (error) {
      console.error("Error fetching calendars:", error);
      throw new Error("Failed to fetch calendar list");
    }
  }
};
var googleCalendarService = new GoogleCalendarService();

// server/integrations/gmail.ts
import { google as google2 } from "googleapis";
var GmailService = class {
  oauth2Client;
  gmail;
  constructor() {
    if (!process.env.GOOGLE_CLIENT_ID || !process.env.GOOGLE_CLIENT_SECRET) {
      throw new Error("Gmail credentials not configured");
    }
    this.oauth2Client = new google2.auth.OAuth2(
      process.env.GOOGLE_CLIENT_ID,
      process.env.GOOGLE_CLIENT_SECRET,
      process.env.REPLIT_DOMAINS ? `https://${process.env.REPLIT_DOMAINS.split(",")[0]}/api/auth/google/callback` : "http://localhost:5000/api/auth/google/callback"
    );
    this.gmail = google2.gmail({ version: "v1", auth: this.oauth2Client });
  }
  // Set credentials (use same tokens as calendar)
  setCredentials(tokens) {
    this.oauth2Client.setCredentials(tokens);
  }
  // Create raw email message
  createRawMessage(message) {
    const messageParts = [
      `To: ${message.to}`,
      `Subject: ${message.subject}`,
      "Content-Type: text/html; charset=utf-8",
      "",
      message.html || message.text
    ];
    return Buffer.from(messageParts.join("\r\n")).toString("base64").replace(/\+/g, "-").replace(/\//g, "_").replace(/=+$/, "");
  }
  // Send email notification
  async sendEmail(message) {
    try {
      const rawMessage = this.createRawMessage(message);
      await this.gmail.users.messages.send({
        userId: "me",
        requestBody: {
          raw: rawMessage
        }
      });
      console.log(`Email sent successfully to ${message.to}`);
      return true;
    } catch (error) {
      console.error("Gmail send error:", error);
      return false;
    }
  }
  // Send new contact notification to admin
  async sendContactNotification(contact) {
    const message = {
      to: "tmhglobal@gmail.com",
      // Your business email
      subject: `New Contact Form Submission - ${contact.firstName} ${contact.lastName}`,
      text: `New Contact: ${contact.firstName} ${contact.lastName} (${contact.email}) - ${contact.serviceInterest}`,
      html: `
        <h2>New Contact Form Submission</h2>
        <div style="font-family: Arial, sans-serif; max-width: 600px;">
          <h3>Contact Information:</h3>
          <ul>
            <li><strong>Name:</strong> ${contact.firstName} ${contact.lastName}</li>
            <li><strong>Email:</strong> ${contact.email}</li>
            <li><strong>Phone:</strong> ${contact.phone || "Not provided"}</li>
            <li><strong>Service Interest:</strong> ${contact.serviceInterest}</li>
            <li><strong>Urgent:</strong> ${contact.isUrgent ? "YES" : "No"}</li>
            <li><strong>SMS Opt-in:</strong> ${contact.smsOptIn ? "YES" : "No"}</li>
          </ul>
          
          <h3>Message:</h3>
          <p style="background: #f5f5f5; padding: 15px; border-radius: 5px;">
            ${contact.message}
          </p>
          
          <p style="margin-top: 20px;">
            <strong>Submitted:</strong> ${new Date(contact.createdAt).toLocaleString()}
          </p>
          
          ${contact.smsOptIn ? '<p style="color: #007cba;"><strong>Note:</strong> This client opted in for SMS updates - add them to your SMS campaigns!</p>' : ""}
        </div>
      `
    };
    return await this.sendEmail(message);
  }
  // Send consultation notification email
  async sendConsultationNotification(consultation) {
    const message = {
      to: "tmhglobal@gmail.com",
      // Your business email
      subject: `New Consultation Scheduled - ${consultation.firstName} ${consultation.lastName}`,
      text: `New Consultation: ${consultation.firstName} ${consultation.lastName} (${consultation.email}) - ${consultation.serviceType} scheduled for ${new Date(consultation.preferredDate).toLocaleDateString()}`,
      html: `
        <h2>New Consultation Scheduled</h2>
        <div style="font-family: Arial, sans-serif; max-width: 600px;">
          <h3>Client Information:</h3>
          <ul>
            <li><strong>Name:</strong> ${consultation.firstName} ${consultation.lastName}</li>
            <li><strong>Email:</strong> ${consultation.email}</li>
            <li><strong>Phone:</strong> ${consultation.phone || "Not provided"}</li>
            <li><strong>Service Type:</strong> ${consultation.serviceType}</li>
            <li><strong>Preferred Date:</strong> ${new Date(consultation.preferredDate).toLocaleString()}</li>
            <li><strong>SMS Opt-in:</strong> ${consultation.smsOptIn ? "YES" : "No"}</li>
          </ul>
          
          <h3>Message:</h3>
          <p style="background: #f5f5f5; padding: 15px; border-radius: 5px;">
            ${consultation.message}
          </p>
          
          <p style="margin-top: 20px;">
            <strong>Scheduled:</strong> ${new Date(consultation.createdAt).toLocaleString()}
          </p>
          
          ${consultation.calendarEventId ? '<p style="color: #28a745;"><strong>\u2705 Google Calendar Event Created</strong></p>' : '<p style="color: #dc3545;">\u274C Google Calendar Event NOT Created</p>'}
          ${consultation.smsOptIn ? '<p style="color: #007cba;"><strong>Note:</strong> This client opted in for SMS updates - add them to your SMS campaigns!</p>' : ""}
        </div>
      `
    };
    return await this.sendEmail(message);
  }
};
var gmailService = new GmailService();

// server/routes.ts
import multer from "multer";
import { randomUUID as randomUUID5 } from "crypto";

// server/paypal.ts
import {
  Client,
  Environment,
  LogLevel,
  OAuthAuthorizationController,
  OrdersController
} from "@paypal/paypal-server-sdk";
var { PAYPAL_CLIENT_ID, PAYPAL_CLIENT_SECRET } = process.env;
if (!PAYPAL_CLIENT_ID) {
  throw new Error("Missing PAYPAL_CLIENT_ID");
}
if (!PAYPAL_CLIENT_SECRET) {
  throw new Error("Missing PAYPAL_CLIENT_SECRET");
}
var client = new Client({
  clientCredentialsAuthCredentials: {
    oAuthClientId: PAYPAL_CLIENT_ID,
    oAuthClientSecret: PAYPAL_CLIENT_SECRET
  },
  timeout: 0,
  environment: process.env.NODE_ENV === "production" ? Environment.Production : Environment.Sandbox,
  logging: {
    logLevel: LogLevel.Info,
    logRequest: {
      logBody: true
    },
    logResponse: {
      logHeaders: true
    }
  }
});
var ordersController = new OrdersController(client);
var oAuthAuthorizationController = new OAuthAuthorizationController(client);
async function getClientToken() {
  const auth = Buffer.from(
    `${PAYPAL_CLIENT_ID}:${PAYPAL_CLIENT_SECRET}`
  ).toString("base64");
  const { result } = await oAuthAuthorizationController.requestToken(
    {
      authorization: `Basic ${auth}`
    },
    { intent: "sdk_init", response_type: "client_token" }
  );
  return result.accessToken;
}
async function createPaypalOrder(req, res) {
  try {
    const { amount, currency, intent } = req.body;
    if (!amount || isNaN(parseFloat(amount)) || parseFloat(amount) <= 0) {
      return res.status(400).json({
        error: "Invalid amount. Amount must be a positive number."
      });
    }
    if (!currency) {
      return res.status(400).json({ error: "Invalid currency. Currency is required." });
    }
    if (!intent) {
      return res.status(400).json({ error: "Invalid intent. Intent is required." });
    }
    const collect = {
      body: {
        intent,
        purchaseUnits: [
          {
            amount: {
              currencyCode: currency,
              value: amount
            }
          }
        ]
      },
      prefer: "return=minimal"
    };
    const { body, ...httpResponse } = await ordersController.createOrder(collect);
    const jsonResponse = JSON.parse(String(body));
    const httpStatusCode = httpResponse.statusCode;
    res.status(httpStatusCode).json(jsonResponse);
  } catch (error) {
    console.error("Failed to create order:", error);
    res.status(500).json({ error: "Failed to create order." });
  }
}
async function capturePaypalOrder(req, res) {
  try {
    const { orderID } = req.params;
    const collect = {
      id: orderID,
      prefer: "return=minimal"
    };
    const { body, ...httpResponse } = await ordersController.captureOrder(collect);
    const jsonResponse = JSON.parse(String(body));
    const httpStatusCode = httpResponse.statusCode;
    res.status(httpStatusCode).json(jsonResponse);
  } catch (error) {
    console.error("Failed to create order:", error);
    res.status(500).json({ error: "Failed to capture order." });
  }
}
async function loadPaypalDefault(req, res) {
  const clientToken = await getClientToken();
  res.json({
    clientToken
  });
}

// server/routes.ts
var upload = multer({
  storage: multer.diskStorage({
    destination: (req, file, cb) => {
      cb(null, "uploads/");
    },
    filename: (req, file, cb) => {
      const uniqueName = `${randomUUID5()}-${file.originalname}`;
      cb(null, uniqueName);
    }
  }),
  limits: {
    fileSize: 10 * 1024 * 1024
    // 10MB limit
  },
  fileFilter: (req, file, cb) => {
    const allowedTypes = [
      "application/pdf",
      "application/msword",
      "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      "application/vnd.ms-excel",
      "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
      "image/jpeg",
      "image/png",
      "image/gif",
      "text/plain"
    ];
    if (allowedTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error("Invalid file type. Please upload documents, images, or text files only."));
    }
  }
});
var trackApiRequest = async (req, res, next) => {
  await storage.incrementRequestCount();
  next();
};
async function registerRoutes(app2) {
  app2.use("/api", trackApiRequest);
  app2.post("/api/contact", async (req, res) => {
    try {
      const validatedData = insertContactSchema.parse(req.body);
      const contact = await storage.createContact(validatedData);
      if (req.session?.googleTokens) {
        try {
          gmailService.setCredentials(req.session.googleTokens);
          await gmailService.sendContactNotification(contact);
          console.log("\u2705 Email notification sent for new contact:", contact.firstName, contact.lastName);
        } catch (error) {
          console.error("\u274C Failed to send email notification:", error);
        }
      } else {
        console.log("\u2139\uFE0F No Google tokens available - email notification skipped");
      }
      if (contact.smsOptIn && ringCentralService.isConfigured()) {
        if (req.session?.ringCentralTokens) {
          ringCentralService.setTokens(req.session.ringCentralTokens);
        }
        if (ringCentralService.hasValidTokens()) {
          try {
            if (contact.phone) {
              const formattedPhone = ringCentralService.formatPhoneNumber(contact.phone);
              await ringCentralService.sendSMS({
                to: [formattedPhone],
                text: `TMH Global LLC: Thank you ${contact.firstName}! We received your inquiry about ${contact.serviceInterest}. We'll respond within 24 hours. Support: support@tmhglobal.com or 703-829-7277. Reply STOP to opt out.`
              });
              console.log(`\u2705 SMS sent to customer: ${formattedPhone}`);
            }
            await ringCentralService.sendSMS({
              to: ["+17038297277"],
              // Your business phone number
              text: `\u{1F6A8} NEW LEAD: ${contact.firstName} ${contact.lastName} interested in ${contact.serviceInterest}. Email: ${contact.email}${contact.phone ? `, Phone: ${contact.phone}` : ""}. Add to SMS campaigns!`
            });
            console.log("\u2705 SMS alert sent for new contact with SMS opt-in");
          } catch (error) {
            console.error("\u274C Failed to send SMS alert:", error);
          }
        } else {
          console.log("\u2139\uFE0F RingCentral OAuth tokens not available - SMS notification skipped. Please authorize RingCentral SMS.");
        }
      }
      console.log("New contact submission:", contact);
      res.json({ success: true, message: "Contact form submitted successfully", id: contact.id });
    } catch (error) {
      res.status(400).json({
        success: false,
        message: "Invalid form data",
        errors: error.errors || error.message
      });
    }
  });
  app2.get("/api/contacts", async (req, res) => {
    try {
      const contacts2 = await storage.getAllContacts();
      res.json(contacts2);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve contacts" });
    }
  });
  app2.get("/api/consultations", async (req, res) => {
    try {
      const consultations2 = await storage.getAllConsultations();
      res.json(consultations2);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve consultations" });
    }
  });
  app2.get("/api/auth/google", (req, res) => {
    try {
      const authUrl = googleCalendarService.getAuthUrl();
      res.json({ authUrl });
    } catch (error) {
      res.status(500).json({ message: "Failed to generate auth URL" });
    }
  });
  app2.get("/api/auth/google/callback", async (req, res) => {
    try {
      const { code, error } = req.query;
      if (error) {
        return res.send(`
          <html>
            <script>
              window.close();
            </script>
          </html>
        `);
      }
      if (!code) {
        return res.status(400).send("Authorization code required");
      }
      const tokens = await googleCalendarService.getTokens(code);
      if (!req.session) {
        req.session = {};
      }
      req.session.googleTokens = tokens;
      try {
        const adminUser = await storage.getUserByUsername("tmh_admin");
        if (adminUser) {
          await storage.upsertOAuthTokens(adminUser.id, "google", {
            accessToken: tokens.access_token,
            refreshToken: tokens.refresh_token || void 0,
            expiresAt: tokens.expiry_date ? new Date(tokens.expiry_date) : void 0,
            scope: tokens.scope,
            accountId: null,
            // Could be email if needed
            metadata: null
          });
          console.log("\u2705 Google tokens saved to database for persistence");
        }
      } catch (error2) {
        console.error("\u274C Failed to save Google tokens to database:", error2);
      }
      res.send(`
        <html>
          <script>
            // Notify parent window and close popup
            if (window.opener) {
              window.opener.postMessage({ type: 'GOOGLE_AUTH_SUCCESS' }, '*');
            }
            window.close();
          </script>
        </html>
      `);
    } catch (error) {
      console.error("Google auth callback error:", error);
      res.send(`
        <html>
          <script>
            // Notify parent window of error and close popup
            if (window.opener) {
              window.opener.postMessage({ type: 'GOOGLE_AUTH_ERROR', error: '${error.message}' }, '*');
            }
            window.close();
          </script>
        </html>
      `);
    }
  });
  app2.post("/api/auth/google/callback", async (req, res) => {
    try {
      const { code } = req.body;
      if (!code) {
        return res.status(400).json({ message: "Authorization code required" });
      }
      const tokens = await googleCalendarService.getTokens(code);
      if (!req.session) {
        req.session = {};
      }
      req.session.googleTokens = tokens;
      res.json({
        success: true,
        message: "Google Calendar connected successfully",
        hasRefreshToken: !!tokens.refresh_token
      });
    } catch (error) {
      console.error("Google auth callback error:", error);
      res.status(500).json({
        message: "Failed to authenticate with Google Calendar",
        error: error.message
      });
    }
  });
  app2.post("/api/sms/configure", async (req, res) => {
    const authHeader = req.headers.authorization;
    if (!authHeader || authHeader !== "Bearer ADMIN_SECRET_KEY") {
      return res.status(401).json({ message: "Unauthorized - Admin access required" });
    }
    try {
      const { clientId, clientSecret } = req.body;
      console.log(`\u{1F4E5} Received RingCentral config: ClientID=${clientId?.substring(0, 10)}..., Secret=${clientSecret ? "[PROVIDED]" : "[MISSING]"}`);
      if (!clientId || !clientSecret) {
        return res.status(400).json({ message: "Client ID and Client Secret are required" });
      }
      process.env.RINGCENTRAL_CLIENT_ID = clientId;
      process.env.RINGCENTRAL_CLIENT_SECRET = clientSecret;
      ringCentralService.updateConfig(clientId, clientSecret);
      const testUrl = ringCentralService.getAuthorizationUrl();
      console.log(`\u{1F527} New OAuth URL: ${testUrl.substring(0, 100)}...`);
      console.log("\u{1F527} RingCentral credentials updated successfully");
      res.json({ message: "RingCentral configuration updated successfully" });
    } catch (error) {
      console.error("RingCentral configuration error:", error);
      res.status(500).json({ message: "Failed to update RingCentral configuration" });
    }
  });
  app2.get("/api/sms/debug", (req, res) => {
    try {
      const config = ringCentralService.getConfig();
      res.json({
        clientId: config.clientId ? `${config.clientId.substring(0, 10)}...` : "NOT SET",
        hasSecret: !!config.clientSecret,
        redirectUri: config.redirectUri
      });
    } catch (error) {
      res.status(500).json({ message: "Failed to get RingCentral config" });
    }
  });
  app2.get("/api/auth/ringcentral", (req, res) => {
    try {
      const authUrl = ringCentralService.getAuthorizationUrl();
      res.json({ authUrl });
    } catch (error) {
      res.status(500).json({ message: "Failed to generate RingCentral auth URL" });
    }
  });
  app2.get("/api/auth/ringcentral/callback", async (req, res) => {
    try {
      res.set("Cache-Control", "no-store");
      console.log("\u{1F527} RingCentral callback received with query:", req.query);
      const { code, error } = req.query;
      if (error) {
        return res.send(`
          <html>
            <script>
              if (window.opener) {
                window.opener.postMessage({ type: 'RINGCENTRAL_AUTH_ERROR', error: '${error}' }, '*');
              }
              window.close();
            </script>
          </html>
        `);
      }
      if (!code) {
        return res.status(400).send("Authorization code required");
      }
      console.log("\u{1F527} RingCentral: Starting token exchange with code:", code);
      const tokens = await ringCentralService.exchangeCodeForToken(code);
      console.log("\u{1F527} RingCentral: Token exchange successful:", !!tokens.accessToken);
      if (!req.session) {
        req.session = {};
      }
      req.session.ringCentralTokens = tokens;
      console.log("\u{1F527} RingCentral: Tokens stored in session");
      try {
        const adminUser = await storage.getUserByUsername("tmh_admin");
        if (adminUser) {
          await storage.upsertOAuthTokens(adminUser.id, "ringcentral", {
            accessToken: tokens.accessToken,
            refreshToken: tokens.refreshToken || void 0,
            expiresAt: tokens.expiresAt,
            scope: null,
            // RingCentral doesn't provide scope in response
            accountId: null,
            // Could store extension ID if available
            metadata: null
          });
          console.log("\u2705 RingCentral tokens saved to database for persistence");
        }
      } catch (error2) {
        console.error("\u274C Failed to save RingCentral tokens to database:", error2);
      }
      ringCentralService.setTokens(tokens);
      console.log("\u{1F527} RingCentral: Tokens set in service, hasValidTokens:", ringCentralService.hasValidTokens());
      res.send(`
        <html>
          <body>
            <p style="font-family: Arial, sans-serif; text-align: center; margin-top: 100px;">
              \u2705 RingCentral SMS connected successfully! <br><br>
              <a href="javascript:void(0)" onclick="window.close()" style="color: #0066cc;">Close this window</a>
            </p>
          </body>
          <script>
            try {
              if (window.opener) {
                window.opener.postMessage({ type: 'RINGCENTRAL_AUTH_SUCCESS' }, '*');
              }
            } catch (e) {}
            
            // Multiple close attempts for browser compatibility
            window.close();
            setTimeout(function() {
              try {
                window.open('', '_self');
                window.close();
              } catch (e) {}
            }, 200);
          </script>
        </html>
      `);
    } catch (error) {
      console.error("RingCentral auth callback error:", error);
      res.send(`
        <html>
          <script>
            // Notify parent window of error and close popup
            if (window.opener) {
              window.opener.postMessage({ type: 'RINGCENTRAL_AUTH_ERROR', error: '${error.message}' }, '*');
            }
            window.close();
          </script>
        </html>
      `);
    }
  });
  app2.get("/api/calendar/status", (req, res) => {
    const isConnected = !!req.session?.googleTokens;
    res.json({
      connected: isConnected,
      email: isConnected ? "tmhglobal@gmail.com" : null
    });
  });
  app2.post("/api/calendar/disconnect", (req, res) => {
    if (req.session?.googleTokens) {
      delete req.session.googleTokens;
    }
    res.json({ success: true, message: "Google Calendar disconnected" });
  });
  app2.get("/api/sms/status", (req, res) => {
    if (req.session?.ringCentralTokens && !ringCentralService.hasValidTokens()) {
      ringCentralService.setTokens(req.session.ringCentralTokens);
    }
    const isConfigured = ringCentralService.isConfigured();
    const hasTokens = !!req.session?.ringCentralTokens;
    const hasValidTokens = hasTokens && ringCentralService.hasValidTokens();
    res.json({
      configured: isConfigured,
      connected: hasValidTokens,
      service: "RingCentral",
      features: isConfigured ? ["SMS Notifications", "Contact Form Alerts"] : [],
      authRequired: isConfigured && !hasValidTokens
    });
  });
  app2.post("/api/sms/disconnect", (req, res) => {
    if (req.session?.ringCentralTokens) {
      delete req.session.ringCentralTokens;
    }
    res.json({ success: true, message: "RingCentral SMS disconnected" });
  });
  app2.get("/api/sms/debug/senders", async (req, res) => {
    try {
      if (req.session?.ringCentralTokens && !ringCentralService.hasValidTokens()) {
        ringCentralService.setTokens(req.session.ringCentralTokens);
      }
      if (!ringCentralService.hasValidTokens()) {
        return res.status(401).json({ message: "RingCentral not authenticated" });
      }
      res.json({ message: "SMS senders discovery triggered. Check logs for results." });
    } catch (error) {
      res.status(500).json({ message: "Failed to discover SMS senders" });
    }
  });
  app2.get("/paypal/setup", async (req, res) => {
    await loadPaypalDefault(req, res);
  });
  app2.post("/paypal/order", async (req, res) => {
    await createPaypalOrder(req, res);
  });
  app2.post("/paypal/order/:orderID/capture", async (req, res) => {
    await capturePaypalOrder(req, res);
  });
  app2.post("/api/square/create-payment", async (req, res) => {
    try {
      const { amount, planId, customer } = req.body;
      if (!amount || !planId || !customer) {
        return res.status(400).json({
          success: false,
          message: "Missing required fields: amount, planId, and customer information"
        });
      }
      const squareCustomer = await squareService.createCustomer(
        customer.email,
        customer.firstName,
        customer.lastName
      );
      const subscriptionData = {
        userId: squareCustomer.id,
        paymentProvider: "square",
        subscriptionId: randomUUID5(),
        status: "active",
        planType: planId,
        amount: parseInt(amount),
        currency: "USD",
        startDate: /* @__PURE__ */ new Date(),
        expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1e3),
        // 30 days
        customerInfo: {
          firstName: customer.firstName,
          lastName: customer.lastName,
          email: customer.email,
          company: customer.company
        }
      };
      const subscription = await storage.createSubscription(subscriptionData);
      res.json({
        success: true,
        subscription,
        customer: squareCustomer,
        message: "Payment processed successfully. Welcome to TMH Global!"
      });
    } catch (error) {
      console.error("Square payment creation error:", error);
      res.status(500).json({
        success: false,
        message: "Failed to process payment. Please try again."
      });
    }
  });
  app2.get("/api/payments/status", (req, res) => {
    const squareConfigured = !!(process.env.SQUARE_ACCESS_TOKEN && process.env.SQUARE_APPLICATION_ID);
    const paypalConfigured = !!(process.env.PAYPAL_CLIENT_ID && process.env.PAYPAL_CLIENT_SECRET);
    res.json({
      square: {
        configured: squareConfigured,
        features: squareConfigured ? ["Credit Card Processing", "One-time Payments"] : []
      },
      paypal: {
        configured: paypalConfigured,
        features: paypalConfigured ? ["PayPal Payments", "Subscription Management"] : []
      }
    });
  });
  app2.get("/api/zoom/status", (req, res) => {
    const zoomConfigured = !!(process.env.ZOOM_API_KEY && process.env.ZOOM_API_SECRET && process.env.ZOOM_ACCOUNT_ID);
    res.json({
      configured: zoomConfigured,
      features: zoomConfigured ? ["Virtual Meetings", "Webinars", "Training Sessions"] : [],
      service: "Zoom"
    });
  });
  app2.post("/api/calendar/check-availability", async (req, res) => {
    try {
      if (!req.session?.googleTokens) {
        return res.status(401).json({ message: "Google Calendar not connected" });
      }
      const { startTime, endTime } = req.body;
      if (!startTime || !endTime) {
        return res.status(400).json({ message: "Start time and end time required" });
      }
      googleCalendarService.setCredentials(req.session.googleTokens);
      const isAvailable = await googleCalendarService.checkAvailability(
        startTime,
        endTime,
        "primary"
      );
      res.json({ available: isAvailable });
    } catch (error) {
      console.error("Availability check error:", error);
      res.status(500).json({
        message: "Failed to check calendar availability",
        error: error.message
      });
    }
  });
  app2.get("/api/calendar/events", async (req, res) => {
    try {
      if (!req.session?.googleTokens) {
        return res.status(401).json({ message: "Google Calendar not connected" });
      }
      const { startDate, endDate } = req.query;
      googleCalendarService.setCredentials(req.session.googleTokens);
      const events = await googleCalendarService.getEvents(
        startDate || (/* @__PURE__ */ new Date()).toISOString(),
        endDate || new Date(Date.now() + 30 * 24 * 60 * 60 * 1e3).toISOString(),
        "primary"
      );
      res.json({ events });
    } catch (error) {
      console.error("Get events error:", error);
      res.status(500).json({
        message: "Failed to fetch calendar events",
        error: error.message
      });
    }
  });
  app2.get("/api/training-videos", async (req, res) => {
    try {
      const videos = await storage.getPublicTrainingVideos();
      res.json(videos);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve training videos" });
    }
  });
  app2.get("/api/training-videos/:id", async (req, res) => {
    try {
      const video = await storage.getTrainingVideo(req.params.id);
      if (!video) {
        return res.status(404).json({ message: "Training video not found" });
      }
      res.json(video);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve training video" });
    }
  });
  app2.post("/api/training-videos", async (req, res) => {
    try {
      const validatedData = insertTrainingVideoSchema.parse(req.body);
      const video = await storage.createTrainingVideo(validatedData);
      res.json({ success: true, video });
    } catch (error) {
      res.status(400).json({
        success: false,
        message: "Invalid video data",
        errors: error.errors || error.message
      });
    }
  });
  app2.post("/api/upload", upload.single("file"), async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ message: "No file uploaded" });
      }
      const fileUpload = await storage.createFileUpload({
        originalName: req.file.originalname,
        fileName: req.file.filename,
        fileSize: req.file.size.toString(),
        mimeType: req.file.mimetype,
        uploadedBy: req.body.uploadedBy || "anonymous"
      });
      res.json({
        success: true,
        message: "File uploaded successfully",
        file: fileUpload
      });
    } catch (error) {
      res.status(500).json({
        success: false,
        message: "Failed to upload file",
        error: error.message
      });
    }
  });
  app2.get("/api/uploads", async (req, res) => {
    try {
      const files = await storage.getAllFileUploads();
      res.json(files);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve uploaded files" });
    }
  });
  app2.post("/api/emergency-contact", async (req, res) => {
    try {
      const { name, email, phone, urgentMessage } = req.body;
      const emergencyContact = await storage.createContact({
        firstName: name.split(" ")[0] || name,
        lastName: name.split(" ").slice(1).join(" ") || "",
        email,
        phone,
        message: urgentMessage,
        isUrgent: true,
        serviceInterest: "emergency"
      });
      console.log("URGENT: Emergency contact submission:", emergencyContact);
      res.json({
        success: true,
        message: "Emergency contact submitted. You will be contacted within 1 hour.",
        id: emergencyContact.id
      });
    } catch (error) {
      res.status(500).json({
        success: false,
        message: "Failed to submit emergency contact",
        error: error.message
      });
    }
  });
  app2.get("/api/posts", async (req, res) => {
    try {
      const posts2 = await storage.getAllPosts();
      const postsWithUsers = await Promise.all(
        posts2.map(async (post) => {
          const user = await storage.getUser(post.userId);
          return { ...post, user };
        })
      );
      res.json(postsWithUsers);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve posts" });
    }
  });
  app2.post("/api/posts", async (req, res) => {
    try {
      const validatedData = insertPostSchema.parse(req.body);
      const post = await storage.createPost(validatedData);
      const user = await storage.getUser(post.userId);
      res.json({ ...post, user });
    } catch (error) {
      res.status(400).json({
        message: "Invalid post data",
        errors: error.errors || error.message
      });
    }
  });
  app2.post("/api/posts/:id/like", async (req, res) => {
    try {
      const postId = req.params.id;
      const { userId } = req.body;
      if (!userId) {
        return res.status(400).json({ message: "User ID required" });
      }
      const existingLike = await storage.getLike(postId, userId);
      if (existingLike) {
        await storage.deleteLike(postId, userId);
        res.json({ liked: false, message: "Post unliked" });
      } else {
        await storage.createLike(postId, userId);
        res.json({ liked: true, message: "Post liked" });
      }
    } catch (error) {
      res.status(500).json({ message: "Failed to process like", error: error.message });
    }
  });
  app2.get("/api/posts/:id/comments", async (req, res) => {
    try {
      const postId = req.params.id;
      const comments2 = await storage.getCommentsByPost(postId);
      const commentsWithUsers = await Promise.all(
        comments2.map(async (comment) => {
          const user = await storage.getUser(comment.userId);
          return { ...comment, user };
        })
      );
      res.json(commentsWithUsers);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve comments" });
    }
  });
  app2.post("/api/posts/:id/comments", async (req, res) => {
    try {
      const postId = req.params.id;
      const { content, userId } = req.body;
      if (!userId || !content) {
        return res.status(400).json({ message: "User ID and content are required" });
      }
      const commentData = { postId, userId, content };
      const validatedData = insertCommentSchema.parse(commentData);
      const comment = await storage.createComment(validatedData);
      const user = await storage.getUser(comment.userId);
      res.json({ ...comment, user });
    } catch (error) {
      res.status(400).json({
        message: "Invalid comment data",
        errors: error.errors || error.message
      });
    }
  });
  app2.get("/api/user", async (req, res) => {
    try {
      const currentUser = await storage.getUserByUsername("tmh_admin");
      if (currentUser) {
        res.json(currentUser);
      } else {
        res.status(404).json({ message: "User not found" });
      }
    } catch (error) {
      console.error("Error getting user profile:", error);
      res.status(500).json({ message: "Failed to get user" });
    }
  });
  app2.get("/api/auth/user", async (req, res) => {
    try {
      const currentUser = await storage.getUserByUsername("tmh_admin");
      if (currentUser) {
        res.json(currentUser);
      } else {
        res.status(404).json({ message: "User not found" });
      }
    } catch (error) {
      console.error("Error getting auth user profile:", error);
      res.status(500).json({ message: "Failed to get user" });
    }
  });
  app2.patch("/api/auth/user/settings", async (req, res) => {
    try {
      const { zoomPmi, zoomPmiPassword } = req.body;
      const currentUser = await storage.getUserByUsername("tmh_admin");
      if (!currentUser) {
        return res.status(404).json({ message: "Admin user not found" });
      }
      const updatedUser = await storage.updateUserSettings(currentUser.id, {
        zoomPmi,
        zoomPmiPassword
      });
      res.json(updatedUser);
    } catch (error) {
      console.error("Error updating user settings:", error);
      res.status(500).json({ message: "Failed to update settings" });
    }
  });
  app2.get("/api/monitoring", async (req, res) => {
    try {
      const [metrics, alerts, summary] = await Promise.all([
        storage.getMetrics(),
        storage.getAlerts(),
        storage.getSummaryStats()
      ]);
      const databaseLimit = 10 * 1024 * 1024 * 1024;
      const userLimit = 1e4;
      const apiLimit = 1e5;
      const databaseUsage = summary.storageUsed / databaseLimit * 100;
      const userUsage = summary.totalUsers / userLimit * 100;
      const apiUsage = summary.apiRequestsToday / apiLimit * 100;
      if (databaseUsage > 75 && !alerts.some((a) => a.resource === "database" && !a.isResolved)) {
        await storage.createAlert(
          "limit_warning",
          "database",
          summary.storageUsed,
          databaseLimit,
          75,
          `Database storage at ${databaseUsage.toFixed(1)}% capacity`
        );
      }
      if (apiUsage > 75 && !alerts.some((a) => a.resource === "api_requests" && !a.isResolved)) {
        await storage.createAlert(
          "high_usage",
          "api_requests",
          summary.apiRequestsToday,
          apiLimit,
          75,
          `API requests at ${apiUsage.toFixed(1)}% of daily limit`
        );
      }
      const updatedAlerts = await storage.getAlerts();
      res.json({
        metrics,
        alerts: updatedAlerts,
        summary
      });
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve monitoring data" });
    }
  });
  app2.get("/api/monitoring/metrics", async (req, res) => {
    try {
      const { type, hours } = req.query;
      const metrics = await storage.getMetrics(
        type,
        hours ? parseInt(hours) : void 0
      );
      res.json(metrics);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve metrics" });
    }
  });
  app2.get("/api/monitoring/alerts", async (req, res) => {
    try {
      const { activeOnly } = req.query;
      const alerts = await storage.getAlerts(activeOnly === "true");
      res.json(alerts);
    } catch (error) {
      res.status(500).json({ message: "Failed to retrieve alerts" });
    }
  });
  app2.post("/api/monitoring/alerts/:id/resolve", async (req, res) => {
    try {
      const alertId = req.params.id;
      res.json({ success: true, message: "Alert resolved" });
    } catch (error) {
      res.status(500).json({ message: "Failed to resolve alert" });
    }
  });
  app2.get("/api/payment-methods", async (req, res) => {
    try {
      const methods = {
        square: squareService.isAvailable(),
        paypal: paypalService.isAvailable(),
        cashapp: true
        // Always available as manual payment
      };
      res.json(methods);
    } catch (error) {
      res.status(500).json({ message: "Failed to check payment methods" });
    }
  });
  app2.post("/api/subscribe", async (req, res) => {
    try {
      const { planId, paymentMethod, amount } = req.body;
      if (!planId || !paymentMethod || !amount) {
        return res.status(400).json({ message: "Missing required fields" });
      }
      const subscriptionData = {
        userId: "demo-user",
        // In real app, get from authentication
        paymentProvider: paymentMethod,
        subscriptionId: randomUUID5(),
        status: "active",
        planType: planId,
        amount: parseInt(amount),
        currency: "USD",
        startDate: /* @__PURE__ */ new Date(),
        expiresAt: new Date(Date.now() + 30 * 24 * 60 * 60 * 1e3)
        // 30 days
      };
      const subscription = await storage.createSubscription(subscriptionData);
      let paymentResult = null;
      if (paymentMethod === "square" && squareService.isAvailable()) {
        paymentResult = { method: "square", status: "pending" };
      } else if (paymentMethod === "paypal" && paypalService.isAvailable()) {
        try {
          const order = await paypalService.createOrder(amount.toString(), "USD");
          paymentResult = {
            method: "paypal",
            status: "pending",
            orderId: order.data.id,
            approvalUrl: order.data.links?.find((link) => link.rel === "approve")?.href
          };
        } catch (error) {
          console.error("PayPal order creation failed:", error);
          paymentResult = { method: "paypal", status: "failed", error: "PayPal unavailable" };
        }
      } else if (paymentMethod === "cashapp") {
        paymentResult = {
          method: "cashapp",
          status: "pending_manual_verification",
          instructions: `Send $${amount} to $TMHGlobal with your email as the note`
        };
      }
      res.json({
        success: true,
        subscription,
        payment: paymentResult,
        message: "Subscription initiated successfully"
      });
    } catch (error) {
      console.error("Subscription creation error:", error);
      res.status(500).json({
        success: false,
        message: "Failed to create subscription"
      });
    }
  });
  app2.get("/api/paypal/setup", async (req, res) => {
    if (!paypalService.isAvailable()) {
      return res.status(503).json({ error: "PayPal service unavailable" });
    }
    try {
      const clientToken = await paypalService.getClientToken();
      res.json({ clientToken });
    } catch (error) {
      res.status(500).json({ error: "Failed to get PayPal client token" });
    }
  });
  app2.post("/api/paypal/order", async (req, res) => {
    if (!paypalService.isAvailable()) {
      return res.status(503).json({ error: "PayPal service unavailable" });
    }
    try {
      const { amount, currency = "USD", intent = "CAPTURE" } = req.body;
      const result = await paypalService.createOrder(amount, currency, intent);
      res.status(result.statusCode).json(result.data);
    } catch (error) {
      res.status(500).json({ error: error.message });
    }
  });
  app2.post("/api/paypal/order/:orderID/capture", async (req, res) => {
    if (!paypalService.isAvailable()) {
      return res.status(503).json({ error: "PayPal service unavailable" });
    }
    try {
      const { orderID } = req.params;
      const result = await paypalService.captureOrder(orderID);
      res.status(result.statusCode).json(result.data);
    } catch (error) {
      res.status(500).json({ error: error.message });
    }
  });
  app2.post("/api/sms", async (req, res) => {
    try {
      const { to, message } = req.body;
      if (!ringCentralService.isConfigured()) {
        return res.status(400).json({
          success: false,
          message: "SMS service not configured. Please contact administrator to set up RingCentral credentials."
        });
      }
      const result = await ringCentralService.sendSMS({
        to: [to],
        text: message
      });
      console.log(`SMS sent successfully to ${to}: ${message}`);
      res.json({
        success: true,
        message: "SMS sent successfully",
        smsId: result.id
      });
    } catch (error) {
      console.error("SMS sending error:", error);
      res.status(500).json({
        success: false,
        message: "Failed to send SMS",
        error: error.message
      });
    }
  });
  app2.get("/api/live-training", async (req, res) => {
    try {
      const category = req.query.category;
      const isPremiumOnly = req.query.premium === "true";
      const sessions = await storage.getLiveTrainingSessions(category, isPremiumOnly);
      res.json(sessions);
    } catch (error) {
      console.error("Error fetching live training sessions:", error);
      res.status(500).json({ message: "Failed to fetch training sessions" });
    }
  });
  app2.get("/api/live-training/upcoming", async (req, res) => {
    try {
      const sessions = await storage.getUpcomingTrainingSessions();
      res.json(sessions);
    } catch (error) {
      console.error("Error fetching upcoming training sessions:", error);
      res.status(500).json({ message: "Failed to fetch upcoming sessions" });
    }
  });
  app2.get("/api/live-training/:id", async (req, res) => {
    try {
      const session2 = await storage.getLiveTrainingSession(req.params.id);
      if (!session2) {
        return res.status(404).json({ message: "Training session not found" });
      }
      res.json(session2);
    } catch (error) {
      console.error("Error fetching training session:", error);
      res.status(500).json({ message: "Failed to fetch training session" });
    }
  });
  app2.post("/api/live-training", async (req, res) => {
    try {
      const sessionData = req.body;
      let zoomMeeting = null;
      if (sessionData.createZoomMeeting) {
        try {
          if (sessionData.sessionType === "webinar") {
            zoomMeeting = await zoomService.createWebinar({
              topic: sessionData.title,
              agenda: sessionData.agenda || sessionData.description,
              startTime: sessionData.scheduledStartTime,
              duration: sessionData.duration,
              timezone: sessionData.timezone,
              registrationRequired: true,
              practiceSession: true
            });
          } else {
            zoomMeeting = await zoomService.createTrainingSession({
              topic: sessionData.title,
              agenda: sessionData.agenda || sessionData.description,
              startTime: sessionData.scheduledStartTime,
              duration: sessionData.duration,
              timezone: sessionData.timezone,
              waitingRoom: true,
              allowRecording: sessionData.recordingEnabled
            });
          }
          sessionData.zoomMeetingId = zoomMeeting.id.toString();
          sessionData.zoomMeetingUuid = zoomMeeting.uuid;
          sessionData.zoomJoinUrl = zoomMeeting.join_url;
          sessionData.zoomStartUrl = zoomMeeting.start_url;
          sessionData.zoomPassword = zoomMeeting.password;
        } catch (zoomError) {
          console.error("Error creating Zoom meeting:", zoomError);
          return res.status(500).json({ message: "Failed to create Zoom meeting" });
        }
      }
      const session2 = await storage.createLiveTrainingSession(sessionData);
      res.json(session2);
    } catch (error) {
      console.error("Error creating training session:", error);
      res.status(500).json({ message: "Failed to create training session" });
    }
  });
  app2.put("/api/live-training/:id", async (req, res) => {
    try {
      const sessionId = req.params.id;
      const updates = req.body;
      if (updates.updateZoomMeeting && updates.zoomMeetingId) {
        try {
          await zoomService.updateMeeting(updates.zoomMeetingId, {
            topic: updates.title,
            agenda: updates.agenda,
            start_time: updates.scheduledStartTime,
            duration: updates.duration
          });
        } catch (zoomError) {
          console.error("Error updating Zoom meeting:", zoomError);
        }
      }
      const session2 = await storage.updateLiveTrainingSession(sessionId, updates);
      if (!session2) {
        return res.status(404).json({ message: "Training session not found" });
      }
      res.json(session2);
    } catch (error) {
      console.error("Error updating training session:", error);
      res.status(500).json({ message: "Failed to update training session" });
    }
  });
  app2.post("/api/live-training/:id/register", async (req, res) => {
    try {
      const sessionId = req.params.id;
      const userId = req.body.userId;
      const existingRegistration = await storage.getTrainingRegistration(sessionId, userId);
      if (existingRegistration) {
        return res.status(400).json({ message: "Already registered for this session" });
      }
      const session2 = await storage.getLiveTrainingSession(sessionId);
      if (!session2) {
        return res.status(404).json({ message: "Training session not found" });
      }
      let zoomRegistration = null;
      if (session2.sessionType === "webinar" && session2.zoomMeetingId) {
        try {
          const user = await storage.getUser(userId);
          if (user) {
            zoomRegistration = await zoomService.addWebinarRegistrant(session2.zoomMeetingId, {
              email: user.email || "",
              first_name: user.firstName || "",
              last_name: user.lastName || "",
              organization: user.company || ""
            });
          }
        } catch (zoomError) {
          console.error("Error registering for Zoom webinar:", zoomError);
        }
      }
      const registration = await storage.createTrainingRegistration({
        sessionId,
        userId,
        zoomRegistrantId: zoomRegistration?.registrant_id || null,
        zoomJoinUrl: zoomRegistration?.join_url || session2.zoomJoinUrl
      });
      res.json(registration);
    } catch (error) {
      console.error("Error registering for training session:", error);
      res.status(500).json({ message: "Failed to register for training session" });
    }
  });
  app2.get("/api/live-training/:id/registrations", async (req, res) => {
    try {
      const registrations = await storage.getTrainingRegistrations(req.params.id);
      res.json(registrations);
    } catch (error) {
      console.error("Error fetching registrations:", error);
      res.status(500).json({ message: "Failed to fetch registrations" });
    }
  });
  app2.get("/api/live-training/:id/resources", async (req, res) => {
    try {
      const resources = await storage.getTrainingResources(req.params.id);
      res.json(resources);
    } catch (error) {
      console.error("Error fetching training resources:", error);
      res.status(500).json({ message: "Failed to fetch resources" });
    }
  });
  app2.post("/api/live-training/:id/resources", upload.single("file"), async (req, res) => {
    try {
      const sessionId = req.params.id;
      const { title, description, resourceType } = req.body;
      let fileUrl = null;
      let fileSize = null;
      let mimeType = null;
      if (req.file) {
        fileUrl = `/uploads/${req.file.filename}`;
        fileSize = req.file.size;
        mimeType = req.file.mimetype;
      }
      const resource = await storage.createTrainingResource({
        sessionId,
        title,
        description,
        resourceType: resourceType || "document",
        fileUrl,
        fileSize,
        mimeType,
        uploadedBy: req.body.uploadedBy || "instructor"
      });
      res.json(resource);
    } catch (error) {
      console.error("Error creating training resource:", error);
      res.status(500).json({ message: "Failed to create resource" });
    }
  });
  app2.post("/api/live-training/instant", async (req, res) => {
    try {
      const { topic, agenda, instructorId, instructorName } = req.body;
      const adminUsers = Array.from(storage.users.values()).filter((u) => u.username === "tmh_admin");
      const userPmi = adminUsers.length > 0 && adminUsers[0].zoomPmi ? {
        pmi: adminUsers[0].zoomPmi,
        password: adminUsers[0].zoomPmiPassword || ""
      } : void 0;
      const zoomMeeting = await zoomService.createInstantMeeting(topic, agenda, userPmi);
      const session2 = await storage.createLiveTrainingSession({
        title: topic,
        description: agenda,
        category: "instant",
        sessionType: "meeting",
        instructorId,
        instructorName,
        zoomMeetingId: zoomMeeting.id.toString(),
        zoomMeetingUuid: zoomMeeting.uuid,
        zoomJoinUrl: zoomMeeting.join_url,
        zoomStartUrl: zoomMeeting.start_url,
        zoomPassword: zoomMeeting.password,
        scheduledStartTime: /* @__PURE__ */ new Date(),
        scheduledEndTime: new Date(Date.now() + 60 * 60 * 1e3),
        // 1 hour
        duration: 60,
        status: "live"
      });
      res.json(session2);
    } catch (error) {
      console.error("Error creating instant training session:", error);
      res.status(500).json({ message: "Failed to create instant session" });
    }
  });
  app2.get("/api/compliance/scans", async (req, res) => {
    try {
      const scans = await storage.getComplianceScans();
      res.json(scans);
    } catch (error) {
      console.error("Error fetching compliance scans:", error);
      res.status(500).json({ message: "Failed to fetch compliance scans" });
    }
  });
  app2.get("/api/compliance/scans/:id", async (req, res) => {
    try {
      const scan = await storage.getComplianceScan(req.params.id);
      if (!scan) {
        return res.status(404).json({ message: "Compliance scan not found" });
      }
      res.json(scan);
    } catch (error) {
      console.error("Error fetching compliance scan:", error);
      res.status(500).json({ message: "Failed to fetch compliance scan" });
    }
  });
  app2.post("/api/compliance/scan", async (req, res) => {
    try {
      const { url } = req.body;
      if (!url) {
        return res.status(400).json({ message: "URL is required" });
      }
      try {
        new URL(url);
      } catch {
        return res.status(400).json({ message: "Invalid URL format" });
      }
      const scanId = await complianceScanner.scanUrl(url, "manual");
      res.json({
        success: true,
        scanId,
        message: "Accessibility scan started successfully"
      });
    } catch (error) {
      console.error("Error starting compliance scan:", error);
      res.status(500).json({ message: "Failed to start compliance scan" });
    }
  });
  app2.get("/api/compliance/issues", async (req, res) => {
    try {
      const severity = req.query.severity;
      const issues = await storage.getComplianceIssues(severity);
      res.json(issues);
    } catch (error) {
      console.error("Error fetching compliance issues:", error);
      res.status(500).json({ message: "Failed to fetch compliance issues" });
    }
  });
  app2.get("/api/compliance/alerts", async (req, res) => {
    try {
      const alerts = await storage.getComplianceAlerts();
      res.json(alerts);
    } catch (error) {
      console.error("Error fetching compliance alerts:", error);
      res.status(500).json({ message: "Failed to fetch compliance alerts" });
    }
  });
  app2.put("/api/compliance/alerts/:id/read", async (req, res) => {
    try {
      const alertId = req.params.id;
      const alert = await storage.markComplianceAlertAsRead(alertId);
      if (!alert) {
        return res.status(404).json({ message: "Alert not found" });
      }
      res.json(alert);
    } catch (error) {
      console.error("Error marking alert as read:", error);
      res.status(500).json({ message: "Failed to mark alert as read" });
    }
  });
  app2.get("/api/compliance/reports/:scanId", async (req, res) => {
    try {
      const scanId = req.params.scanId;
      const report = await storage.getComplianceReport(scanId);
      if (!report) {
        return res.status(404).json({ message: "Compliance report not found" });
      }
      res.json(report);
    } catch (error) {
      console.error("Error fetching compliance report:", error);
      res.status(500).json({ message: "Failed to fetch compliance report" });
    }
  });
  app2.get("/api/training/polls/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const polls = await storage.getTrainingPolls(sessionId);
      res.json(polls);
    } catch (error) {
      console.error("Error fetching training polls:", error);
      res.status(500).json({ message: "Failed to fetch training polls" });
    }
  });
  app2.post("/api/training/polls/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const { question, options, createdBy } = req.body;
      const poll = await storage.createTrainingPoll({
        sessionId,
        question,
        options,
        createdBy,
        isActive: true
      });
      res.json(poll);
    } catch (error) {
      console.error("Error creating training poll:", error);
      res.status(500).json({ message: "Failed to create training poll" });
    }
  });
  app2.post("/api/training/polls/:pollId/vote", async (req, res) => {
    try {
      const pollId = req.params.pollId;
      const { optionIndex, userId } = req.body;
      const result = await storage.voteOnTrainingPoll(pollId, optionIndex, userId);
      res.json(result);
    } catch (error) {
      console.error("Error voting on poll:", error);
      res.status(500).json({ message: "Failed to vote on poll" });
    }
  });
  app2.get("/api/training/qa/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const questions = await storage.getTrainingQAQuestions(sessionId);
      res.json(questions);
    } catch (error) {
      console.error("Error fetching Q&A questions:", error);
      res.status(500).json({ message: "Failed to fetch Q&A questions" });
    }
  });
  app2.post("/api/training/qa/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const { question, askedBy } = req.body;
      const qaQuestion = await storage.createTrainingQAQuestion({
        sessionId,
        question,
        askedBy,
        status: "pending"
      });
      res.json(qaQuestion);
    } catch (error) {
      console.error("Error creating Q&A question:", error);
      res.status(500).json({ message: "Failed to create Q&A question" });
    }
  });
  app2.put("/api/training/qa/:questionId/answer", async (req, res) => {
    try {
      const questionId = req.params.questionId;
      const { answer, answeredBy } = req.body;
      const question = await storage.answerTrainingQAQuestion(questionId, answer, answeredBy);
      res.json(question);
    } catch (error) {
      console.error("Error answering Q&A question:", error);
      res.status(500).json({ message: "Failed to answer Q&A question" });
    }
  });
  app2.get("/api/training/breakouts/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const breakoutRooms = await storage.getTrainingBreakoutRooms(sessionId);
      res.json(breakoutRooms);
    } catch (error) {
      console.error("Error fetching breakout rooms:", error);
      res.status(500).json({ message: "Failed to fetch breakout rooms" });
    }
  });
  app2.post("/api/training/breakouts/:sessionId", async (req, res) => {
    try {
      const sessionId = req.params.sessionId;
      const { name, maxParticipants, topic } = req.body;
      const breakoutRoom = await storage.createTrainingBreakoutRoom({
        sessionId,
        name,
        maxParticipants,
        topic,
        status: "active",
        participants: []
      });
      res.json(breakoutRoom);
    } catch (error) {
      console.error("Error creating breakout room:", error);
      res.status(500).json({ message: "Failed to create breakout room" });
    }
  });
  app2.get("/api/consultations/availability", async (req, res) => {
    try {
      const date = req.query.date;
      const availableSlots = await storage.getConsultationAvailability(date);
      res.json(availableSlots);
    } catch (error) {
      console.error("Error fetching availability:", error);
      res.status(500).json({ message: "Failed to fetch availability" });
    }
  });
  app2.post("/api/consultations", async (req, res) => {
    try {
      const validatedData = insertConsultationSchema.parse(req.body);
      console.log("\u{1F4DD} Processing consultation booking:", validatedData.firstName, validatedData.lastName);
      let conflictWarning = null;
      if (req.session?.googleTokens && validatedData.preferredDate) {
        console.log("\u{1F517} Google tokens available, checking calendar availability...");
        googleCalendarService.setCredentials(req.session.googleTokens);
        const startTime = new Date(validatedData.preferredDate);
        const endTime = new Date(startTime.getTime() + 60 * 60 * 1e3);
        const isAvailable = await googleCalendarService.checkAvailability(
          startTime.toISOString(),
          endTime.toISOString(),
          "primary"
        );
        if (!isAvailable) {
          conflictWarning = "Note: There may be a scheduling conflict. We'll coordinate with you to resolve any timing issues.";
          console.log("\u26A0\uFE0F Calendar conflict detected, but allowing booking to proceed");
        }
        console.log("\u{1F5D3}\uFE0F Creating Google Calendar event for consultation...");
        const calendarEvent = await googleCalendarService.createEvent({
          summary: `Consultation: ${validatedData.firstName} ${validatedData.lastName}`,
          description: `Business consultation meeting.
Service: ${validatedData.serviceType}
Message: ${validatedData.message}`,
          startDateTime: startTime.toISOString(),
          endDateTime: endTime.toISOString(),
          attendeeEmail: validatedData.email,
          timeZone: "America/New_York"
        });
        console.log("\u2705 Google Calendar event created successfully:", calendarEvent.id);
        validatedData.calendarEventId = calendarEvent.id;
      } else {
        console.log("\u2139\uFE0F Google tokens not available - calendar event creation skipped");
      }
      const consultation = await storage.createConsultation(validatedData);
      if (req.session?.googleTokens) {
        try {
          gmailService.setCredentials(req.session.googleTokens);
          await gmailService.sendConsultationNotification(consultation);
          console.log("\u2705 Email notification sent for new consultation:", consultation.firstName, consultation.lastName);
        } catch (error) {
          console.error("\u274C Failed to send consultation email notification:", error);
        }
      } else {
        console.log("\u2139\uFE0F No Google tokens available - email notification skipped");
      }
      console.log("\u{1F4F1} Checking RingCentral SMS configuration...");
      if (ringCentralService.isConfigured()) {
        if (req.session?.ringCentralTokens) {
          console.log("\u{1F4F1} Loading RingCentral tokens from session...");
          ringCentralService.setTokens(req.session.ringCentralTokens);
        }
        if (ringCentralService.hasValidTokens()) {
          console.log("\u{1F4F1} RingCentral tokens valid, sending SMS alert...");
          try {
            const dateStr = consultation.preferredDate ? new Date(consultation.preferredDate).toLocaleDateString() : "TBD";
            if (consultation.phone) {
              const formattedPhone = ringCentralService.formatPhoneNumber(consultation.phone);
              await ringCentralService.sendSMS({
                to: [formattedPhone],
                text: `TMH Global LLC: Consultation confirmed! We'll contact you soon about ${consultation.serviceType} services. Support: support@tmhglobal.com or 703-829-7277. Reply STOP to opt out.`
              });
              console.log(`\u2705 SMS sent to customer: ${formattedPhone}`);
            }
            await ringCentralService.sendSMS({
              to: ["+17038297277"],
              // Your business phone number
              text: `\u{1F4C5} NEW CONSULTATION: ${consultation.firstName} ${consultation.lastName} scheduled for ${dateStr}. Service: ${consultation.serviceType}. Email: ${consultation.email}${consultation.phone ? `, Phone: ${consultation.phone}` : ""}`
            });
            console.log("\u2705 SMS alert sent for new consultation");
          } catch (error) {
            console.error("\u274C Failed to send consultation SMS alert:", error);
          }
        } else {
          console.log("\u2139\uFE0F RingCentral OAuth tokens not available - SMS notification skipped. Please authorize RingCentral SMS.");
        }
      } else {
        console.log("\u2139\uFE0F RingCentral not configured - SMS notification skipped");
      }
      console.log("New consultation request:", consultation);
      res.json({
        success: true,
        message: conflictWarning || "Consultation scheduled successfully and added to calendar",
        id: consultation.id,
        calendarIntegrated: !!req.session?.googleTokens,
        hasConflict: !!conflictWarning
      });
    } catch (error) {
      console.error("Consultation scheduling error:", error);
      res.status(400).json({
        success: false,
        message: error.message || "Invalid consultation data",
        errors: error.errors || error.message
      });
    }
  });
  app2.put("/api/consultations/:id/cancel", async (req, res) => {
    try {
      const consultationId = req.params.id;
      const consultation = await storage.updateConsultationStatus(consultationId, "cancelled");
      res.json(consultation);
    } catch (error) {
      console.error("Error cancelling consultation:", error);
      res.status(500).json({ message: "Failed to cancel consultation" });
    }
  });
  app2.put("/api/consultations/:id/reschedule", async (req, res) => {
    try {
      const consultationId = req.params.id;
      const { scheduledDate, scheduledTime } = req.body;
      const consultation = await storage.rescheduleConsultation(consultationId, scheduledDate, scheduledTime || "10:00");
      res.json(consultation);
    } catch (error) {
      console.error("Error rescheduling consultation:", error);
      res.status(500).json({ message: "Failed to reschedule consultation" });
    }
  });
  app2.put("/api/consultations/:id/complete", async (req, res) => {
    try {
      const consultationId = req.params.id;
      const { notes } = req.body;
      const consultation = await storage.completeConsultation(consultationId, notes);
      res.json(consultation);
    } catch (error) {
      console.error("Error completing consultation:", error);
      res.status(500).json({ message: "Failed to complete consultation" });
    }
  });
  app2.get("/api/sms-campaigns", async (req, res) => {
    try {
      const campaigns = await storage.getSMSCampaigns();
      res.json(campaigns);
    } catch (error) {
      console.error("Error fetching SMS campaigns:", error);
      res.status(500).json({ message: "Failed to fetch SMS campaigns" });
    }
  });
  app2.get("/api/sms-campaigns/stats", async (req, res) => {
    try {
      const stats = await storage.getSMSCampaignStats();
      res.json(stats);
    } catch (error) {
      console.error("Error fetching SMS campaign stats:", error);
      res.status(500).json({ message: "Failed to fetch SMS campaign statistics" });
    }
  });
  app2.post("/api/sms-campaigns", async (req, res) => {
    try {
      const { name, message, campaignType, targetAudience, tierTarget, scheduledAt } = req.body;
      if (!name || !message || !campaignType || !targetAudience) {
        return res.status(400).json({ message: "Missing required fields" });
      }
      if (message.length > 320) {
        return res.status(400).json({ message: "Message too long. Maximum 320 characters." });
      }
      const campaign = await storage.createSMSCampaign({
        name,
        message,
        campaignType,
        targetAudience,
        tierTarget,
        scheduledAt: scheduledAt ? new Date(scheduledAt) : null,
        status: "draft"
      });
      res.json(campaign);
    } catch (error) {
      console.error("Error creating SMS campaign:", error);
      res.status(500).json({ message: "Failed to create SMS campaign" });
    }
  });
  app2.post("/api/sms-campaigns/:id/send", async (req, res) => {
    try {
      const campaignId = req.params.id;
      if (!ringCentralService.isConfigured()) {
        return res.status(400).json({
          message: "RingCentral is not configured. Please check your API credentials."
        });
      }
      const campaign = await storage.getSMSCampaign(campaignId);
      if (!campaign) {
        return res.status(404).json({ message: "Campaign not found" });
      }
      if (campaign.status !== "draft") {
        return res.status(400).json({ message: "Campaign has already been sent" });
      }
      let recipients = [];
      if (campaign.targetAudience === "all" || campaign.targetAudience === "members") {
        const members = await storage.getUsersWithSMSOptIn(campaign.tierTarget);
        recipients.push(...members.map((m) => ({
          phone: m.phone,
          type: "member",
          id: m.id
        })));
      }
      if (campaign.targetAudience === "all" || campaign.targetAudience === "non-members") {
        const subscribers = await storage.getActiveSMSSubscribers();
        recipients.push(...subscribers.map((s) => ({
          phone: s.phone,
          type: "subscriber",
          id: s.id
        })));
      }
      if (recipients.length === 0) {
        return res.status(400).json({ message: "No recipients found for this campaign" });
      }
      await storage.updateSMSCampaignStatus(campaignId, "sending", recipients.length);
      let successCount = 0;
      let failureCount = 0;
      const batchSize = 100;
      for (let i = 0; i < recipients.length; i += batchSize) {
        const batch = recipients.slice(i, i + batchSize);
        try {
          const formattedPhones = batch.map((r) => ringCentralService.formatPhoneNumber(r.phone)).filter((phone) => ringCentralService.isValidPhoneNumber(phone));
          if (formattedPhones.length > 0) {
            if (formattedPhones.length === 1) {
              await ringCentralService.sendSMS({
                to: formattedPhones,
                text: campaign.message
              });
            } else {
              await ringCentralService.sendBatchSMS({
                from: process.env.RINGCENTRAL_USERNAME || "",
                messages: formattedPhones.map((phone) => ({ to: [phone] })),
                text: campaign.message
              });
            }
            successCount += formattedPhones.length;
            for (let j = 0; j < batch.length; j++) {
              const recipient = batch[j];
              await storage.createSMSLog({
                campaignId,
                recipientPhone: recipient.phone,
                recipientType: recipient.type,
                recipientId: recipient.id,
                message: campaign.message,
                status: "sent"
              });
            }
          }
        } catch (error) {
          console.error(`Failed to send batch ${i}-${i + batch.length}:`, error);
          failureCount += batch.length;
          for (const recipient of batch) {
            await storage.createSMSLog({
              campaignId,
              recipientPhone: recipient.phone,
              recipientType: recipient.type,
              recipientId: recipient.id,
              message: campaign.message,
              status: "failed",
              errorMessage: error instanceof Error ? error.message : "Unknown error"
            });
          }
        }
        if (i + batchSize < recipients.length) {
          await new Promise((resolve) => setTimeout(resolve, 1e3));
        }
      }
      await storage.completeSMSCampaign(campaignId, successCount, failureCount);
      res.json({
        success: true,
        totalRecipients: recipients.length,
        successful: successCount,
        failed: failureCount
      });
    } catch (error) {
      console.error("Error sending SMS campaign:", error);
      res.status(500).json({ message: "Failed to send SMS campaign" });
    }
  });
  app2.get("/api/sms-subscribers", async (req, res) => {
    try {
      const subscribers = await storage.getAllSMSSubscribers();
      res.json(subscribers);
    } catch (error) {
      console.error("Error fetching SMS subscribers:", error);
      res.status(500).json({ message: "Failed to fetch SMS subscribers" });
    }
  });
  app2.post("/api/sms-subscribers", async (req, res) => {
    try {
      const { phone, firstName, lastName, email, source, preferences } = req.body;
      if (!phone) {
        return res.status(400).json({ message: "Phone number is required" });
      }
      if (!ringCentralService.isValidPhoneNumber(phone)) {
        return res.status(400).json({ message: "Invalid phone number format" });
      }
      const formattedPhone = ringCentralService.formatPhoneNumber(phone);
      const subscriber = await storage.createSMSSubscriber({
        phone: formattedPhone,
        firstName,
        lastName,
        email,
        source: source || "website",
        preferences: preferences || "all",
        isActive: true
      });
      res.json(subscriber);
    } catch (error) {
      if (error.code === "23505") {
        return res.status(400).json({ message: "Phone number is already subscribed" });
      }
      console.error("Error creating SMS subscriber:", error);
      res.status(500).json({ message: "Failed to create SMS subscriber" });
    }
  });
  app2.post("/api/sms-subscribers/:phone/opt-out", async (req, res) => {
    try {
      const phone = req.params.phone;
      const formattedPhone = ringCentralService.formatPhoneNumber(phone);
      await storage.optOutSMSSubscriber(formattedPhone);
      res.json({ success: true, message: "Successfully opted out of SMS notifications" });
    } catch (error) {
      console.error("Error opting out SMS subscriber:", error);
      res.status(500).json({ message: "Failed to opt out of SMS notifications" });
    }
  });
  const httpServer = createServer(app2);
  return httpServer;
}

// server/vite.ts
import express from "express";
import fs from "fs";
import path2 from "path";
import { createServer as createViteServer, createLogger } from "vite";

// vite.config.ts
import { defineConfig } from "vite";
import react from "@vitejs/plugin-react";
import path from "path";
import runtimeErrorOverlay from "@replit/vite-plugin-runtime-error-modal";
var vite_config_default = defineConfig({
  plugins: [
    react(),
    runtimeErrorOverlay(),
    ...process.env.NODE_ENV !== "production" && process.env.REPL_ID !== void 0 ? [
      await import("@replit/vite-plugin-cartographer").then(
        (m) => m.cartographer()
      )
    ] : []
  ],
  resolve: {
    alias: {
      "@": path.resolve(import.meta.dirname, "client", "src"),
      "@shared": path.resolve(import.meta.dirname, "shared"),
      "@assets": path.resolve(import.meta.dirname, "attached_assets")
    }
  },
  root: path.resolve(import.meta.dirname, "client"),
  build: {
    outDir: path.resolve(import.meta.dirname, "dist/public"),
    emptyOutDir: true
  },
  server: {
    fs: {
      strict: true,
      deny: ["**/.*"]
    }
  }
});

// server/vite.ts
import { nanoid } from "nanoid";
var viteLogger = createLogger();
function log(message, source = "express") {
  const formattedTime = (/* @__PURE__ */ new Date()).toLocaleTimeString("en-US", {
    hour: "numeric",
    minute: "2-digit",
    second: "2-digit",
    hour12: true
  });
  console.log(`${formattedTime} [${source}] ${message}`);
}
async function setupVite(app2, server) {
  const serverOptions = {
    middlewareMode: true,
    hmr: { server },
    allowedHosts: true
  };
  const vite = await createViteServer({
    ...vite_config_default,
    configFile: false,
    customLogger: {
      ...viteLogger,
      error: (msg, options) => {
        viteLogger.error(msg, options);
        process.exit(1);
      }
    },
    server: serverOptions,
    appType: "custom"
  });
  app2.use(vite.middlewares);
  app2.use("*", async (req, res, next) => {
    const url = req.originalUrl;
    try {
      const clientTemplate = path2.resolve(
        import.meta.dirname,
        "..",
        "client",
        "index.html"
      );
      let template = await fs.promises.readFile(clientTemplate, "utf-8");
      template = template.replace(
        `src="/src/main.tsx"`,
        `src="/src/main.tsx?v=${nanoid()}"`
      );
      const page = await vite.transformIndexHtml(url, template);
      res.status(200).set({ "Content-Type": "text/html" }).end(page);
    } catch (e) {
      vite.ssrFixStacktrace(e);
      next(e);
    }
  });
}
function serveStatic(app2) {
  const distPath = path2.resolve(import.meta.dirname, "public");
  if (!fs.existsSync(distPath)) {
    throw new Error(
      `Could not find the build directory: ${distPath}, make sure to build the client first`
    );
  }
  app2.use(express.static(distPath));
  app2.use("*", (_req, res) => {
    res.sendFile(path2.resolve(distPath, "index.html"));
  });
}

// server/index.ts
process.on("uncaughtException", (error) => {
  console.error("Uncaught Exception:", error);
  console.error("Stack:", error.stack);
});
process.on("unhandledRejection", (reason, promise) => {
  console.error("Unhandled Rejection at:", promise, "reason:", reason);
});
var app = express2();
var PgSession = connectPgSimple(session);
if (!process.env.SESSION_SECRET) {
  console.warn("\u26A0\uFE0F  SESSION_SECRET not set. Using fallback for development only.");
  if (process.env.NODE_ENV === "production") {
    throw new Error("SESSION_SECRET must be set in production environment");
  }
}
app.use(session({
  store: new PgSession({
    pool,
    tableName: "user_sessions",
    createTableIfMissing: true
  }),
  secret: process.env.SESSION_SECRET || "tmh-global-dev-fallback-secret",
  resave: false,
  saveUninitialized: false,
  cookie: {
    secure: process.env.NODE_ENV === "production",
    // Secure in production
    httpOnly: true,
    maxAge: 24 * 60 * 60 * 1e3,
    // 24 hours
    sameSite: "lax"
    // CSRF protection
  }
}));
app.use(express2.json());
app.use(express2.urlencoded({ extended: false }));
app.use(async (req, res, next) => {
  try {
    if (req.path.startsWith("/api")) {
      const adminUser = await storage.getUserByUsername("tmh_admin");
      if (adminUser) {
        if (!req.session) {
          req.session = {};
        }
        if (!req.session.googleTokens) {
          const googleTokens = await storage.getOAuthTokens(adminUser.id, "google");
          if (googleTokens) {
            req.session.googleTokens = {
              access_token: googleTokens.accessToken,
              refresh_token: googleTokens.refreshToken || void 0,
              scope: googleTokens.scope || "",
              token_type: "Bearer",
              expiry_date: googleTokens.expiresAt ? googleTokens.expiresAt.getTime() : void 0
            };
            googleCalendarService.setCredentials(req.session.googleTokens);
            console.log("\u{1F504} Google tokens hydrated from database");
          }
        }
        if (!req.session.ringCentralTokens) {
          const ringCentralTokens = await storage.getOAuthTokens(adminUser.id, "ringcentral");
          if (ringCentralTokens) {
            req.session.ringCentralTokens = {
              accessToken: ringCentralTokens.accessToken,
              refreshToken: ringCentralTokens.refreshToken || void 0,
              expiresAt: ringCentralTokens.expiresAt || /* @__PURE__ */ new Date()
            };
            ringCentralService.setTokens(req.session.ringCentralTokens);
            console.log("\u{1F504} RingCentral tokens hydrated from database");
          }
        }
      }
    }
  } catch (error) {
    console.error("\u274C Token hydration error:", error);
  }
  next();
});
app.use((req, res, next) => {
  const start = Date.now();
  const path3 = req.path;
  let capturedJsonResponse = void 0;
  const originalResJson = res.json;
  res.json = function(bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };
  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path3.startsWith("/api")) {
      let logLine = `${req.method} ${path3} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }
      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "\u2026";
      }
      log(logLine);
    }
  });
  next();
});
(async () => {
  const server = await registerRoutes(app);
  app.use((err, _req, res, _next) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";
    console.error("Express error handler:", err);
    res.status(status).json({ message });
  });
  if (app.get("env") === "development") {
    await setupVite(app, server);
  } else {
    serveStatic(app);
  }
  const port = parseInt(process.env.PORT || "5000", 10);
  server.listen({
    port,
    host: "0.0.0.0",
    reusePort: true
  }, () => {
    log(`serving on port ${port}`);
  });
})();
