import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { 
  Play,
  Lock,
  Clock,
  Users,
  FileText,
  BookOpen,
  CheckCircle,
  Star,
  Shield,
  Crown,
  ArrowRight,
  TrendingUp,
  Rocket
} from "lucide-react";
import { Link, useLocation } from "wouter";

interface TrainingModule {
  id: string;
  title: string;
  description: string;
  duration: string;
  difficulty: 'beginner' | 'intermediate' | 'advanced';
  category: string;
  isPremium: boolean;
  isCompleted?: boolean;
  thumbnailUrl?: string;
}

const trainingModules: TrainingModule[] = [
  // Accounting & Bookkeeping (60% focus)
  {
    id: 'quickbooks-fundamentals',
    title: 'QuickBooks Fundamentals for Small Business',
    description: 'Master essential bookkeeping and accounting with QuickBooks, from setup to financial reporting.',
    duration: '2h 30min',
    difficulty: 'beginner',
    category: 'Accounting',
    isPremium: false
  },
  {
    id: 'tax-preparation-basics',
    title: 'Small Business Tax Preparation Essentials',
    description: 'Learn tax preparation fundamentals, deductions, and compliance requirements for small businesses.',
    duration: '3h 15min',
    difficulty: 'intermediate',
    category: 'Tax Preparation',
    isPremium: true
  },
  {
    id: 'payroll-administration',
    title: 'Payroll Administration & Compliance',
    description: 'Complete guide to payroll processing, tax withholdings, and regulatory compliance.',
    duration: '2h 45min',
    difficulty: 'intermediate',
    category: 'Payroll',
    isPremium: true
  },
  {
    id: 'financial-reporting',
    title: 'Financial Reporting & Analysis',
    description: 'Create professional financial statements and perform meaningful business analysis.',
    duration: '3h 00min',
    difficulty: 'advanced',
    category: 'Accounting',
    isPremium: true
  },
  {
    id: 'cash-flow-management',
    title: 'Cash Flow Management Strategies',
    description: 'Master cash flow forecasting, budgeting, and working capital management.',
    duration: '2h 20min',
    difficulty: 'intermediate',
    category: 'Financial Management',
    isPremium: true
  },
  {
    id: 'year-end-closing',
    title: 'Year-End Closing Procedures',
    description: 'Step-by-step year-end closing process for accurate financial reporting.',
    duration: '2h 15min',
    difficulty: 'advanced',
    category: 'Accounting',
    isPremium: true
  },
  // Business Development (20% focus)
  {
    id: 'business-plan-development',
    title: 'Strategic Business Plan Development',
    description: 'Create comprehensive business plans for growth, funding, and strategic planning.',
    duration: '3h 45min',
    difficulty: 'advanced',
    category: 'Business Development',
    isPremium: true
  },
  {
    id: 'market-analysis',
    title: 'Market Research & Competitive Analysis',
    description: 'Conduct effective market research and competitive analysis for business growth.',
    duration: '2h 30min',
    difficulty: 'intermediate',
    category: 'Business Development',
    isPremium: true
  },
  // Government Contracting (20% focus)
  {
    id: 'intro-gov-contracting',
    title: 'Introduction to Government Contracting',
    description: 'Essential basics every contractor needs to know before entering the federal marketplace.',
    duration: '45 min',
    difficulty: 'beginner',
    category: 'Government Contracting',
    isPremium: false
  },
  {
    id: 'sam-registration',
    title: 'S.A.M. Registration Complete Guide',
    description: 'Step-by-step process for System for Award Management registration and maintenance.',
    duration: '1h 20min',
    difficulty: 'beginner',
    category: 'Government Contracting',
    isPremium: true
  },
  {
    id: 'vpat-documentation',
    title: 'VPAT Documentation & Section 508 Compliance',
    description: 'Master accessibility requirements and create professional VPAT documents.',
    duration: '2h 15min',
    difficulty: 'intermediate',
    category: 'Compliance',
    isPremium: true
  },
  {
    id: 'cybersecurity-compliance',
    title: 'NIST 800-171 & CMMC Compliance',
    description: 'Navigate cybersecurity requirements for defense contractors.',
    duration: '2h 45min',
    difficulty: 'advanced',
    category: 'Cybersecurity',
    isPremium: true
  },
  {
    id: 'subcontracting-strategies',
    title: 'Subcontracting & Teaming Strategies',
    description: 'Build strategic partnerships and subcontracting relationships.',
    duration: '1h 50min',
    difficulty: 'intermediate',
    category: 'Partnerships',
    isPremium: true
  }
];

const getDifficultyColor = (difficulty: string) => {
  switch (difficulty) {
    case 'beginner': return 'bg-green-100 text-green-800';
    case 'intermediate': return 'bg-yellow-100 text-yellow-800';
    case 'advanced': return 'bg-red-100 text-red-800';
    default: return 'bg-gray-100 text-gray-800';
  }
};

const getCategoryIcon = (category: string) => {
  switch (category) {
    case 'Accounting': return FileText;
    case 'Tax Preparation': return FileText;
    case 'Payroll': return Users;
    case 'Financial Management': return TrendingUp;
    case 'Business Development': return Rocket;
    case 'Government Contracting': return Shield;
    case 'Compliance': return Shield;
    case 'Cybersecurity': return Shield;
    case 'Partnerships': return Users;
    default: return BookOpen;
  }
};

export default function Training() {
  const [, setLocation] = useLocation();
  
  // Check if user has premium access
  const { data: user } = useQuery({
    queryKey: ["/api/user"],
  });

  const isPremiumUser = user && typeof user === 'object' && 'isPremium' in user && user.isPremium;

  const freeModules = trainingModules.filter(module => !module.isPremium);
  const premiumModules = trainingModules.filter(module => module.isPremium);

  const handleModuleClick = (module: TrainingModule) => {
    if (module.isPremium && !isPremiumUser) {
      return; // Prevent navigation for locked content
    }
    // Navigate to the client portal where training videos are available
    setLocation('/client-portal');
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50">
      {/* Header */}
      <div className="bg-white border-b shadow-sm">
        <div className="max-w-7xl mx-auto px-6 py-8">
          <div className="text-center">
            <h1 className="text-3xl font-bold text-gray-900 mb-4">Professional Training Library</h1>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Master essential business skills with our comprehensive training library focused on accounting,
              tax preparation, payroll management, and business development. Includes specialized government contracting content.
            </p>
          </div>
        </div>
      </div>

      <div className="max-w-6xl mx-auto px-6 py-12">
        
        {/* Premium Access Banner */}
        {!isPremiumUser && (
          <Alert className="mb-8 bg-gradient-to-r from-blue-50 to-indigo-50 border-blue-200">
            <Crown className="h-4 w-4 text-blue-600" />
            <AlertDescription>
              <div className="flex items-center justify-between">
                <div>
                  <strong className="text-blue-900">Unlock Full Training Library</strong>
                  <p className="text-blue-700 mt-1">
                    Get access to all premium training modules, including advanced accounting techniques, 
                    tax preparation strategies, payroll management, and specialized government contracting insights.
                  </p>
                </div>
                <Link href="/premium">
                  <Button className="bg-blue-600 hover:bg-blue-700 ml-4" data-testid="upgrade-premium">
                    <Crown className="h-4 w-4 mr-2" />
                    Upgrade to Premium
                  </Button>
                </Link>
              </div>
            </AlertDescription>
          </Alert>
        )}

        {/* Free Training Modules */}
        <div className="mb-12">
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-2xl font-bold text-gray-900">Free Training Modules</h2>
            <Badge variant="secondary" className="bg-green-100 text-green-800">
              {freeModules.length} modules
            </Badge>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {freeModules.map((module) => {
              const IconComponent = getCategoryIcon(module.category);
              return (
                <Card 
                  key={module.id} 
                  className="cursor-pointer transition-all duration-300 hover:shadow-lg hover:scale-[1.02]"
                  onClick={() => handleModuleClick(module)}
                  data-testid={`module-${module.id}`}
                >
                  <CardHeader className="pb-4">
                    <div className="flex items-start justify-between">
                      <div className="bg-blue-100 p-2 rounded-lg">
                        <IconComponent className="h-5 w-5 text-blue-600" />
                      </div>
                      <Badge className={getDifficultyColor(module.difficulty)}>
                        {module.difficulty}
                      </Badge>
                    </div>
                    <CardTitle className="text-lg leading-tight">{module.title}</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-gray-600 text-sm mb-4 line-clamp-3">{module.description}</p>
                    <div className="flex items-center justify-between text-sm text-gray-500">
                      <div className="flex items-center gap-1">
                        <Clock className="h-4 w-4" />
                        {module.duration}
                      </div>
                      <Badge variant="outline">{module.category}</Badge>
                    </div>
                    <Button className="w-full mt-4 bg-blue-600 hover:bg-blue-700">
                      <Play className="h-4 w-4 mr-2" />
                      Start Module
                    </Button>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>

        {/* Premium Training Modules */}
        <div>
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-2xl font-bold text-gray-900 flex items-center gap-2">
              <Crown className="h-6 w-6 text-yellow-500" />
              Premium Training Modules
            </h2>
            <Badge variant="secondary" className="bg-yellow-100 text-yellow-800">
              {premiumModules.length} modules
            </Badge>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {premiumModules.map((module) => {
              const IconComponent = getCategoryIcon(module.category);
              const isLocked = !isPremiumUser;
              
              return (
                <Card 
                  key={module.id} 
                  className={`transition-all duration-300 ${
                    isLocked 
                      ? 'opacity-75 cursor-not-allowed' 
                      : 'cursor-pointer hover:shadow-lg hover:scale-[1.02]'
                  } relative overflow-hidden`}
                  onClick={() => handleModuleClick(module)}
                  data-testid={`module-${module.id}`}
                >
                  {isLocked && (
                    <div className="absolute inset-0 bg-black/10 z-10 flex items-center justify-center">
                      <div className="bg-white rounded-full p-3 shadow-lg">
                        <Lock className="h-6 w-6 text-gray-600" />
                      </div>
                    </div>
                  )}
                  
                  <CardHeader className="pb-4">
                    <div className="flex items-start justify-between">
                      <div className="bg-yellow-100 p-2 rounded-lg">
                        <IconComponent className="h-5 w-5 text-yellow-600" />
                      </div>
                      <div className="flex gap-2">
                        <Badge className={getDifficultyColor(module.difficulty)}>
                          {module.difficulty}
                        </Badge>
                        <Badge className="bg-yellow-100 text-yellow-800">Premium</Badge>
                      </div>
                    </div>
                    <CardTitle className="text-lg leading-tight">{module.title}</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <p className="text-gray-600 text-sm mb-4 line-clamp-3">{module.description}</p>
                    <div className="flex items-center justify-between text-sm text-gray-500 mb-4">
                      <div className="flex items-center gap-1">
                        <Clock className="h-4 w-4" />
                        {module.duration}
                      </div>
                      <Badge variant="outline">{module.category}</Badge>
                    </div>
                    
                    {isLocked ? (
                      <Link href="/premium" className="block">
                        <Button variant="outline" className="w-full border-yellow-200 text-yellow-700 hover:bg-yellow-50">
                          <Crown className="h-4 w-4 mr-2" />
                          Unlock with Premium
                        </Button>
                      </Link>
                    ) : (
                      <Button className="w-full bg-yellow-600 hover:bg-yellow-700">
                        <Play className="h-4 w-4 mr-2" />
                        Start Module
                      </Button>
                    )}
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>

        {/* Success Stories Section */}
        <div className="mt-16 text-center">
          <Card className="border-0 shadow-lg bg-gradient-to-r from-green-50 to-emerald-50">
            <CardContent className="p-8">
              <h3 className="text-2xl font-bold text-gray-900 mb-4">Success Stories</h3>
              <p className="text-gray-600 mb-6 max-w-2xl mx-auto">
                Join thousands of successful government contractors who have used our training 
                to win contracts worth millions of dollars.
              </p>
              <div className="grid md:grid-cols-3 gap-6">
                <div className="text-center">
                  <div className="text-3xl font-bold text-green-600">$2.3M</div>
                  <div className="text-sm text-gray-600">Average contract value</div>
                </div>
                <div className="text-center">
                  <div className="text-3xl font-bold text-green-600">85%</div>
                  <div className="text-sm text-gray-600">Success rate increase</div>
                </div>
                <div className="text-center">
                  <div className="text-3xl font-bold text-green-600">30 days</div>
                  <div className="text-sm text-gray-600">Average time to first win</div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}