import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Textarea } from "@/components/ui/textarea";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { 
  MessageSquare, 
  Send, 
  Users, 
  Calendar,
  BarChart3,
  Plus,
  Clock,
  CheckCircle,
  XCircle,
  AlertTriangle
} from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

interface SMSCampaign {
  id: string;
  name: string;
  message: string;
  campaignType: string;
  targetAudience: string;
  tierTarget?: string;
  status: string;
  scheduledAt?: string;
  sentAt?: string;
  totalRecipients: number;
  successfulSends: number;
  failedSends: number;
  createdAt: string;
}

interface CampaignStats {
  totalCampaigns: number;
  totalSent: number;
  activeSubscribers: number;
  avgDeliveryRate: number;
}

interface NewCampaign {
  name: string;
  message: string;
  campaignType: string;
  targetAudience: string;
  tierTarget?: string;
  scheduledAt?: string;
}

const campaignTypes = [
  { value: "training", label: "Training Announcement", description: "New courses, workshops, live sessions" },
  { value: "promotion", label: "Special Promotion", description: "Discounts, limited offers, tier upgrades" },
  { value: "reminder", label: "Appointment Reminder", description: "Consultation reminders, payment due dates" },
  { value: "announcement", label: "General Announcement", description: "Company news, feature updates" }
];

const targetAudiences = [
  { value: "all", label: "Everyone", description: "All members and subscribers" },
  { value: "members", label: "Members Only", description: "Registered platform members" },
  { value: "non-members", label: "Subscribers Only", description: "Non-member SMS subscribers" },
  { value: "tier-specific", label: "Subscription Tier", description: "Target specific subscription level" }
];

const subscriptionTiers = [
  { value: "free", label: "Free Members" },
  { value: "starter", label: "Starter ($97/month)" },
  { value: "professional", label: "Professional ($197/month)" },
  { value: "premium", label: "Premium ($297/month)" }
];

function CampaignStatusBadge({ status }: { status: string }) {
  const config = {
    draft: { color: "bg-gray-100 text-gray-800", icon: Clock },
    scheduled: { color: "bg-blue-100 text-blue-800", icon: Calendar },
    sending: { color: "bg-yellow-100 text-yellow-800", icon: Send },
    completed: { color: "bg-green-100 text-green-800", icon: CheckCircle },
    cancelled: { color: "bg-red-100 text-red-800", icon: XCircle },
    failed: { color: "bg-red-100 text-red-800", icon: AlertTriangle }
  }[status] || { color: "bg-gray-100 text-gray-800", icon: Clock };

  const Icon = config.icon;

  return (
    <Badge variant="secondary" className={`${config.color} flex items-center gap-1`}>
      <Icon className="h-3 w-3" />
      {status.charAt(0).toUpperCase() + status.slice(1)}
    </Badge>
  );
}

export default function SMSCampaigns() {
  const [showCreateForm, setShowCreateForm] = useState(false);
  const [newCampaign, setNewCampaign] = useState<NewCampaign>({
    name: "",
    message: "",
    campaignType: "training",
    targetAudience: "all"
  });

  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch campaigns
  const { data: campaigns = [], isLoading } = useQuery<SMSCampaign[]>({
    queryKey: ["/api/sms-campaigns"],
  });

  // Fetch campaign statistics
  const { data: stats } = useQuery<CampaignStats>({
    queryKey: ["/api/sms-campaigns/stats"],
  });

  // Create campaign mutation
  const createCampaignMutation = useMutation({
    mutationFn: async (campaign: NewCampaign) => {
      const response = await apiRequest("POST", "/api/sms-campaigns", campaign);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/sms-campaigns"] });
      setShowCreateForm(false);
      setNewCampaign({
        name: "",
        message: "",
        campaignType: "training",
        targetAudience: "all"
      });
      toast({
        title: "Campaign Created",
        description: "Your SMS campaign has been created successfully.",
      });
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to create campaign. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Send campaign mutation
  const sendCampaignMutation = useMutation({
    mutationFn: async (campaignId: string) => {
      const response = await apiRequest("POST", `/api/sms-campaigns/${campaignId}/send`);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/sms-campaigns"] });
      toast({
        title: "Campaign Sent",
        description: "Your SMS campaign is being sent to recipients.",
      });
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to send campaign. Please try again.",
        variant: "destructive",
      });
    },
  });

  const handleCreateCampaign = () => {
    if (!newCampaign.name.trim() || !newCampaign.message.trim()) {
      toast({
        title: "Validation Error",
        description: "Please provide both campaign name and message.",
        variant: "destructive",
      });
      return;
    }

    if (newCampaign.message.length > 160) {
      toast({
        title: "Message Too Long",
        description: "SMS messages should be 160 characters or less for best delivery rates.",
        variant: "destructive",
      });
      return;
    }

    createCampaignMutation.mutate(newCampaign);
  };

  const getDeliveryRate = (campaign: SMSCampaign) => {
    if (campaign.totalRecipients === 0) return 0;
    return Math.round((campaign.successfulSends / campaign.totalRecipients) * 100);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 dark:from-gray-900 dark:to-gray-800 py-8">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="flex justify-between items-center mb-8">
          <div>
            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">SMS Campaigns</h1>
            <p className="text-gray-600 dark:text-gray-300">Promote training and specials to your community</p>
          </div>
          <Button onClick={() => setShowCreateForm(true)} data-testid="button-new-campaign">
            <Plus className="h-4 w-4 mr-2" />
            New Campaign
          </Button>
        </div>

        {/* Statistics Cards */}
        {stats && (
          <div className="grid md:grid-cols-4 gap-6 mb-8">
            <Card>
              <CardContent className="p-6">
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-blue-100 rounded-lg">
                    <MessageSquare className="h-5 w-5 text-blue-600" />
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Total Campaigns</p>
                    <p className="text-2xl font-bold">{stats.totalCampaigns || 0}</p>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardContent className="p-6">
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-green-100 rounded-lg">
                    <Send className="h-5 w-5 text-green-600" />
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Messages Sent</p>
                    <p className="text-2xl font-bold">{stats.totalSent || 0}</p>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardContent className="p-6">
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-purple-100 rounded-lg">
                    <Users className="h-5 w-5 text-purple-600" />
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Active Subscribers</p>
                    <p className="text-2xl font-bold">{stats.activeSubscribers || 0}</p>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardContent className="p-6">
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-yellow-100 rounded-lg">
                    <BarChart3 className="h-5 w-5 text-yellow-600" />
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Delivery Rate</p>
                    <p className="text-2xl font-bold">{stats.avgDeliveryRate || 0}%</p>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        )}

        <Tabs defaultValue="campaigns" className="space-y-6">
          <TabsList>
            <TabsTrigger value="campaigns">All Campaigns</TabsTrigger>
            <TabsTrigger value="create" className={showCreateForm ? "bg-blue-100" : ""}>
              {showCreateForm ? "Creating Campaign" : "Create Campaign"}
            </TabsTrigger>
          </TabsList>

          <TabsContent value="campaigns" className="space-y-6">
            {/* Campaign List */}
            {isLoading ? (
              <div className="text-center py-8">
                <div className="animate-spin w-8 h-8 border-4 border-primary border-t-transparent rounded-full mx-auto" />
                <p className="mt-2 text-gray-600">Loading campaigns...</p>
              </div>
            ) : campaigns.length === 0 ? (
              <Card>
                <CardContent className="text-center py-12">
                  <MessageSquare className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-semibold mb-2">No campaigns yet</h3>
                  <p className="text-gray-600 mb-4">Create your first SMS campaign to start reaching your community.</p>
                  <Button onClick={() => setShowCreateForm(true)}>
                    <Plus className="h-4 w-4 mr-2" />
                    Create First Campaign
                  </Button>
                </CardContent>
              </Card>
            ) : (
              <div className="grid gap-6">
                {campaigns.map((campaign: SMSCampaign) => (
                  <Card key={campaign.id}>
                    <CardHeader>
                      <div className="flex justify-between items-start">
                        <div>
                          <CardTitle className="flex items-center gap-2">
                            {campaign.name}
                            <CampaignStatusBadge status={campaign.status} />
                          </CardTitle>
                          <CardDescription>
                            {campaignTypes.find(t => t.value === campaign.campaignType)?.label} • 
                            {targetAudiences.find(a => a.value === campaign.targetAudience)?.label}
                            {campaign.tierTarget ? ` (${subscriptionTiers.find(t => t.value === campaign.tierTarget)?.label})` : ''}
                          </CardDescription>
                        </div>
                        <div className="flex gap-2">
                          {campaign.status === 'draft' && (
                            <Button
                              size="sm"
                              onClick={() => sendCampaignMutation.mutate(campaign.id)}
                              disabled={sendCampaignMutation.isPending}
                              data-testid={`button-send-${campaign.id}`}
                            >
                              <Send className="h-4 w-4 mr-1" />
                              Send Now
                            </Button>
                          )}
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent>
                      <div className="space-y-4">
                        <div>
                          <Label className="text-sm font-medium">Message:</Label>
                          <div className="bg-gray-50 dark:bg-gray-800 p-3 rounded-md mt-1">
                            <p className="text-sm">{campaign.message}</p>
                            <p className="text-xs text-gray-500 mt-1">{campaign.message.length}/160 characters</p>
                          </div>
                        </div>

                        {(campaign.status !== 'draft' && campaign.totalRecipients > 0) && (
                          <div className="grid md:grid-cols-3 gap-4 text-sm">
                            <div>
                              <p className="text-gray-600">Total Recipients</p>
                              <p className="font-semibold">{campaign.totalRecipients}</p>
                            </div>
                            <div>
                              <p className="text-gray-600">Successful Sends</p>
                              <p className="font-semibold text-green-600">{campaign.successfulSends}</p>
                            </div>
                            <div>
                              <p className="text-gray-600">Delivery Rate</p>
                              <p className="font-semibold">{getDeliveryRate(campaign)}%</p>
                            </div>
                          </div>
                        )}

                        <div className="text-xs text-gray-500">
                          Created {new Date(campaign.createdAt).toLocaleDateString()}
                          {campaign.sentAt && ` • Sent ${new Date(campaign.sentAt).toLocaleDateString()}`}
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          <TabsContent value="create" className="space-y-6">
            {/* Create Campaign Form */}
            <Card>
              <CardHeader>
                <CardTitle>Create New SMS Campaign</CardTitle>
                <CardDescription>
                  Reach your community with training announcements, promotions, and important updates
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <Alert>
                  <AlertTriangle className="h-4 w-4" />
                  <AlertDescription>
                    <strong>SMS Best Practices:</strong> Keep messages under 160 characters for best delivery rates. 
                    Include clear call-to-action and ensure compliance with SMS regulations.
                  </AlertDescription>
                </Alert>

                <div className="grid md:grid-cols-2 gap-6">
                  <div className="space-y-4">
                    <div>
                      <Label htmlFor="campaign-name">Campaign Name</Label>
                      <Input
                        id="campaign-name"
                        placeholder="e.g., Section 508 Workshop Announcement"
                        value={newCampaign.name}
                        onChange={(e) => setNewCampaign({ ...newCampaign, name: e.target.value })}
                        data-testid="input-campaign-name"
                      />
                    </div>

                    <div>
                      <Label htmlFor="campaign-type">Campaign Type</Label>
                      <Select 
                        value={newCampaign.campaignType} 
                        onValueChange={(value) => setNewCampaign({ ...newCampaign, campaignType: value })}
                      >
                        <SelectTrigger data-testid="select-campaign-type">
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          {campaignTypes.map((type) => (
                            <SelectItem key={type.value} value={type.value}>
                              <div>
                                <div className="font-medium">{type.label}</div>
                                <div className="text-xs text-gray-500">{type.description}</div>
                              </div>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="target-audience">Target Audience</Label>
                      <Select 
                        value={newCampaign.targetAudience} 
                        onValueChange={(value) => {
                          setNewCampaign({ 
                            ...newCampaign, 
                            targetAudience: value,
                            tierTarget: value === 'tier-specific' ? 'starter' : undefined
                          });
                        }}
                      >
                        <SelectTrigger data-testid="select-target-audience">
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          {targetAudiences.map((audience) => (
                            <SelectItem key={audience.value} value={audience.value}>
                              <div>
                                <div className="font-medium">{audience.label}</div>
                                <div className="text-xs text-gray-500">{audience.description}</div>
                              </div>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>

                    {newCampaign.targetAudience === 'tier-specific' && (
                      <div>
                        <Label htmlFor="tier-target">Subscription Tier</Label>
                        <Select 
                          value={newCampaign.tierTarget} 
                          onValueChange={(value) => setNewCampaign({ ...newCampaign, tierTarget: value })}
                        >
                          <SelectTrigger data-testid="select-tier-target">
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            {subscriptionTiers.map((tier) => (
                              <SelectItem key={tier.value} value={tier.value}>
                                {tier.label}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </div>
                    )}
                  </div>

                  <div className="space-y-4">
                    <div>
                      <Label htmlFor="message">SMS Message</Label>
                      <Textarea
                        id="message"
                        placeholder="Your SMS message here... Keep it short and actionable!"
                        value={newCampaign.message}
                        onChange={(e) => setNewCampaign({ ...newCampaign, message: e.target.value })}
                        className="min-h-[120px]"
                        maxLength={300}
                        data-testid="textarea-message"
                      />
                      <p className={`text-xs mt-1 ${newCampaign.message.length > 160 ? 'text-red-500' : 'text-gray-500'}`}>
                        {newCampaign.message.length}/160 characters 
                        {newCampaign.message.length > 160 && ' (May be sent as multiple messages)'}
                      </p>
                    </div>

                    <div className="bg-blue-50 dark:bg-blue-900/20 p-4 rounded-lg">
                      <h4 className="font-medium mb-2 text-blue-800 dark:text-blue-300">Message Examples:</h4>
                      <div className="space-y-2 text-sm">
                        <p className="text-blue-700 dark:text-blue-400">
                          <strong>Training:</strong> "New Section 508 workshop Jan 15, 2pm ET. Register: tmhglobal.com/register"
                        </p>
                        <p className="text-blue-700 dark:text-blue-400">
                          <strong>Promotion:</strong> "Limited time: 30% off Starter plan. Upgrade today: tmhglobal.com/pricing"
                        </p>
                      </div>
                    </div>
                  </div>
                </div>

                <div className="flex gap-4 pt-4">
                  <Button
                    onClick={handleCreateCampaign}
                    disabled={createCampaignMutation.isPending}
                    data-testid="button-create-campaign"
                  >
                    {createCampaignMutation.isPending ? "Creating..." : "Create Campaign"}
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => {
                      setShowCreateForm(false);
                      setNewCampaign({
                        name: "",
                        message: "",
                        campaignType: "training",
                        targetAudience: "all"
                      });
                    }}
                  >
                    Cancel
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}