import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { Settings as SettingsIcon, Video, Save, Calendar, ExternalLink, CheckCircle, XCircle, MessageSquare, Smartphone, CreditCard, DollarSign, Edit3, Wrench } from "lucide-react";
import type { User } from "@shared/schema";

export default function Settings() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  
  const { data: user, isLoading } = useQuery<User>({
    queryKey: ["/api/auth/user"],
  });

  // Check calendar connection status
  const { data: calendarStatus } = useQuery<{ connected: boolean; email?: string }>({
    queryKey: ["/api/calendar/status"],
  });

  // Check SMS/RingCentral status
  const { data: smsStatus } = useQuery<{ configured: boolean; connected?: boolean; service?: string; features?: string[]; authRequired?: boolean }>({
    queryKey: ["/api/sms/status"],
  });

  // Check payment services status
  const { data: paymentStatus } = useQuery<{ 
    square: { configured: boolean; features?: string[] }; 
    paypal: { configured: boolean; features?: string[] }; 
  }>({
    queryKey: ["/api/payments/status"],
  });

  const [zoomPmi, setZoomPmi] = useState("");
  const [zoomPmiPassword, setZoomPmiPassword] = useState("");
  const [ringCentralClientId, setRingCentralClientId] = useState("");
  const [ringCentralClientSecret, setRingCentralClientSecret] = useState("");
  const [showRingCentralConfig, setShowRingCentralConfig] = useState(false);

  // Initialize form values when user data loads
  useEffect(() => {
    if (user) {
      setZoomPmi(user.zoomPmi || "");
      setZoomPmiPassword(user.zoomPmiPassword || "");
    }
  }, [user]);

  // Listen for OAuth auth completion (Google and RingCentral)
  useEffect(() => {
    const handleMessage = (event: MessageEvent) => {
      if (event.data.type === 'GOOGLE_AUTH_SUCCESS') {
        toast({
          title: "Success",
          description: "Google Calendar connected successfully!",
        });
        queryClient.invalidateQueries({ queryKey: ["/api/calendar/status"] });
      } else if (event.data.type === 'GOOGLE_AUTH_ERROR') {
        toast({
          title: "Error",
          description: event.data.error || "Failed to connect Google Calendar",
          variant: "destructive",
        });
      } else if (event.data.type === 'RINGCENTRAL_AUTH_SUCCESS') {
        toast({
          title: "Success",
          description: "RingCentral SMS connected successfully!",
        });
        queryClient.invalidateQueries({ queryKey: ["/api/sms/status"] });
      } else if (event.data.type === 'RINGCENTRAL_AUTH_ERROR') {
        toast({
          title: "Error",
          description: event.data.error || "Failed to connect RingCentral SMS",
          variant: "destructive",
        });
      }
    };

    window.addEventListener('message', handleMessage);
    return () => window.removeEventListener('message', handleMessage);
  }, [toast, queryClient]);

  const updateSettings = useMutation({
    mutationFn: async (data: { zoomPmi: string; zoomPmiPassword: string }) => {
      await apiRequest("PATCH", "/api/auth/user/settings", data);
    },
    onSuccess: () => {
      toast({
        title: "Settings Updated",
        description: "Your Zoom PMI settings have been saved successfully.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/auth/user"] });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to update settings. Please try again.",
        variant: "destructive",
      });
      console.error("Settings update error:", error);
    },
  });

  // RingCentral configuration mutation
  const updateRingCentralConfig = useMutation({
    mutationFn: async (data: { clientId: string; clientSecret: string }) => {
      await apiRequest("POST", "/api/sms/configure", data);
    },
    onSuccess: () => {
      toast({
        title: "RingCentral Configured",
        description: "Your RingCentral credentials have been saved successfully.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/sms/status"] });
      setShowRingCentralConfig(false);
      setRingCentralClientId("");
      setRingCentralClientSecret("");
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to save RingCentral configuration. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Google Calendar connection mutation
  const connectCalendar = useMutation({
    mutationFn: async () => {
      const response = await apiRequest("GET", "/api/auth/google") as unknown as { authUrl: string };
      return response;
    },
    onSuccess: (data) => {
      // Open Google OAuth in a new window
      window.open(data.authUrl, '_blank', 'width=500,height=600');
      toast({
        title: "Authorization Required",
        description: "Please complete the authorization in the new window.",
      });
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to initiate Google Calendar connection.",
        variant: "destructive",
      });
    },
  });

  // Disconnect calendar mutation
  const disconnectCalendar = useMutation({
    mutationFn: async () => {
      await apiRequest("POST", "/api/calendar/disconnect");
    },
    onSuccess: () => {
      toast({
        title: "Disconnected",
        description: "Google Calendar has been disconnected.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/calendar/status"] });
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to disconnect Google Calendar.",
        variant: "destructive",
      });
    },
  });

  // RingCentral SMS connection mutation
  const connectRingCentral = useMutation({
    mutationFn: async () => {
      // Open popup window immediately to preserve user gesture
      const popup = window.open('about:blank', 'ringcentral_oauth', 'width=500,height=600');
      if (!popup) {
        throw new Error('Popup blocked. Please allow popups for this site.');
      }
      
      try {
        const response = await apiRequest("GET", "/api/auth/ringcentral");
        const data = await response.json() as { authUrl: string };
        // Navigate the popup to the auth URL
        if (data.authUrl) {
          popup.location.href = data.authUrl;
        } else {
          throw new Error('Auth URL not received from server');
        }
        return data;
      } catch (error) {
        popup.close();
        throw error;
      }
    },
    onSuccess: () => {
      toast({
        title: "Authorization Required",
        description: "Please complete the RingCentral authorization in the new window.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to initiate RingCentral SMS connection.",
        variant: "destructive",
      });
    },
  });

  // Disconnect RingCentral SMS mutation
  const disconnectRingCentral = useMutation({
    mutationFn: async () => {
      await apiRequest("POST", "/api/sms/disconnect");
    },
    onSuccess: () => {
      toast({
        title: "Disconnected",
        description: "RingCentral SMS has been disconnected.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/sms/status"] });
    },
    onError: () => {
      toast({
        title: "Error",
        description: "Failed to disconnect RingCentral SMS.",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    updateSettings.mutate({ zoomPmi, zoomPmiPassword });
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50 p-6">
        <div className="max-w-4xl mx-auto">
          <div className="animate-pulse space-y-6">
            <div className="h-8 bg-gray-200 rounded w-64"></div>
            <div className="h-64 bg-gray-200 rounded"></div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50 p-6">
      <div className="max-w-4xl mx-auto">
        <div className="flex items-center gap-3 mb-8">
          <SettingsIcon className="h-8 w-8 text-primary" aria-hidden="true" />
          <h1 className="text-3xl font-bold text-gray-900">Settings</h1>
        </div>

        <div className="grid gap-6">
          {/* Google Calendar Integration */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-3">
                <Calendar className="h-5 w-5 text-primary" aria-hidden="true" />
                <div>
                  <CardTitle>Google Calendar Integration</CardTitle>
                  <CardDescription>
                    Connect your Google Calendar for scheduling and email notifications
                  </CardDescription>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-center justify-between p-4 border rounded-lg">
                  <div className="flex items-center gap-3">
                    {calendarStatus?.connected ? (
                      <CheckCircle className="h-5 w-5 text-green-500" />
                    ) : (
                      <XCircle className="h-5 w-5 text-gray-400" />
                    )}
                    <div>
                      <p className="font-medium">
                        {calendarStatus?.connected ? "Connected" : "Not Connected"}
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {calendarStatus?.connected 
                          ? `Connected as: ${calendarStatus.email}`
                          : "Connect to enable consultation scheduling and email alerts"
                        }
                      </p>
                    </div>
                  </div>
                  
                  {calendarStatus?.connected ? (
                    <Button 
                      variant="outline" 
                      onClick={() => disconnectCalendar.mutate()}
                      disabled={disconnectCalendar.isPending}
                      data-testid="button-disconnect-calendar"
                    >
                      Disconnect
                    </Button>
                  ) : (
                    <Button 
                      onClick={async () => {
                        try {
                          const response = await fetch("/api/auth/google");
                          const data = await response.json();
                          if (data.authUrl) {
                            // Open in popup window now that redirect URI is fixed
                            window.open(data.authUrl, '_blank', 'width=500,height=600');
                            toast({ 
                              title: "Authorization Started", 
                              description: "Complete the authorization in the popup window" 
                            });
                          }
                        } catch (error) {
                          console.error('Failed to start Google auth:', error);
                          toast({ title: "Error", description: "Failed to start Google auth", variant: "destructive" });
                        }
                      }}
                      disabled={connectCalendar.isPending}
                      data-testid="button-connect-calendar"
                      className="flex items-center gap-2"
                    >
                      <ExternalLink className="h-4 w-4" />
                      Connect Google Calendar
                    </Button>
                  )}
                </div>
                
                {calendarStatus?.connected && (
                  <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                    <h4 className="font-semibold text-green-900 mb-2">✅ Features Enabled:</h4>
                    <ul className="text-sm text-green-800 space-y-1 list-disc list-inside">
                      <li>Automatic consultation scheduling</li>
                      <li>Calendar availability checking</li>
                      <li>Email notifications for contact forms</li>
                      <li>Conflict detection and prevention</li>
                    </ul>
                  </div>
                )}
                
                {!calendarStatus?.connected && (
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <h4 className="font-semibold text-blue-900 mb-2">📅 Why Connect Google Calendar?</h4>
                    <ul className="text-sm text-blue-800 space-y-1 list-disc list-inside">
                      <li>Automatically schedule consultations</li>
                      <li>Check availability before booking</li>
                      <li>Send email alerts for new contact submissions</li>
                      <li>Prevent double-booking conflicts</li>
                    </ul>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Zoom Integration Settings */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-3">
                <Video className="h-5 w-5 text-primary" aria-hidden="true" />
                <div>
                  <CardTitle>Zoom Integration</CardTitle>
                  <CardDescription>
                    Configure your Zoom Personal Meeting ID (PMI) for instant training sessions
                  </CardDescription>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-4">
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-2">
                    <Label htmlFor="zoom-pmi">
                      Personal Meeting ID (PMI)
                    </Label>
                    <Input
                      id="zoom-pmi"
                      data-testid="input-zoom-pmi"
                      type="text"
                      placeholder="e.g., 784 359 9500"
                      value={zoomPmi}
                      onChange={(e) => setZoomPmi(e.target.value)}
                      className="font-mono"
                    />
                    <p className="text-sm text-muted-foreground">
                      Your Personal Meeting ID for phone consultations
                    </p>
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="zoom-pmi-password">
                      Meeting Password (Optional)
                    </Label>
                    <Input
                      id="zoom-pmi-password"
                      data-testid="input-zoom-pmi-password"
                      type="password"
                      placeholder="Optional meeting password"
                      value={zoomPmiPassword}
                      onChange={(e) => setZoomPmiPassword(e.target.value)}
                    />
                    <p className="text-sm text-muted-foreground">
                      Optional password for additional security
                    </p>
                  </div>
                </div>

                <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                  <h4 className="font-semibold text-blue-900 mb-2">📞 Phone Consultation Setup:</h4>
                  <ol className="text-sm text-blue-800 space-y-1 list-decimal list-inside">
                    <li>Clients will call your main business number</li>
                    <li>Use your PMI for dedicated consultation sessions</li>
                    <li>Password provides additional meeting security</li>
                    <li>Format: XXX XXX XXXX (with or without spaces)</li>
                  </ol>
                </div>

                <div className="flex gap-3 pt-4">
                  <Button 
                    type="submit" 
                    disabled={updateSettings.isPending}
                    data-testid="button-save-settings"
                    className="flex items-center gap-2"
                  >
                    <Save className="h-4 w-4" />
                    {updateSettings.isPending ? "Saving..." : "Save Settings"}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>

          {/* RingCentral SMS Integration */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-3">
                <MessageSquare className="h-5 w-5 text-primary" aria-hidden="true" />
                <div>
                  <CardTitle>SMS Notifications</CardTitle>
                  <CardDescription>
                    RingCentral integration for SMS alerts when customers opt-in
                  </CardDescription>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-center justify-between p-4 border rounded-lg">
                  <div className="flex items-center gap-3">
                    {smsStatus?.connected ? (
                      <CheckCircle className="h-5 w-5 text-green-500" />
                    ) : smsStatus?.configured ? (
                      <XCircle className="h-5 w-5 text-amber-500" />
                    ) : (
                      <XCircle className="h-5 w-5 text-gray-400" />
                    )}
                    <div>
                      <p className="font-medium">
                        {smsStatus?.connected 
                          ? "RingCentral Connected" 
                          : smsStatus?.configured 
                            ? "Authorization Required"
                            : "SMS Not Configured"
                        }
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {smsStatus?.connected 
                          ? `Ready to send SMS notifications via ${smsStatus.service}`
                          : smsStatus?.configured
                            ? "RingCentral is configured but requires OAuth authorization"
                            : "Configure RingCentral API credentials to enable SMS alerts"
                        }
                      </p>
                    </div>
                  </div>
                  
                  {smsStatus?.connected ? (
                    <div className="flex gap-2">
                      <Button 
                        variant="outline" 
                        onClick={() => disconnectRingCentral.mutate()}
                        disabled={disconnectRingCentral.isPending}
                        data-testid="button-disconnect-sms"
                      >
                        Disconnect
                      </Button>
                      <Button 
                        variant="outline" 
                        size="sm"
                        data-testid="button-edit-config-sms"
                        onClick={() => setShowRingCentralConfig(true)}
                      >
                        <Edit3 className="h-4 w-4 mr-1" />
                        Edit Config
                      </Button>
                    </div>
                  ) : smsStatus?.configured ? (
                    <div className="flex gap-2">
                      <Button 
                        onClick={() => connectRingCentral.mutate()}
                        disabled={connectRingCentral.isPending}
                        data-testid="button-connect-sms"
                        className="flex items-center gap-2"
                      >
                        <ExternalLink className="h-4 w-4" />
                        Authorize SMS
                      </Button>
                      <Button 
                        variant="outline" 
                        size="sm"
                        data-testid="button-edit-config-sms"
                        onClick={() => setShowRingCentralConfig(true)}
                      >
                        <Edit3 className="h-4 w-4 mr-1" />
                        Edit Config
                      </Button>
                    </div>
                  ) : (
                    <Button 
                      variant="outline" 
                      size="sm"
                      data-testid="button-configure-sms"
                      onClick={() => setShowRingCentralConfig(true)}
                    >
                      <Wrench className="h-4 w-4 mr-1" />
                      Configure
                    </Button>
                  )}
                </div>
                
                {smsStatus?.configured && (
                  <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                    <h4 className="font-semibold text-green-900 mb-2">📱 SMS Features Enabled:</h4>
                    <ul className="text-sm text-green-800 space-y-1 list-disc list-inside">
                      <li>Automatic SMS alerts for urgent contact submissions</li>
                      <li>SMS notifications for customers who opt-in</li>
                      <li>Business phone number integration</li>
                      <li>Instant alerts for new consultation requests</li>
                    </ul>
                  </div>
                )}
                
                {!smsStatus?.configured && !showRingCentralConfig && (
                  <div className="bg-amber-50 border border-amber-200 rounded-lg p-4">
                    <h4 className="font-semibold text-amber-900 mb-2">⚠️ SMS Configuration Required</h4>
                    <p className="text-sm text-amber-800 mb-3">
                      Configure your RingCentral API credentials to enable SMS notifications.
                    </p>
                    <ul className="text-sm text-amber-800 space-y-1 list-disc list-inside">
                      <li>RingCentral Client ID & Secret</li>
                      <li>Account Username & Password</li>
                      <li>Extension Number</li>
                      <li>Server URL Configuration</li>
                    </ul>
                  </div>
                )}
                
                {showRingCentralConfig && (
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <h4 className="font-semibold text-blue-900 mb-4">🔧 Configure RingCentral API</h4>
                    <div className="space-y-4">
                      <div>
                        <Label htmlFor="ringcentral-client-id">Client ID</Label>
                        <Input
                          id="ringcentral-client-id"
                          type="text"
                          placeholder="Enter your RingCentral Client ID"
                          value={ringCentralClientId}
                          onChange={(e) => setRingCentralClientId(e.target.value)}
                          data-testid="input-ringcentral-client-id"
                        />
                      </div>
                      <div>
                        <Label htmlFor="ringcentral-client-secret">Client Secret</Label>
                        <Input
                          id="ringcentral-client-secret"
                          type="password"
                          placeholder="Enter your RingCentral Client Secret"
                          value={ringCentralClientSecret}
                          onChange={(e) => setRingCentralClientSecret(e.target.value)}
                          data-testid="input-ringcentral-client-secret"
                        />
                      </div>
                      <div className="flex gap-3">
                        <Button 
                          onClick={() => updateRingCentralConfig.mutate({ 
                            clientId: ringCentralClientId, 
                            clientSecret: ringCentralClientSecret 
                          })}
                          disabled={updateRingCentralConfig.isPending || !ringCentralClientId.trim() || !ringCentralClientSecret.trim()}
                          data-testid="button-save-ringcentral"
                          className="flex items-center gap-2"
                        >
                          <Save className="h-4 w-4" />
                          {updateRingCentralConfig.isPending ? "Saving..." : "Save Configuration"}
                        </Button>
                        <Button 
                          variant="outline"
                          onClick={() => {
                            setShowRingCentralConfig(false);
                            setRingCentralClientId("");
                            setRingCentralClientSecret("");
                          }}
                          data-testid="button-cancel-ringcentral"
                        >
                          Cancel
                        </Button>
                      </div>
                      <div className="text-xs text-blue-800">
                        <p><strong>Need these credentials?</strong></p>
                        <p>1. Go to <a href="https://developers.ringcentral.com/console" target="_blank" className="underline">RingCentral Developer Console</a></p>
                        <p>2. Create or select your app</p>
                        <p>3. Copy the Client ID and Client Secret from Settings</p>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Payment Services Integration */}
          <Card>
            <CardHeader>
              <div className="flex items-center gap-3">
                <CreditCard className="h-5 w-5 text-primary" aria-hidden="true" />
                <div>
                  <CardTitle>Payment Processing</CardTitle>
                  <CardDescription>
                    Square and PayPal integration for secure payment processing
                  </CardDescription>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-6">
                {/* Square Payment Status */}
                <div className="flex items-center justify-between p-4 border rounded-lg">
                  <div className="flex items-center gap-3">
                    {paymentStatus?.square?.configured ? (
                      <CheckCircle className="h-5 w-5 text-green-500" />
                    ) : (
                      <XCircle className="h-5 w-5 text-gray-400" />
                    )}
                    <div>
                      <p className="font-medium flex items-center gap-2">
                        <CreditCard className="h-4 w-4" />
                        Square Payments
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {paymentStatus?.square?.configured 
                          ? "Ready to process credit card payments"
                          : "Configure Square API for credit card processing"
                        }
                      </p>
                    </div>
                  </div>
                  
                  <Button 
                    variant="outline" 
                    size="sm"
                    data-testid="button-configure-square"
                    onClick={() => {
                      toast({ 
                        title: "Square Configuration", 
                        description: "Square API credentials configured via environment variables. Contact system administrator for SQUARE_ACCESS_TOKEN and SQUARE_APPLICATION_ID setup.",
                        duration: 5000
                      });
                    }}
                  >
                    <Wrench className="h-4 w-4 mr-1" />
                    {paymentStatus?.square?.configured ? "Manage" : "Setup"}
                  </Button>
                </div>

                {/* PayPal Payment Status */}
                <div className="flex items-center justify-between p-4 border rounded-lg">
                  <div className="flex items-center gap-3">
                    {paymentStatus?.paypal?.configured ? (
                      <CheckCircle className="h-5 w-5 text-green-500" />
                    ) : (
                      <XCircle className="h-5 w-5 text-gray-400" />
                    )}
                    <div>
                      <p className="font-medium flex items-center gap-2">
                        <DollarSign className="h-4 w-4" />
                        PayPal Payments
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {paymentStatus?.paypal?.configured 
                          ? "Ready for PayPal and subscription payments"
                          : "Configure PayPal API for online payments"
                        }
                      </p>
                    </div>
                  </div>
                  
                  <Button 
                    variant="outline" 
                    size="sm"
                    data-testid="button-configure-paypal"
                    onClick={() => {
                      toast({ 
                        title: "PayPal Configuration", 
                        description: "PayPal API credentials configured via environment variables. Contact system administrator for PAYPAL_CLIENT_ID and PAYPAL_CLIENT_SECRET setup.",
                        duration: 5000
                      });
                    }}
                  >
                    <Wrench className="h-4 w-4 mr-1" />
                    {paymentStatus?.paypal?.configured ? "Manage" : "Setup"}
                  </Button>
                </div>

                {/* Payment Features Summary */}
                {(paymentStatus?.square?.configured || paymentStatus?.paypal?.configured) ? (
                  <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                    <h4 className="font-semibold text-green-900 mb-2">💳 Payment Features Available:</h4>
                    <div className="grid gap-2">
                      {paymentStatus?.square?.configured && (
                        <div className="text-sm text-green-800">
                          <strong>Square:</strong> {paymentStatus.square.features?.join(", ")}
                        </div>
                      )}
                      {paymentStatus?.paypal?.configured && (
                        <div className="text-sm text-green-800">
                          <strong>PayPal:</strong> {paymentStatus.paypal.features?.join(", ")}
                        </div>
                      )}
                    </div>
                  </div>
                ) : (
                  <div className="bg-amber-50 border border-amber-200 rounded-lg p-4">
                    <h4 className="font-semibold text-amber-900 mb-2">⚠️ Payment Configuration Required</h4>
                    <p className="text-sm text-amber-800 mb-3">
                      Configure payment services to accept online payments for your accounting services.
                    </p>
                    <ul className="text-sm text-amber-800 space-y-1 list-disc list-inside">
                      <li>Square: Access Token & Application ID</li>
                      <li>PayPal: Client ID & Client Secret</li>
                      <li>Secure payment processing for consultations</li>
                      <li>Subscription management capabilities</li>
                    </ul>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>

          {/* Account Information */}
          <Card>
            <CardHeader>
              <CardTitle>Account Information</CardTitle>
              <CardDescription>Your account details</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <Label>Username</Label>
                  <p className="text-sm text-muted-foreground mt-1" data-testid="text-username">
                    {user?.username}
                  </p>
                </div>
                <div>
                  <Label>Email</Label>
                  <p className="text-sm text-muted-foreground mt-1" data-testid="text-email">
                    {user?.email || "Not provided"}
                  </p>
                </div>
                <div>
                  <Label>Company</Label>
                  <p className="text-sm text-muted-foreground mt-1" data-testid="text-company">
                    {user?.company || "Not provided"}
                  </p>
                </div>
                <div>
                  <Label>Premium Status</Label>
                  <p className="text-sm text-muted-foreground mt-1" data-testid="text-premium-status">
                    {user?.isPremium ? "Premium Member" : "Free Member"}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}