import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import PricingTiers from "@/components/services/pricing-tiers";
import ComplianceChecklist from "@/components/services/compliance-checklist";
import ServiceComparison from "@/components/services/service-comparison";
import { useLocation } from "wouter";
import { 
  Calculator, 
  Users, 
  TrendingUp, 
  Search, 
  Rocket, 
  FileText,
  CheckCircle,
  ArrowRight,
  Shield,
  BarChart3,
  CheckSquare
} from "lucide-react";

const services = [
  {
    id: "bookkeeping",
    icon: Calculator,
    title: "Bookkeeping & Accounting",
    description: "Complete financial record keeping, accounting services, and management for businesses of all sizes.",
    features: [
      "Monthly financial reconciliation",
      "Accounts payable and receivable management",
      "Expense tracking and categorization",
      "Financial reporting and analysis",
      "QuickBooks setup and maintenance",
      "Bank statement reconciliation",
      "Monthly financial statements",
      "General ledger management"
    ],
    pricing: "Starting at $500/month"
  },
  {
    id: "tax",
    icon: FileText,
    title: "Tax Preparation & Planning",
    description: "Comprehensive tax services including preparation, planning, and year-round compliance support.",
    features: [
      "Individual tax return preparation",
      "Business tax return preparation",
      "Tax planning and strategy",
      "Quarterly estimated payments",
      "Tax compliance monitoring",
      "IRS representation and support",
      "Multi-state tax filing",
      "Year-round tax consultation"
    ],
    pricing: "Starting at $200 for individual returns"
  },
  {
    id: "hr",
    icon: Users,
    title: "Payroll & HR Services",
    description: "Complete payroll processing and human resources support to manage your workforce effectively.",
    features: [
      "Full payroll processing and administration",
      "Benefits administration and enrollment",
      "Employee handbook development",
      "HR compliance monitoring",
      "Performance management systems",
      "Recruitment and onboarding support",
      "Training program development",
      "Workers compensation management"
    ],
    pricing: "Custom pricing based on employee count"
  },
  {
    id: "cfo",
    icon: TrendingUp,
    title: "CFO & Financial Strategy",
    description: "Strategic financial leadership, planning, and comprehensive financial management for growing businesses.",
    features: [
      "Financial strategy development",
      "Cash flow management and forecasting",
      "Budget planning and analysis",
      "Financial reporting and KPI tracking",
      "Investment and expansion planning",
      "Risk management and mitigation",
      "Board presentation preparation",
      "Financial process optimization"
    ],
    pricing: "Starting at $2,500/month"
  },
  {
    id: "business-development",
    icon: Rocket,
    title: "Business Development & Growth",
    description: "Strategic growth planning, market expansion, and business development services for sustainable success.",
    features: [
      "Business plan development and refinement",
      "Market analysis and research",
      "Growth strategy planning",
      "Competitive analysis and positioning",
      "Financial modeling and projections",
      "Investor presentation preparation",
      "Partnership and alliance development",
      "Revenue stream optimization"
    ],
    pricing: "Starting at $1,500/month"
  },
  {
    id: "auditing",
    icon: Search,
    title: "Auditing & Compliance",
    description: "Independent financial audits, compliance verification, and specialized government contracting support.",
    features: [
      "Financial statement audits",
      "Internal controls assessment",
      "Government compliance auditing",
      "Fraud investigation and prevention",
      "Risk assessment and mitigation",
      "Regulatory compliance reviews",
      "Audit report preparation",
      "Section 508 compliance support"
    ],
    pricing: "Project-based pricing"
  }
];

export default function Services() {
  const [, navigate] = useLocation();
  const [selectedServices, setSelectedServices] = useState<Record<string, string>>({});

  const handleContactUs = () => {
    navigate("/contact");
  };

  const handleSelectTier = (serviceId: string, tier: string) => {
    setSelectedServices(prev => ({ ...prev, [serviceId]: tier }));
  };

  const handleRequestAssistance = (itemId: string) => {
    navigate("/contact");
  };

  return (
    <div className="py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <h1 className="text-4xl lg:text-5xl font-bold text-foreground mb-6">
            Comprehensive Business & Financial Services
          </h1>
          <p className="text-xl text-muted-foreground max-w-3xl mx-auto">
            Full-service accounting, tax preparation, payroll, and business development solutions. 
            Specialized expertise in government contracting and regulatory compliance.
          </p>
        </div>

        {/* Services Grid */}
        <div className="space-y-16">
          {services.map((service, index) => (
            <div key={service.id} id={service.id} className="scroll-mt-20">
              <Card className="service-card">
                <div className="grid lg:grid-cols-2 gap-8 p-8">
                  <div>
                    <div className="flex items-center mb-6">
                      <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mr-4">
                        <service.icon className="h-6 w-6 text-primary" aria-hidden="true" />
                      </div>
                      <div>
                        <h2 className="text-2xl font-bold text-foreground mb-2">{service.title}</h2>
                        <p className="text-muted-foreground">{service.description}</p>
                      </div>
                    </div>

                    <div className="mb-6">
                      <h3 className="text-lg font-semibold text-foreground mb-4">What's Included:</h3>
                      <ul className="space-y-3">
                        {service.features.map((feature, featureIndex) => (
                          <li key={featureIndex} className="flex items-start">
                            <CheckCircle className="h-5 w-5 text-accent mr-3 mt-0.5 flex-shrink-0" aria-hidden="true" />
                            <span className="text-muted-foreground">{feature}</span>
                          </li>
                        ))}
                      </ul>
                    </div>

                    <div className="mb-6">
                      <p className="text-lg font-semibold text-primary">{service.pricing}</p>
                    </div>

                    <Button 
                      onClick={handleContactUs}
                      className="btn-primary"
                      data-testid={`button-contact-${service.id}`}
                    >
                      Get Started
                      <ArrowRight className="ml-2 h-4 w-4" aria-hidden="true" />
                    </Button>
                  </div>

                  <div className="flex items-center justify-center">
                    <div className="w-full h-64 bg-muted/50 rounded-lg flex items-center justify-center">
                      <service.icon className="h-24 w-24 text-primary/30" aria-hidden="true" />
                    </div>
                  </div>
                </div>
              </Card>
            </div>
          ))}
        </div>

        {/* Enhanced Service Features */}
        <div className="mt-20">
          <Tabs defaultValue="pricing" className="w-full">
            <TabsList className="grid w-full grid-cols-3 mb-12">
              <TabsTrigger value="pricing" className="flex items-center space-x-2">
                <BarChart3 className="h-4 w-4" />
                <span>Service Packages</span>
              </TabsTrigger>
              <TabsTrigger value="comparison" className="flex items-center space-x-2">
                <CheckSquare className="h-4 w-4" />
                <span>Compare Services</span>
              </TabsTrigger>
              <TabsTrigger value="compliance" className="flex items-center space-x-2">
                <Shield className="h-4 w-4" />
                <span>Compliance Guide</span>
              </TabsTrigger>
            </TabsList>

            <TabsContent value="pricing">
              <div className="space-y-16">
                {services.slice(0, 3).map((service, index) => (
                  <div key={service.id} className="scroll-mt-20">
                    <PricingTiers 
                      serviceId={service.id}
                      serviceTitle={service.title}
                      onSelectTier={handleSelectTier}
                    />
                  </div>
                ))}
              </div>
            </TabsContent>

            <TabsContent value="comparison">
              <ServiceComparison 
                selectedServices={Object.keys(selectedServices)}
                onSelectService={handleSelectTier}
              />
            </TabsContent>

            <TabsContent value="compliance">
              <ComplianceChecklist 
                showPersonalized={true}
                onRequestAssistance={handleRequestAssistance}
              />
            </TabsContent>
          </Tabs>
        </div>

        {/* Call to Action */}
        <div className="mt-20 text-center bg-muted/30 rounded-2xl p-12">
          <h2 className="text-3xl font-bold text-foreground mb-4">
            Ready to Get Started?
          </h2>
          <p className="text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Choose from our comprehensive service packages or let us create a custom solution 
            that perfectly fits your business needs and government contracting requirements.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button 
              onClick={handleContactUs}
              className="btn-primary"
              size="lg"
              data-testid="button-package-consultation"
            >
              <Shield className="mr-2 h-4 w-4" />
              Schedule Consultation
            </Button>
            <Button 
              onClick={() => window.location.href = '/government-contracting'}
              variant="outline"
              className="btn-secondary"
              size="lg"
              data-testid="button-learn-more-gov"
            >
              <CheckSquare className="mr-2 h-4 w-4" />
              Learn About Government Contracting
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
}
