import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Check, Star, Crown, Zap } from "lucide-react";
import { SUBSCRIPTION_TIERS, type SubscriptionTier } from "@shared/subscription-tiers";
import { useAuth } from "../hooks/useAuth";
import { useMutation } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

function TierIcon({ tierId }: { tierId: string }) {
  switch (tierId) {
    case 'starter':
      return <Zap className="h-6 w-6 text-blue-500" />;
    case 'professional':
      return <Star className="h-6 w-6 text-purple-500" />;
    case 'premium':
      return <Crown className="h-6 w-6 text-yellow-500" />;
    default:
      return <Check className="h-6 w-6 text-green-500" />;
  }
}

function TierCard({ tier }: { tier: SubscriptionTier }) {
  const { user, isAuthenticated } = useAuth();
  const { toast } = useToast();

  const handleSelectPlan = () => {
    if (!isAuthenticated) {
      window.location.href = "/api/login";
      return;
    }
    
    // For paid tiers, redirect to Square checkout
    if (tier.price > 0) {
      window.location.href = `/checkout?plan=${tier.id}`;
    } else {
      toast({
        title: "Welcome to TMH Global!",
        description: "Your free account is ready. Start exploring government contracting resources.",
      });
    }
  };

  return (
    <Card 
      className={`relative ${tier.popular ? 'border-2 border-purple-500 shadow-lg' : 'border border-gray-200'} 
                 hover:shadow-md transition-shadow`}
      data-testid={`tier-card-${tier.id}`}
    >
      {tier.popular && (
        <Badge className="absolute -top-3 left-1/2 transform -translate-x-1/2 bg-purple-500">
          Most Popular
        </Badge>
      )}
      
      <CardHeader className="text-center pb-4">
        <div className="flex justify-center mb-3">
          <TierIcon tierId={tier.id} />
        </div>
        <CardTitle className="text-2xl font-bold">{tier.name}</CardTitle>
        <CardDescription className="text-gray-600 dark:text-gray-300">
          {tier.description}
        </CardDescription>
        <div className="mt-4">
          <span className="text-4xl font-bold text-gray-900 dark:text-white">
            ${tier.price}
          </span>
          <span className="text-gray-600 dark:text-gray-300">
            /{tier.billingPeriod}
          </span>
        </div>
      </CardHeader>

      <CardContent className="space-y-3">
        {tier.features.map((feature, index) => (
          <div key={index} className="flex items-start gap-3">
            <Check className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
            <span className="text-sm text-gray-700 dark:text-gray-300">{feature}</span>
          </div>
        ))}
      </CardContent>

      <CardFooter>
        <Button 
          className={`w-full ${tier.popular ? 'bg-purple-600 hover:bg-purple-700' : ''}`}
          onClick={handleSelectPlan}
          data-testid={`button-select-${tier.id}`}
        >
          {tier.cta}
        </Button>
      </CardFooter>
    </Card>
  );
}

export default function Pricing() {
  const { user } = useAuth();
  
  // Show only paid tiers (free is available by default)
  const paidTiers = SUBSCRIPTION_TIERS.filter(tier => tier.price > 0);

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100 dark:from-gray-900 dark:to-gray-800 py-12">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-4xl font-bold text-gray-900 dark:text-white mb-4">
            Choose Your Government Contracting Success Plan
          </h1>
          <p className="text-xl text-gray-600 dark:text-gray-300 max-w-3xl mx-auto">
            Accelerate your government contracting journey with our comprehensive training, 
            compliance tools, and expert guidance. Start free and upgrade as you grow.
          </p>
        </div>

        {/* Pricing Tiers */}
        <div className="grid md:grid-cols-3 gap-8 max-w-7xl mx-auto">
          {paidTiers.map((tier) => (
            <TierCard key={tier.id} tier={tier} />
          ))}
        </div>

        {/* Free Tier Banner */}
        <div className="mt-12 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-8 text-center">
          <div className="flex justify-center mb-4">
            <Check className="h-8 w-8 text-green-500" />
          </div>
          <h3 className="text-2xl font-bold text-green-800 dark:text-green-300 mb-2">
            Start Free Today
          </h3>
          <p className="text-green-700 dark:text-green-400 mb-4">
            Get immediate access to our community, basic training videos, and government contracting guides.
            No credit card required!
          </p>
          <Button 
            variant="outline" 
            className="border-green-500 text-green-700 hover:bg-green-50 dark:text-green-300 dark:hover:bg-green-900/30"
            onClick={() => window.location.href = "/api/login"}
            data-testid="button-start-free"
          >
            Start Free Account
          </Button>
        </div>

        {/* Feature Comparison */}
        <div className="mt-16 bg-white dark:bg-gray-800 rounded-lg shadow-lg p-8">
          <h2 className="text-3xl font-bold text-center mb-8 text-gray-900 dark:text-white">
            What's Included in Each Plan
          </h2>
          
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            <div className="space-y-4">
              <h3 className="text-xl font-semibold text-blue-600 dark:text-blue-400 flex items-center gap-2">
                <Zap className="h-5 w-5" />
                Starter Features
              </h3>
              <ul className="space-y-2 text-sm text-gray-600 dark:text-gray-300">
                <li>• 2 live training sessions per month</li>
                <li>• Advanced training video library</li>
                <li>• Section 508 compliance checklist</li>
                <li>• Basic VPAT templates</li>
                <li>• Priority email support</li>
                <li>• Government contracting roadmap</li>
              </ul>
            </div>

            <div className="space-y-4">
              <h3 className="text-xl font-semibold text-purple-600 dark:text-purple-400 flex items-center gap-2">
                <Star className="h-5 w-5" />
                Professional Features
              </h3>
              <ul className="space-y-2 text-sm text-gray-600 dark:text-gray-300">
                <li>• Unlimited live training sessions</li>
                <li>• Interactive training workshops</li>
                <li>• 1 consultation per month</li>
                <li>• Advanced Section 508 monitoring</li>
                <li>• Custom VPAT generation</li>
                <li>• Phone support & SAM assistance</li>
              </ul>
            </div>

            <div className="space-y-4">
              <h3 className="text-xl font-semibold text-yellow-600 dark:text-yellow-400 flex items-center gap-2">
                <Crown className="h-5 w-5" />
                Premium Features
              </h3>
              <ul className="space-y-2 text-sm text-gray-600 dark:text-gray-300">
                <li>• Unlimited one-on-one consultations</li>
                <li>• White-label compliance reports</li>
                <li>• Dedicated account manager</li>
                <li>• Custom training programs</li>
                <li>• Advanced analytics</li>
                <li>• Direct access to compliance experts</li>
              </ul>
            </div>
          </div>
        </div>

        {/* FAQ Section */}
        <div className="mt-16 text-center">
          <h2 className="text-3xl font-bold mb-8 text-gray-900 dark:text-white">
            Frequently Asked Questions
          </h2>
          <div className="grid md:grid-cols-2 gap-8 max-w-4xl mx-auto">
            <div className="text-left">
              <h3 className="font-semibold text-lg mb-2 text-gray-900 dark:text-white">
                Can I upgrade or downgrade anytime?
              </h3>
              <p className="text-gray-600 dark:text-gray-300">
                Yes! You can change your plan at any time. Upgrades take effect immediately, 
                and downgrades apply at your next billing cycle.
              </p>
            </div>
            <div className="text-left">
              <h3 className="font-semibold text-lg mb-2 text-gray-900 dark:text-white">
                What payment methods do you accept?
              </h3>
              <p className="text-gray-600 dark:text-gray-300">
                We accept all major credit cards, PayPal, Square payments, and Cash App 
                for maximum convenience and security.
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}