import { useState, useEffect } from "react";
import { useMutation, useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { 
  Crown, 
  Shield, 
  CheckCircle, 
  Star, 
  CreditCard,
  DollarSign,
  Users,
  BookOpen,
  Video,
  FileText,
  Lock,
  Smartphone
} from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "../hooks/useAuth";
import { SUBSCRIPTION_TIERS, getTierById, hasAccessToTier } from "@shared/subscription-tiers";

interface PaymentOption {
  id: string;
  name: string;
  icon: any;
  description: string;
  available: boolean;
  primary?: boolean;
}

const getPaymentOptions = (paymentMethods?: any): PaymentOption[] => [
  {
    id: 'square',
    name: 'Square',
    icon: CreditCard,
    description: 'Credit/Debit Cards • Apple Pay • Google Pay',
    available: paymentMethods?.square || false,
    primary: true
  },
  {
    id: 'paypal',
    name: 'PayPal',
    icon: DollarSign,
    description: 'PayPal account or cards through PayPal',
    available: paymentMethods?.paypal || false
  },
  {
    id: 'cashapp',
    name: 'Cash App',
    icon: Smartphone,
    description: 'Manual payment via Cash App ($TMHGlobal)',
    available: paymentMethods?.cashapp !== false // Cash App is always available unless explicitly disabled
  }
];

// Use subscription tiers from shared config
const plans = SUBSCRIPTION_TIERS.filter(tier => tier.price > 0); // Show paid plans

export default function Premium() {
  const [selectedPlan, setSelectedPlan] = useState(plans[0] || SUBSCRIPTION_TIERS[1]);
  const [isProcessing, setIsProcessing] = useState(false);
  const { toast } = useToast();
  const [, setLocation] = useLocation();

  // Check if user has premium access
  const { user, isLoading, isAuthenticated } = useAuth();

  const { data: paymentMethods } = useQuery({
    queryKey: ["/api/payment-methods"],
  });

  const paymentOptions = getPaymentOptions(paymentMethods);
  const [selectedPayment, setSelectedPayment] = useState<PaymentOption>(paymentOptions[0]);

  // Update selected payment when payment methods data loads
  useEffect(() => {
    if (paymentMethods) {
      const availableOptions = getPaymentOptions(paymentMethods);
      const firstAvailable = availableOptions.find(p => p.available);
      if (firstAvailable) {
        setSelectedPayment(firstAvailable);
      } else {
        // If no methods are available, default to Cash App as it's always available
        const cashApp = availableOptions.find(p => p.id === 'cashapp');
        if (cashApp) {
          setSelectedPayment(cashApp);
        }
      }
    }
  }, [paymentMethods]);

  const subscribeMutation = useMutation({
    mutationFn: async ({ planId, paymentMethod }: { planId: string, paymentMethod: string }) => {
      const response = await apiRequest('POST', '/api/subscribe', {
        planId,
        paymentMethod,
        amount: selectedPlan.price,
        userId: 'demo-user' // In a real app, this would come from authentication
      });
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Subscription Activated!",
        description: "Welcome to TMH Global Premium. You now have access to all training content.",
      });
      setIsProcessing(false);
    },
    onError: (error: any) => {
      toast({
        title: "Payment Failed",
        description: error.message || "Unable to process payment. Please try again.",
        variant: "destructive",
      });
      setIsProcessing(false);
    },
  });

  const handleSubscribe = async () => {
    if (!selectedPayment.available) {
      toast({
        title: "Payment Method Unavailable",
        description: `${selectedPayment.name} is currently not configured. Please choose a different payment method or try Cash App.`,
        variant: "destructive",
      });
      return;
    }

    setIsProcessing(true);
    
    if (selectedPayment.id === 'cashapp') {
      // Handle Cash App manual payment
      toast({
        title: "Cash App Payment Instructions",
        description: `Please send $${selectedPlan.price} to $TMHGlobal with your email address as the note. Your premium access will be activated within 24 hours after payment verification.`,
      });
      setIsProcessing(false);
      return;
    }

    // For Square and PayPal, attempt the subscription
    subscribeMutation.mutate({
      planId: selectedPlan.id,
      paymentMethod: selectedPayment.id
    });
  };

  if (user && typeof user === 'object' && 'isPremium' in user && (user as any).isPremium) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50">
        {/* Header */}
        <div className="bg-white border-b shadow-sm">
          <div className="max-w-7xl mx-auto px-6 py-6">
            <div className="flex items-center gap-4">
              <div className="bg-gradient-to-r from-yellow-500 to-orange-500 p-3 rounded-lg">
                <Crown className="h-6 w-6 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold text-gray-900">Premium Member</h1>
                <p className="text-gray-600">Welcome back! You have full access to all training content.</p>
              </div>
            </div>
          </div>
        </div>

        <div className="max-w-4xl mx-auto px-6 py-8">
          <Card className="border-0 shadow-lg bg-gradient-to-r from-blue-50 to-indigo-50">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <CheckCircle className="h-5 w-5 text-green-600" />
                Premium Training Access
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <h3 className="font-semibold mb-3">Your Premium Benefits:</h3>
                  <ul className="space-y-2">
                    <li className="flex items-center gap-2">
                      <Video className="h-4 w-4 text-blue-600" />
                      <span>All training video libraries</span>
                    </li>
                    <li className="flex items-center gap-2">
                      <Shield className="h-4 w-4 text-blue-600" />
                      <span>Government contracting masterclasses</span>
                    </li>
                    <li className="flex items-center gap-2">
                      <FileText className="h-4 w-4 text-blue-600" />
                      <span>S.A.M. registration guidance</span>
                    </li>
                    <li className="flex items-center gap-2">
                      <Users className="h-4 w-4 text-blue-600" />
                      <span>Exclusive community access</span>
                    </li>
                  </ul>
                </div>
                <div className="flex items-center justify-center">
                  <Button 
                    size="lg" 
                    className="bg-gradient-to-r from-blue-600 to-indigo-600"
                    onClick={() => setLocation("/training")}
                    data-testid="button-access-training"
                  >
                    <BookOpen className="h-4 w-4 mr-2" />
                    Access Training Portal
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50">
      {/* Header */}
      <div className="bg-white border-b shadow-sm">
        <div className="max-w-7xl mx-auto px-6 py-8 text-center">
          <div className="inline-flex items-center gap-2 mb-4">
            <Crown className="h-8 w-8 text-yellow-500" />
            <h1 className="text-3xl font-bold text-gray-900">Premium Training Access</h1>
          </div>
          <p className="text-xl text-gray-600 max-w-2xl mx-auto">
            Unlock exclusive government contracting expertise with comprehensive training content, 
            expert guidance, and direct access to our community of successful contractors.
          </p>
        </div>
      </div>

      <div className="max-w-6xl mx-auto px-6 py-12">
        
        {/* Plan Selection */}
        <div className="mb-12">
          <h2 className="text-2xl font-bold text-center mb-8">Choose Your Plan</h2>
          <div className="grid md:grid-cols-2 gap-6">
            {plans.map((plan) => (
              <Card 
                key={plan.id} 
                className={`cursor-pointer transition-all duration-300 ${
                  selectedPlan.id === plan.id 
                    ? 'ring-2 ring-blue-600 shadow-xl' 
                    : 'hover:shadow-lg'
                } ${plan.highlighted ? 'border-blue-200 bg-blue-50/50' : ''}`}
                onClick={() => setSelectedPlan(plan)}
                data-testid={`plan-${plan.id}`}
              >
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <CardTitle className="text-xl">{plan.name}</CardTitle>
                    {plan.badge && (
                      <Badge variant="secondary" className="bg-blue-100 text-blue-800">
                        {plan.badge}
                      </Badge>
                    )}
                  </div>
                  <div className="flex items-baseline gap-1">
                    <span className="text-3xl font-bold">${plan.price}</span>
                    <span className="text-gray-600">/{plan.interval}</span>
                  </div>
                  <p className="text-gray-600">{plan.description}</p>
                </CardHeader>
                <CardContent>
                  <ul className="space-y-3">
                    {plan.features.map((feature, index) => (
                      <li key={index} className="flex items-center gap-2">
                        <CheckCircle className="h-4 w-4 text-green-600" />
                        <span className="text-sm">{feature}</span>
                      </li>
                    ))}
                  </ul>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Payment Method Selection */}
        <div className="mb-8">
          <h3 className="text-xl font-semibold mb-6 text-center">Payment Method</h3>
          {paymentMethods && typeof paymentMethods === 'object' && !(paymentMethods as any)?.square && !(paymentMethods as any)?.paypal ? (
            <Alert className="mb-4 bg-yellow-50 border-yellow-200">
              <AlertDescription>
                <strong>Note:</strong> Credit card processing is currently being configured. Cash App payments are available for immediate access.
              </AlertDescription>
            </Alert>
          ) : null}
          <div className="grid md:grid-cols-3 gap-4">
            {paymentOptions.map((option) => {
              const IconComponent = option.icon;
              return (
                <Card 
                  key={option.id}
                  className={`cursor-pointer transition-all duration-300 ${
                    selectedPayment.id === option.id 
                      ? 'ring-2 ring-blue-600 shadow-lg' 
                      : 'hover:shadow-md'
                  } ${!option.available ? 'opacity-50' : ''}`}
                  onClick={() => option.available && setSelectedPayment(option)}
                  data-testid={`payment-${option.id}`}
                >
                  <CardContent className="p-4">
                    <div className="flex items-center gap-3">
                      <div className={`p-2 rounded-lg ${option.primary ? 'bg-blue-100' : 'bg-gray-100'}`}>
                        <IconComponent className={`h-5 w-5 ${option.primary ? 'text-blue-600' : 'text-gray-600'}`} />
                      </div>
                      <div className="flex-1">
                        <div className="font-medium flex items-center gap-2">
                          {option.name}
                          {option.primary && option.available && (
                            <Badge variant="secondary" className="text-xs bg-blue-100 text-blue-800">
                              Primary
                            </Badge>
                          )}
                          {!option.available && (
                            <Badge variant="secondary" className="text-xs bg-gray-100 text-gray-600">
                              Unavailable
                            </Badge>
                          )}
                        </div>
                        <div className="text-sm text-gray-600">
                          {!option.available && option.id !== 'cashapp' 
                            ? 'Service not configured' 
                            : option.description
                          }
                        </div>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>

        {/* Cash App Instructions */}
        {selectedPayment.id === 'cashapp' && (
          <Alert className="mb-6 bg-green-50 border-green-200">
            <Smartphone className="h-4 w-4" />
            <AlertDescription>
              <strong>Cash App Payment Instructions:</strong><br />
              Send ${selectedPlan.price} to <strong>$TMHGlobal</strong> with your email address as the payment note. 
              Premium access will be activated within 24 hours of payment verification.
            </AlertDescription>
          </Alert>
        )}

        {/* Checkout Summary */}
        <Card className="border-0 shadow-xl">
          <CardHeader className="bg-gradient-to-r from-blue-600 to-indigo-600 text-white rounded-t-lg">
            <CardTitle className="flex items-center justify-between">
              <span>Order Summary</span>
              <Lock className="h-5 w-5" />
            </CardTitle>
          </CardHeader>
          <CardContent className="p-6">
            <div className="space-y-4">
              <div className="flex justify-between items-center">
                <span>{selectedPlan.name}</span>
                <span className="font-semibold">${selectedPlan.price}</span>
              </div>
              <div className="flex justify-between items-center text-sm text-gray-600">
                <span>Payment Method</span>
                <span>{selectedPayment.name}</span>
              </div>
              <hr />
              <div className="flex justify-between items-center text-lg font-bold">
                <span>Total</span>
                <span>${selectedPlan.price}</span>
              </div>
            </div>
            
            <Button
              onClick={handleSubscribe}
              disabled={isProcessing}
              className="w-full mt-6 bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-lg py-6"
              data-testid="subscribe-button"
            >
              {isProcessing ? (
                <div className="flex items-center gap-2">
                  <div className="animate-spin w-4 h-4 border-2 border-white border-t-transparent rounded-full" />
                  Processing...
                </div>
              ) : (
                <>
                  <Crown className="h-5 w-5 mr-2" />
                  Start Premium Access
                </>
              )}
            </Button>
            
            <p className="text-center text-sm text-gray-600 mt-4">
              Secure payment processing • Cancel anytime • 30-day money-back guarantee
            </p>
          </CardContent>
        </Card>

        {/* Trust Indicators */}
        <div className="mt-12 text-center">
          <div className="flex items-center justify-center gap-8 opacity-60">
            <div className="flex items-center gap-2">
              <Shield className="h-4 w-4" />
              <span className="text-sm">Secure Payment</span>
            </div>
            <div className="flex items-center gap-2">
              <Star className="h-4 w-4" />
              <span className="text-sm">Money-Back Guarantee</span>
            </div>
            <div className="flex items-center gap-2">
              <Users className="h-4 w-4" />
              <span className="text-sm">1000+ Members</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}