import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { 
  Activity, 
  Database, 
  Users, 
  FileText, 
  AlertTriangle, 
  CheckCircle, 
  TrendingUp, 
  Server,
  HardDrive,
  Zap,
  DollarSign
} from "lucide-react";

interface SystemMetric {
  id: string;
  metricType: string;
  metricName: string;
  value: number;
  metadata?: string;
  timestamp: Date;
}

interface UsageAlert {
  id: string;
  alertType: string;
  resource: string;
  currentValue: number;
  limitValue: number;
  threshold: number;
  message: string;
  isResolved: boolean;
  createdAt: Date;
}

interface MonitoringData {
  metrics: SystemMetric[];
  alerts: UsageAlert[];
  summary: {
    totalUsers: number;
    totalPosts: number;
    totalComments: number;
    apiRequestsToday: number;
    storageUsed: number;
    premiumUsers: number;
  };
}

export default function Monitoring() {
  // Fetch monitoring data
  const { data: monitoringData, isLoading } = useQuery<MonitoringData>({
    queryKey: ["/api/monitoring"],
    refetchInterval: 30000, // Refresh every 30 seconds
  });

  const formatBytes = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  const getAlertColor = (alertType: string) => {
    switch (alertType) {
      case 'limit_exceeded':
        return 'bg-red-100 text-red-800 border-red-300';
      case 'limit_warning':
        return 'bg-yellow-100 text-yellow-800 border-yellow-300';
      case 'high_usage':
        return 'bg-blue-100 text-blue-800 border-blue-300';
      default:
        return 'bg-gray-100 text-gray-800 border-gray-300';
    }
  };

  const getProgressColor = (percentage: number) => {
    if (percentage >= 90) return 'bg-red-500';
    if (percentage >= 75) return 'bg-yellow-500';
    return 'bg-green-500';
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50 p-6">
        <div className="max-w-7xl mx-auto">
          <div className="animate-pulse space-y-6">
            <div className="h-8 bg-gray-200 rounded w-64"></div>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              {[1, 2, 3, 4].map((i) => (
                <div key={i} className="h-32 bg-gray-200 rounded"></div>
              ))}
            </div>
          </div>
        </div>
      </div>
    );
  }

  const summary = monitoringData?.summary || {
    totalUsers: 0,
    totalPosts: 0,
    totalComments: 0,
    apiRequestsToday: 0,
    storageUsed: 0,
    premiumUsers: 0,
  };

  const alerts = monitoringData?.alerts || [];
  const activeAlerts = alerts.filter(alert => !alert.isResolved);

  // Calculate usage percentages (mock limits for demo)
  const databaseLimit = 10 * 1024 * 1024 * 1024; // 10GB in bytes
  const userLimit = 10000; // Mock user limit
  const apiLimit = 100000; // Mock daily API limit

  const databaseUsage = (summary.storageUsed / databaseLimit) * 100;
  const userUsage = (summary.totalUsers / userLimit) * 100;
  const apiUsage = (summary.apiRequestsToday / apiLimit) * 100;

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50">
      {/* Header */}
      <div className="bg-white border-b shadow-sm">
        <div className="max-w-7xl mx-auto px-6 py-6">
          <div className="flex items-center gap-4">
            <div className="bg-gradient-to-r from-blue-600 to-indigo-600 p-3 rounded-lg">
              <Activity className="h-6 w-6 text-white" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-gray-900">System Monitoring</h1>
              <p className="text-gray-600">Real-time platform metrics and usage tracking</p>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-6 py-8 space-y-8">
        
        {/* Active Alerts */}
        {activeAlerts.length > 0 && (
          <div className="space-y-4">
            <h2 className="text-xl font-semibold text-gray-900 flex items-center gap-2">
              <AlertTriangle className="h-5 w-5 text-orange-600" />
              Active Alerts
            </h2>
            {activeAlerts.map((alert) => (
              <Alert key={alert.id} className={getAlertColor(alert.alertType)}>
                <AlertTriangle className="h-4 w-4" />
                <AlertTitle className="capitalize">{alert.alertType.replace('_', ' ')}</AlertTitle>
                <AlertDescription>
                  <strong>{alert.resource}:</strong> {alert.message}
                  <br />
                  Current: {alert.currentValue} | Limit: {alert.limitValue} | Threshold: {alert.threshold}%
                </AlertDescription>
              </Alert>
            ))}
          </div>
        )}

        {/* Key Metrics Overview */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <Card className="border-0 shadow-sm bg-white">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-gray-600">
                Total Users
                <Users className="h-4 w-4 text-blue-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-gray-900" data-testid="metric-total-users">
                {summary.totalUsers.toLocaleString()}
              </div>
              <div className="mt-2">
                <Progress value={userUsage} className="h-2" />
                <p className="text-xs text-gray-500 mt-1">
                  {userUsage.toFixed(1)}% of limit
                </p>
              </div>
            </CardContent>
          </Card>

          <Card className="border-0 shadow-sm bg-white">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-gray-600">
                Premium Members
                <DollarSign className="h-4 w-4 text-yellow-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-gray-900" data-testid="metric-premium-users">
                {summary.premiumUsers.toLocaleString()}
              </div>
              <div className="mt-2">
                <Badge variant="secondary" className="bg-yellow-100 text-yellow-800">
                  {summary.totalUsers > 0 ? ((summary.premiumUsers / summary.totalUsers) * 100).toFixed(1) : 0}% conversion
                </Badge>
              </div>
            </CardContent>
          </Card>

          <Card className="border-0 shadow-sm bg-white">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-gray-600">
                Database Usage
                <Database className="h-4 w-4 text-green-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-gray-900" data-testid="metric-storage-used">
                {formatBytes(summary.storageUsed)}
              </div>
              <div className="mt-2">
                <Progress value={databaseUsage} className="h-2" />
                <p className="text-xs text-gray-500 mt-1">
                  {databaseUsage.toFixed(1)}% of 10GB limit
                </p>
              </div>
            </CardContent>
          </Card>

          <Card className="border-0 shadow-sm bg-white">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-gray-600">
                API Requests Today
                <Zap className="h-4 w-4 text-purple-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-gray-900" data-testid="metric-api-requests">
                {summary.apiRequestsToday.toLocaleString()}
              </div>
              <div className="mt-2">
                <Progress value={apiUsage} className="h-2" />
                <p className="text-xs text-gray-500 mt-1">
                  {apiUsage.toFixed(1)}% of daily limit
                </p>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Community Activity Metrics */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <Card className="border-0 shadow-sm bg-white">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <TrendingUp className="h-5 w-5 text-blue-600" />
                Community Activity
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <FileText className="h-4 w-4 text-gray-500" />
                  <span className="text-sm text-gray-600">Total Posts</span>
                </div>
                <span className="font-semibold text-gray-900" data-testid="metric-total-posts">
                  {summary.totalPosts.toLocaleString()}
                </span>
              </div>
              
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <Activity className="h-4 w-4 text-gray-500" />
                  <span className="text-sm text-gray-600">Total Comments</span>
                </div>
                <span className="font-semibold text-gray-900" data-testid="metric-total-comments">
                  {summary.totalComments.toLocaleString()}
                </span>
              </div>

              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <Users className="h-4 w-4 text-gray-500" />
                  <span className="text-sm text-gray-600">Engagement Rate</span>
                </div>
                <span className="font-semibold text-gray-900">
                  {summary.totalPosts > 0 ? (summary.totalComments / summary.totalPosts).toFixed(1) : 0} comments/post
                </span>
              </div>
            </CardContent>
          </Card>

          <Card className="border-0 shadow-sm bg-white">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Server className="h-5 w-5 text-green-600" />
                System Health
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <CheckCircle className="h-4 w-4 text-green-500" />
                  <span className="text-sm text-gray-600">Platform Status</span>
                </div>
                <Badge variant="secondary" className="bg-green-100 text-green-800">
                  Operational
                </Badge>
              </div>
              
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <HardDrive className="h-4 w-4 text-blue-500" />
                  <span className="text-sm text-gray-600">Database Status</span>
                </div>
                <Badge variant="secondary" className={`${databaseUsage > 90 ? 'bg-red-100 text-red-800' : 'bg-green-100 text-green-800'}`}>
                  {databaseUsage > 90 ? 'High Usage' : 'Healthy'}
                </Badge>
              </div>

              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <Activity className="h-4 w-4 text-purple-500" />
                  <span className="text-sm text-gray-600">API Health</span>
                </div>
                <Badge variant="secondary" className={`${apiUsage > 90 ? 'bg-yellow-100 text-yellow-800' : 'bg-green-100 text-green-800'}`}>
                  {apiUsage > 90 ? 'High Load' : 'Normal'}
                </Badge>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Usage Recommendations */}
        <Card className="border-0 shadow-sm bg-gradient-to-r from-blue-50 to-indigo-50">
          <CardHeader>
            <CardTitle className="text-gray-900">Usage Recommendations</CardTitle>
          </CardHeader>
          <CardContent className="space-y-3">
            {databaseUsage > 75 && (
              <div className="flex items-start gap-3 p-3 bg-yellow-50 rounded-lg border border-yellow-200">
                <AlertTriangle className="h-5 w-5 text-yellow-600 mt-0.5" />
                <div>
                  <h4 className="font-medium text-yellow-800">Database Storage Warning</h4>
                  <p className="text-sm text-yellow-700">
                    Database usage is at {databaseUsage.toFixed(1)}%. Consider archiving old data or upgrading your plan.
                  </p>
                </div>
              </div>
            )}

            {apiUsage > 75 && (
              <div className="flex items-start gap-3 p-3 bg-blue-50 rounded-lg border border-blue-200">
                <Zap className="h-5 w-5 text-blue-600 mt-0.5" />
                <div>
                  <h4 className="font-medium text-blue-800">High API Usage</h4>
                  <p className="text-sm text-blue-700">
                    API requests are at {apiUsage.toFixed(1)}% of daily limit. Consider optimizing client-side caching.
                  </p>
                </div>
              </div>
            )}

            {userUsage > 50 && (
              <div className="flex items-start gap-3 p-3 bg-green-50 rounded-lg border border-green-200">
                <TrendingUp className="h-5 w-5 text-green-600 mt-0.5" />
                <div>
                  <h4 className="font-medium text-green-800">Growing Community</h4>
                  <p className="text-sm text-green-700">
                    Your community is growing well! Consider preparing for scale at {userLimit.toLocaleString()} users.
                  </p>
                </div>
              </div>
            )}

            {databaseUsage < 50 && apiUsage < 50 && userUsage < 50 && (
              <div className="flex items-start gap-3 p-3 bg-green-50 rounded-lg border border-green-200">
                <CheckCircle className="h-5 w-5 text-green-600 mt-0.5" />
                <div>
                  <h4 className="font-medium text-green-800">System Running Optimally</h4>
                  <p className="text-sm text-green-700">
                    All systems are operating within normal parameters. Great job managing your platform!
                  </p>
                </div>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}