import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Checkbox } from "@/components/ui/checkbox";
import { useToast } from "@/hooks/use-toast";
import { Video, Calendar, Users, Clock, Trophy, Star, Play, ExternalLink, Download, BookOpen } from "lucide-react";
import { format, isAfter, differenceInMinutes } from "date-fns";
import type { LiveTrainingSession, TrainingRegistration } from "@shared/schema";
import InteractiveTrainingSession from "@/components/training/interactive-training-session";

export default function LiveTraining() {
  const [selectedCategory, setSelectedCategory] = useState<string>("all");
  const [showCreateDialog, setShowCreateDialog] = useState(false);
  const [activeTab, setActiveTab] = useState("upcoming");
  const [selectedSessionId, setSelectedSessionId] = useState<string | null>(null);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch live training sessions
  const { data: sessions = [], isLoading: sessionsLoading } = useQuery({
    queryKey: ["/api/live-training", selectedCategory],
    queryFn: () => apiRequest("GET", `/api/live-training?${selectedCategory !== 'all' ? `category=${selectedCategory}` : ''}`).then(res => res.json())
  });

  // Fetch upcoming sessions
  const { data: upcomingSessions = [], isLoading: upcomingLoading } = useQuery({
    queryKey: ["/api/live-training/upcoming"],
    queryFn: () => apiRequest("GET", "/api/live-training/upcoming").then(res => res.json())
  });

  // Current user
  const { data: user } = useQuery({
    queryKey: ["/api/user"],
    retry: false,
  });

  // Register for session mutation
  const registerMutation = useMutation({
    mutationFn: async (sessionId: string) => {
      const response = await apiRequest("POST", `/api/live-training/${sessionId}/register`, {
        userId: (user as any)?.id
      });
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Registration Successful",
        description: "You've been registered for the training session. Check your email for Zoom details.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/live-training"] });
    },
    onError: (error: any) => {
      toast({
        title: "Registration Failed",
        description: error.message || "Failed to register for the session.",
        variant: "destructive",
      });
    },
  });

  // Create instant meeting mutation
  const instantMeetingMutation = useMutation({
    mutationFn: async (data: { topic: string; agenda: string }) => {
      const response = await apiRequest("POST", "/api/live-training/instant", {
        ...data,
        instructorId: (user as any)?.id,
        instructorName: `${(user as any)?.firstName} ${(user as any)?.lastName}`.trim() || (user as any)?.username
      });
      return response.json();
    },
    onSuccess: (session) => {
      toast({
        title: "Instant Meeting Created",
        description: "Your live training session is ready. You can start immediately.",
      });
      // Open Zoom start URL in new tab
      if (session.zoomStartUrl) {
        window.open(session.zoomStartUrl, '_blank');
      }
      queryClient.invalidateQueries({ queryKey: ["/api/live-training"] });
    },
    onError: (error: any) => {
      toast({
        title: "Meeting Creation Failed",
        description: error.message || "Failed to create instant meeting.",
        variant: "destructive",
      });
    },
  });

  const getSessionStatusColor = (session: LiveTrainingSession) => {
    const now = new Date();
    const startTime = new Date(session.scheduledStartTime);
    const endTime = new Date(session.scheduledEndTime);

    if (session.status === 'live' || (isAfter(now, startTime) && isAfter(endTime, now))) {
      return "bg-green-500";
    } else if (isAfter(startTime, now)) {
      return "bg-blue-500";
    } else {
      return "bg-gray-500";
    }
  };

  const getSessionStatusText = (session: LiveTrainingSession) => {
    const now = new Date();
    const startTime = new Date(session.scheduledStartTime);
    const endTime = new Date(session.scheduledEndTime);

    if (session.status === 'completed') {
      return 'Completed';
    } else if (session.status === 'cancelled') {
      return 'Cancelled';
    } else if (session.status === 'live' || (isAfter(now, startTime) && isAfter(endTime, now))) {
      return 'Live Now';
    } else if (isAfter(startTime, now)) {
      const minutesUntil = differenceInMinutes(startTime, now);
      if (minutesUntil < 60) {
        return `Starting in ${minutesUntil}m`;
      }
      return 'Upcoming';
    } else {
      return 'Ended';
    }
  };

  const canJoinSession = (session: LiveTrainingSession) => {
    const now = new Date();
    const startTime = new Date(session.scheduledStartTime);
    const endTime = new Date(session.scheduledEndTime);
    
    return session.status === 'live' || (isAfter(now, new Date(startTime.getTime() - 15 * 60 * 1000)) && isAfter(endTime, now));
  };

  const SessionCard = ({ session }: { session: LiveTrainingSession }) => (
    <Card key={session.id} className="relative overflow-hidden" data-testid={`session-card-${session.id}`}>
      <div className={`absolute top-0 left-0 w-full h-1 ${getSessionStatusColor(session)}`} />
      
      <CardHeader className="pb-3">
        <div className="flex items-start justify-between">
          <div className="flex-1">
            <CardTitle className="text-lg mb-2" data-testid={`session-title-${session.id}`}>{session.title}</CardTitle>
            <CardDescription className="text-sm mb-3" data-testid={`session-description-${session.id}`}>
              {session.description}
            </CardDescription>
          </div>
          <Badge 
            variant="secondary" 
            className={`${getSessionStatusColor(session)} text-white ml-2`}
            data-testid={`session-status-${session.id}`}
          >
            {getSessionStatusText(session)}
          </Badge>
        </div>

        <div className="flex flex-wrap gap-2 text-sm text-muted-foreground">
          <div className="flex items-center gap-1" data-testid={`session-category-${session.id}`}>
            <BookOpen className="w-4 h-4" />
            <span className="capitalize">{session.category}</span>
          </div>
          
          <div className="flex items-center gap-1" data-testid={`session-type-${session.id}`}>
            <Video className="w-4 h-4" />
            <span className="capitalize">{session.sessionType}</span>
          </div>

          <div className="flex items-center gap-1" data-testid={`session-duration-${session.id}`}>
            <Clock className="w-4 h-4" />
            <span>{session.duration} minutes</span>
          </div>

          {session.maxParticipants && (
            <div className="flex items-center gap-1" data-testid={`session-capacity-${session.id}`}>
              <Users className="w-4 h-4" />
              <span>Max {session.maxParticipants}</span>
            </div>
          )}
        </div>
      </CardHeader>

      <CardContent>
        <div className="space-y-4">
          <div className="text-sm">
            <div className="font-medium mb-1">Schedule</div>
            <div className="text-muted-foreground" data-testid={`session-schedule-${session.id}`}>
              {format(new Date(session.scheduledStartTime), "PPP 'at' p")}
            </div>
            <div className="text-muted-foreground">
              Instructor: <span className="text-foreground">{session.instructorName}</span>
            </div>
          </div>

          {session.learningObjectives && session.learningObjectives.length > 0 && (
            <div className="text-sm">
              <div className="font-medium mb-2">Learning Objectives</div>
              <ul className="list-disc list-inside space-y-1 text-muted-foreground">
                {session.learningObjectives.map((objective, index) => (
                  <li key={index}>{objective}</li>
                ))}
              </ul>
            </div>
          )}

          <div className="flex gap-2 pt-2">
            {canJoinSession(session) && session.zoomJoinUrl && (
              <Button 
                size="sm" 
                className="flex items-center gap-2"
                onClick={() => window.open(session.zoomJoinUrl!, '_blank')}
                data-testid={`join-session-${session.id}`}
              >
                <ExternalLink className="w-4 h-4" />
                Join Session
              </Button>
            )}

            {!canJoinSession(session) && (
              <Button 
                size="sm" 
                variant="outline"
                onClick={() => registerMutation.mutate(session.id)}
                disabled={registerMutation.isPending}
                data-testid={`register-session-${session.id}`}
              >
                {registerMutation.isPending ? "Registering..." : "Register"}
              </Button>
            )}

            {session.recordingUrl && (
              <Button 
                size="sm" 
                variant="outline"
                onClick={() => window.open(session.recordingUrl!, '_blank')}
                data-testid={`view-recording-${session.id}`}
              >
                <Play className="w-4 h-4 mr-1" />
                Recording
              </Button>
            )}
          </div>

          {session.isPremiumOnly && (
            <Badge variant="outline" className="text-amber-600 border-amber-200 bg-amber-50">
              <Trophy className="w-3 h-3 mr-1" />
              Premium Only
            </Badge>
          )}
        </div>
      </CardContent>
    </Card>
  );

  const InstantMeetingDialog = () => {
    const [topic, setTopic] = useState("");
    const [agenda, setAgenda] = useState("");

    const handleCreate = () => {
      if (!topic.trim()) {
        toast({
          title: "Topic Required",
          description: "Please enter a topic for the training session.",
          variant: "destructive",
        });
        return;
      }

      instantMeetingMutation.mutate({
        topic: topic.trim(),
        agenda: agenda.trim() || undefined
      });
    };

    return (
      <Dialog>
        <DialogTrigger asChild>
          <Button className="flex items-center gap-2" data-testid="create-instant-meeting">
            <Video className="w-4 h-4" />
            Start Instant Session
          </Button>
        </DialogTrigger>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Create Instant Training Session</DialogTitle>
            <DialogDescription>
              Start a live training session immediately with Zoom integration.
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            <div>
              <Label htmlFor="topic">Session Topic</Label>
              <Input
                id="topic"
                value={topic}
                onChange={(e) => setTopic(e.target.value)}
                placeholder="Enter session topic..."
                data-testid="instant-session-topic"
              />
            </div>

            <div>
              <Label htmlFor="agenda">Agenda (Optional)</Label>
              <Textarea
                id="agenda"
                value={agenda}
                onChange={(e) => setAgenda(e.target.value)}
                placeholder="What will you cover in this session?"
                rows={3}
                data-testid="instant-session-agenda"
              />
            </div>

            <div className="flex justify-end gap-2">
              <Button
                onClick={handleCreate}
                disabled={instantMeetingMutation.isPending}
                data-testid="create-instant-session"
              >
                {instantMeetingMutation.isPending ? "Creating..." : "Start Session"}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    );
  };

  if (sessionsLoading || upcomingLoading) {
    return (
      <div className="container mx-auto px-4 py-8 space-y-8">
        <div className="text-center space-y-4">
          <div className="h-8 bg-gray-200 rounded w-1/3 mx-auto animate-pulse" />
          <div className="h-4 bg-gray-200 rounded w-1/2 mx-auto animate-pulse" />
        </div>
        <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
          {Array.from({ length: 6 }).map((_, i) => (
            <div key={i} className="h-64 bg-gray-200 rounded-lg animate-pulse" />
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8 space-y-8">
      {/* Header */}
      <div className="text-center space-y-4">
        <h1 className="text-4xl font-bold tracking-tight" data-testid="page-title">
          Live Training Sessions
        </h1>
        <p className="text-xl text-muted-foreground max-w-2xl mx-auto" data-testid="page-description">
          Join interactive training sessions powered by Zoom. Learn from experts, 
          engage with peers, and advance your professional skills.
        </p>
      </div>

      {/* Quick Actions */}
      <div className="flex flex-col sm:flex-row gap-4 justify-center">
        <InstantMeetingDialog />
        <Button variant="outline" className="flex items-center gap-2" data-testid="schedule-session">
          <Calendar className="w-4 h-4" />
          Schedule Session
        </Button>
      </div>

      {/* Filters and Tabs */}
      <div className="flex flex-col sm:flex-row gap-4 items-center justify-between">
        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full sm:w-auto">
          <TabsList>
            <TabsTrigger value="upcoming" data-testid="tab-upcoming">Upcoming</TabsTrigger>
            <TabsTrigger value="all" data-testid="tab-all">All Sessions</TabsTrigger>
            <TabsTrigger value="interactive" data-testid="tab-interactive">Interactive</TabsTrigger>
            <TabsTrigger value="recorded" data-testid="tab-recorded">Recordings</TabsTrigger>
          </TabsList>
        </Tabs>

        <Select value={selectedCategory} onValueChange={setSelectedCategory}>
          <SelectTrigger className="w-[180px]" data-testid="category-filter">
            <SelectValue placeholder="Filter by category" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">All Categories</SelectItem>
            <SelectItem value="compliance">Compliance</SelectItem>
            <SelectItem value="tax">Tax Preparation</SelectItem>
            <SelectItem value="hr">HR Management</SelectItem>
            <SelectItem value="accounting">Accounting</SelectItem>
            <SelectItem value="business">Business Development</SelectItem>
          </SelectContent>
        </Select>
      </div>

      {/* Session Content */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsContent value="upcoming" className="space-y-6">
          {upcomingSessions.length === 0 ? (
            <Card>
              <CardContent className="text-center py-12">
                <Video className="w-12 h-12 mx-auto text-muted-foreground mb-4" />
                <h3 className="text-lg font-semibold mb-2">No Upcoming Sessions</h3>
                <p className="text-muted-foreground">
                  There are no scheduled training sessions at the moment. Check back soon!
                </p>
              </CardContent>
            </Card>
          ) : (
            <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3" data-testid="upcoming-sessions-grid">
              {upcomingSessions.map((session: LiveTrainingSession) => (
                <SessionCard key={session.id} session={session} />
              ))}
            </div>
          )}
        </TabsContent>

        <TabsContent value="all" className="space-y-6">
          {sessions.length === 0 ? (
            <Card>
              <CardContent className="text-center py-12">
                <Video className="w-12 h-12 mx-auto text-muted-foreground mb-4" />
                <h3 className="text-lg font-semibold mb-2">No Training Sessions</h3>
                <p className="text-muted-foreground">
                  No training sessions found for the selected category.
                </p>
              </CardContent>
            </Card>
          ) : (
            <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3" data-testid="all-sessions-grid">
              {sessions.map((session: LiveTrainingSession) => (
                <SessionCard key={session.id} session={session} />
              ))}
            </div>
          )}
        </TabsContent>

        <TabsContent value="recorded" className="space-y-6">
          {sessions.filter((s: any) => s.recordingUrl).length === 0 ? (
            <Card>
              <CardContent className="text-center py-12">
                <Play className="w-12 h-12 mx-auto text-muted-foreground mb-4" />
                <h3 className="text-lg font-semibold mb-2">No Recordings Available</h3>
                <p className="text-muted-foreground">
                  Session recordings will appear here after completion.
                </p>
              </CardContent>
            </Card>
          ) : (
            <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3" data-testid="recorded-sessions-grid">
              {sessions
                .filter((s: any) => s.recordingUrl)
                .map((session: LiveTrainingSession) => (
                  <SessionCard key={session.id} session={session} />
                ))}
            </div>
          )}
        </TabsContent>

        <TabsContent value="interactive" className="space-y-6">
          {selectedSessionId ? (
            <InteractiveTrainingSession 
              sessionId={selectedSessionId}
              isInstructor={true} // This would be determined by user role
              currentUserId={(user as any)?.id || ''}
            />
          ) : (
            <Card>
              <CardContent className="text-center py-12">
                <Video className="w-12 h-12 mx-auto text-muted-foreground mb-4" />
                <h3 className="text-lg font-semibold mb-2">Interactive Training Session</h3>
                <p className="text-muted-foreground mb-6">
                  Select a live session to access interactive features like polls, Q&A, and breakout rooms
                </p>
                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                  {sessions
                    .filter((session: any) => session.status === 'live')
                    .map((session: LiveTrainingSession) => (
                      <Card 
                        key={session.id} 
                        className="cursor-pointer hover:bg-accent"
                        onClick={() => setSelectedSessionId(session.id)}
                        data-testid={`interactive-session-${session.id}`}
                      >
                        <CardContent className="p-4">
                          <h4 className="font-medium">{session.title}</h4>
                          <p className="text-sm text-muted-foreground">
                            {session.instructorName}
                          </p>
                          <Badge className="mt-2">Live Now</Badge>
                        </CardContent>
                      </Card>
                    ))}
                </div>
                {sessions.filter((session: any) => session.status === 'live').length === 0 && (
                  <p className="text-sm text-muted-foreground mt-4">
                    No live sessions available for interactive features
                  </p>
                )}
              </CardContent>
            </Card>
          )}
        </TabsContent>
      </Tabs>

      {/* Features Banner */}
      <Card className="bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-blue-950 dark:to-indigo-950">
        <CardContent className="text-center py-8">
          <h3 className="text-2xl font-bold mb-4">Enhanced Learning Experience</h3>
          <div className="grid md:grid-cols-3 gap-6 text-sm">
            <div className="space-y-2">
              <Video className="w-8 h-8 mx-auto text-blue-600" />
              <h4 className="font-semibold">HD Video & Audio</h4>
              <p className="text-muted-foreground">Crystal clear Zoom sessions with recording capabilities</p>
            </div>
            <div className="space-y-2">
              <Users className="w-8 h-8 mx-auto text-green-600" />
              <h4 className="font-semibold">Interactive Learning</h4>
              <p className="text-muted-foreground">Chat, polls, breakout rooms, and Q&A sessions</p>
            </div>
            <div className="space-y-2">
              <Download className="w-8 h-8 mx-auto text-purple-600" />
              <h4 className="font-semibold">Resource Sharing</h4>
              <p className="text-muted-foreground">Access slides, documents, and materials instantly</p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}