import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import ServiceCard from "@/components/services/service-card";
import ComplianceBadges from "@/components/compliance/compliance-badges";
import SAMProgressTracker from "@/components/sam/sam-progress-tracker";
import { useLocation } from "wouter";
import logoBackground from "@assets/images/logo-primary.jpg";
import { 
  Calculator, 
  Users, 
  TrendingUp, 
  Search, 
  Rocket, 
  FileText,
  Shield,
  File,
  Gavel,
  CalendarCheck
} from "lucide-react";

const services = [
  {
    icon: Calculator,
    title: "Bookkeeping & Accounting",
    description: "Complete financial record keeping, monthly reconciliation, expense tracking, and comprehensive financial reporting for businesses.",
    link: "/services#bookkeeping"
  },
  {
    icon: FileText,
    title: "Tax Preparation & Planning",
    description: "Individual and business tax preparation, strategic tax planning, quarterly filings, and year-round compliance support.",
    link: "/services#tax"
  },
  {
    icon: Users,
    title: "Payroll & HR Services",
    description: "Complete payroll processing, benefits administration, HR compliance, and employee management solutions.",
    link: "/services#hr"
  },
  {
    icon: TrendingUp,
    title: "CFO & Financial Strategy",
    description: "Strategic financial leadership, cash flow management, budgeting, forecasting, and financial planning expertise.",
    link: "/services#cfo"
  },
  {
    icon: Rocket,
    title: "Business Development",
    description: "Growth strategy, market analysis, business plan development, and expansion planning for sustainable business growth.",
    link: "/services#business-development"
  },
  {
    icon: Shield,
    title: "Government Contracting",
    description: "S.A.M. registration, VPAT documentation, compliance support, and specialized assistance for federal contracting.",
    link: "/government-contracting"
  }
];

export default function Home() {
  const [, navigate] = useLocation();

  const handleGetStarted = () => {
    navigate("/contact");
  };

  const handleSAMConsultation = () => {
    navigate("/government-contracting");
  };

  const handleGovContractingConsult = () => {
    navigate("/contact");
  };

  return (
    <div className="flex flex-col min-h-screen">
      {/* Hero Section */}
      <section className="bg-hero-gradient py-20 lg:py-32 relative overflow-hidden" aria-labelledby="hero-heading">
        {/* Background Logo */}
        <div 
          className="absolute inset-0 flex items-center justify-center pointer-events-none"
          style={{
            backgroundImage: `url(${logoBackground})`,
            backgroundSize: '600px 600px',
            backgroundPosition: '75% center',
            backgroundRepeat: 'no-repeat',
            opacity: 0.25,
            zIndex: 0
          }}
        />
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div className="animate-fade-in">
              <h1 id="hero-heading" className="text-4xl lg:text-6xl font-bold text-foreground mb-6">
                Professional Accounting & 
                <span className="text-primary"> Business Solutions</span> 
                Services
              </h1>
              <p className="text-xl text-muted-foreground mb-8 leading-relaxed">
                Full-service accounting, tax preparation, and payroll solutions with business development expertise. 
                Specialized government contracting support including S.A.M. registration and Section 508 compliance.
              </p>
              <div className="flex flex-col sm:flex-row gap-4 mb-8">
                <Button 
                  onClick={handleGetStarted}
                  className="btn-primary"
                  size="lg"
                  data-testid="button-get-started"
                >
                  <Rocket className="mr-2 h-5 w-5" aria-hidden="true" />
                  Get Started Today
                </Button>
                <Button 
                  onClick={handleSAMConsultation}
                  variant="outline"
                  className="btn-secondary"
                  size="lg"
                  data-testid="button-sam-consultation"
                >
                  <Shield className="mr-2 h-5 w-5" aria-hidden="true" />
                  S.A.M. Registration Help
                </Button>
              </div>
              
              <ComplianceBadges />
            </div>
            
            <div className="lg:pl-8">
              <img 
                src="https://images.unsplash.com/photo-1560472354-b33ff0c44a43?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=1000&h=800" 
                alt="Professional diverse business team in a modern conference room discussing financial documents and compliance materials" 
                className="rounded-xl shadow-2xl w-full h-auto"
                loading="eager"
              />
            </div>
          </div>
        </div>
      </section>

      {/* SAM Registration Process Section */}
      <section className="py-20 bg-muted/20" aria-labelledby="sam-process-heading">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <SAMProgressTracker onGetStarted={handleSAMConsultation} />
        </div>
      </section>

      {/* Services Overview */}
      <section id="services" className="py-20 bg-muted/30" aria-labelledby="services-heading">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 id="services-heading" className="text-3xl lg:text-4xl font-bold text-foreground mb-4">
              Comprehensive Financial Services
            </h2>
            <p className="text-xl text-muted-foreground max-w-3xl mx-auto">
              From bookkeeping to complex government compliance, we provide full-spectrum 
              accounting and business development services tailored to your needs.
            </p>
          </div>
          
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {services.map((service, index) => (
              <ServiceCard 
                key={service.title}
                icon={service.icon}
                title={service.title}
                description={service.description}
                link={service.link}
                index={index}
              />
            ))}
          </div>
        </div>
      </section>

      {/* Government Contracting Section */}
      <section id="government-contracting" className="py-20 bg-background" aria-labelledby="gov-contracting-heading">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 id="gov-contracting-heading" className="text-3xl lg:text-4xl font-bold text-foreground mb-6">
                Government Contracting
                <span className="text-primary"> Expertise</span>
              </h2>
              <p className="text-xl text-muted-foreground mb-8">
                Navigate the complex world of federal contracting with confidence. 
                Our specialized team ensures your business meets all compliance requirements 
                and maximizes opportunities in the government marketplace.
              </p>
              
              <div className="space-y-6">
                <div className="flex items-start">
                  <div className="w-8 h-8 bg-primary/10 rounded-lg flex items-center justify-center mt-1 mr-4">
                    <Shield className="h-4 w-4 text-primary" aria-hidden="true" />
                  </div>
                  <div>
                    <h3 className="text-lg font-semibold text-foreground mb-2">S.A.M. Registration & Maintenance</h3>
                    <p className="text-muted-foreground">
                      Complete System for Award Management registration assistance, 
                      annual renewals, and ongoing compliance monitoring.
                    </p>
                  </div>
                </div>
                
                <div className="flex items-start">
                  <div className="w-8 h-8 bg-primary/10 rounded-lg flex items-center justify-center mt-1 mr-4">
                    <File className="h-4 w-4 text-primary" aria-hidden="true" />
                  </div>
                  <div>
                    <h3 className="text-lg font-semibold text-foreground mb-2">VPAT Documentation</h3>
                    <p className="text-muted-foreground">
                      Voluntary Product Accessibility Template preparation ensuring 
                      your products meet Section 508 and WCAG 2.1 AA standards.
                    </p>
                  </div>
                </div>
                
                <div className="flex items-start">
                  <div className="w-8 h-8 bg-primary/10 rounded-lg flex items-center justify-center mt-1 mr-4">
                    <Gavel className="h-4 w-4 text-primary" aria-hidden="true" />
                  </div>
                  <div>
                    <h3 className="text-lg font-semibold text-foreground mb-2">Compliance Documentation</h3>
                    <p className="text-muted-foreground">
                      Complete compliance documentation package including FAR, 
                      DFARS, and agency-specific requirements preparation.
                    </p>
                  </div>
                </div>
              </div>
              
              <div className="mt-8">
                <Button 
                  onClick={handleGovContractingConsult}
                  className="btn-primary"
                  size="lg"
                  data-testid="button-sam-consultation-main"
                >
                  <CalendarCheck className="mr-2 h-5 w-5" aria-hidden="true" />
                  Schedule SAM Consultation
                </Button>
              </div>
            </div>
            
            <div>
              {/* TODO: Replace this with your office photo */}
              {/* Step 1: Add your office photo to attached_assets/images/office/front-office.jpg */}
              {/* Step 2: Import it at the top: import officeImage from "@assets/images/office/front-office.jpg" */}
              {/* Step 3: Replace src={officeImage} below and update alt text */}
              
              <img 
                src="https://images.unsplash.com/photo-1493612276216-ee3925520721?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=1000&h=800" 
                alt="Modern government building with American flag representing federal contracting opportunities and compliance requirements" 
                className="rounded-xl shadow-lg w-full h-auto"
                data-testid="office-hero-image"
              />
              
              {/* When you're ready to use your own image, replace the above with: */}
              {/* <img 
                src={officeImage} 
                alt="TMH Global LLC office building" 
                className="rounded-xl shadow-lg w-full h-auto"
                data-testid="office-hero-image"
              /> */}
            </div>
          </div>
          
          {/* S.A.M. Registration Process */}
          <div className="mt-20 bg-muted/50 rounded-2xl p-8 lg:p-12">
            <h3 className="text-2xl lg:text-3xl font-bold text-foreground mb-8 text-center">
              S.A.M. Registration Process
            </h3>
            <div className="grid md:grid-cols-4 gap-8">
              {[
                {
                  step: "1",
                  title: "Initial Assessment",
                  description: "Review business structure and gather required documentation"
                },
                {
                  step: "2",
                  title: "Registration Submission",
                  description: "Complete SAM.gov registration with accurate business information"
                },
                {
                  step: "3",
                  title: "Compliance Verification",
                  description: "Ensure all representations and certifications are complete"
                },
                {
                  step: "4",
                  title: "Ongoing Maintenance",
                  description: "Annual renewals and updates to maintain active status"
                }
              ].map((item, index) => (
                <div key={item.step} className="text-center">
                  <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
                    <span className="text-2xl font-bold text-primary">{item.step}</span>
                  </div>
                  <h4 className="text-lg font-semibold text-foreground mb-2">{item.title}</h4>
                  <p className="text-muted-foreground">
                    {item.description}
                  </p>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Call to Action */}
      <section className="py-20 bg-primary/5" aria-labelledby="cta-heading">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <h2 id="cta-heading" className="text-3xl lg:text-4xl font-bold text-foreground mb-4">
            Ready to Get Started?
          </h2>
          <p className="text-xl text-muted-foreground mb-8 max-w-3xl mx-auto">
            Contact us today for a consultation. We'll help you navigate 
            government contracting requirements and optimize your financial operations.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button 
              onClick={() => navigate("/contact")}
              className="btn-primary"
              size="lg"
              data-testid="button-contact-cta"
            >
              Schedule Free Consultation
            </Button>
            <Button 
              onClick={() => navigate("/client-portal")}
              variant="outline"
              className="btn-secondary"
              size="lg"
              data-testid="button-portal-cta"
            >
              Access Client Portal
            </Button>
          </div>
        </div>
      </section>
    </div>
  );
}
