import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { useLocation } from "wouter";
import { 
  Shield, 
  File, 
  Gavel, 
  CheckCircle, 
  AlertTriangle,
  ExternalLink,
  CalendarCheck,
  Clock,
  Users,
  Award
} from "lucide-react";

const complianceRequirements = [
  {
    title: "SAM.gov Registration",
    description: "Required for all federal contracting opportunities",
    urgency: "Critical",
    deadline: "Before bid submission"
  },
  {
    title: "VPAT Documentation",
    description: "Accessibility conformance reporting for ICT products",
    urgency: "High",
    deadline: "With proposal submission"
  },
  {
    title: "Section 508 Compliance",
    description: "Accessibility standards for federal ICT",
    urgency: "Critical",
    deadline: "Product delivery"
  },
  {
    title: "WCAG 2.1 AA Standards",
    description: "Web Content Accessibility Guidelines compliance",
    urgency: "High",
    deadline: "Product delivery"
  }
];

const samRegistrationSteps = [
  {
    step: 1,
    title: "Initial Assessment",
    description: "Review business structure and gather required documentation including DUNS number, tax information, and business licenses.",
    timeframe: "1-2 business days"
  },
  {
    step: 2,
    title: "Registration Submission",
    description: "Complete SAM.gov registration with accurate business information, banking details, and points of contact.",
    timeframe: "3-5 business days"
  },
  {
    step: 3,
    title: "Compliance Verification",
    description: "Ensure all representations and certifications are complete and accurate for your business type.",
    timeframe: "2-3 business days"
  },
  {
    step: 4,
    title: "Ongoing Maintenance",
    description: "Annual renewals and updates to maintain active status and compliance with changing requirements.",
    timeframe: "Ongoing annually"
  }
];

export default function GovernmentContracting() {
  const [, navigate] = useLocation();

  const handleScheduleConsultation = () => {
    navigate("/contact");
  };

  const handleSAMRegistration = () => {
    navigate("/contact");
  };

  return (
    <div className="py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <h1 className="text-4xl lg:text-5xl font-bold text-foreground mb-6">
            Government Contracting Services
          </h1>
          <p className="text-xl text-muted-foreground max-w-3xl mx-auto mb-8">
            Navigate federal contracting requirements with confidence. Our specialized expertise 
            ensures compliance with SAM.gov, Section 508, and WCAG 2.1 AA standards.
          </p>
          <div className="flex flex-wrap justify-center gap-4">
            <Badge variant="secondary" className="compliance-badge">
              <Shield className="w-4 h-4 mr-2" aria-hidden="true" />
              SAM.gov Certified
            </Badge>
            <Badge variant="secondary" className="compliance-badge">
              <CheckCircle className="w-4 h-4 mr-2" aria-hidden="true" />
              Section 508 Expert
            </Badge>
            <Badge variant="secondary" className="compliance-badge">
              <Award className="w-4 h-4 mr-2" aria-hidden="true" />
              WCAG 2.1 AA Specialist
            </Badge>
          </div>
        </div>

        {/* Key Services */}
        <div className="grid lg:grid-cols-3 gap-8 mb-20">
          <Card className="service-card">
            <CardHeader>
              <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mb-4">
                <Shield className="h-6 w-6 text-primary" aria-hidden="true" />
              </div>
              <CardTitle>SAM.gov Registration</CardTitle>
              <CardDescription>
                Complete registration and maintenance services for the System for Award Management
              </CardDescription>
            </CardHeader>
            <CardContent>
              <ul className="space-y-2 mb-6">
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  Initial registration assistance
                </li>
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  Annual renewal management
                </li>
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  Compliance monitoring
                </li>
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  Documentation support
                </li>
              </ul>
              <Button 
                onClick={handleSAMRegistration}
                className="w-full btn-primary"
                data-testid="button-sam-registration"
              >
                Start SAM Registration
              </Button>
            </CardContent>
          </Card>

          <Card className="service-card">
            <CardHeader>
              <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mb-4">
                <File className="h-6 w-6 text-primary" aria-hidden="true" />
              </div>
              <CardTitle>VPAT Documentation</CardTitle>
              <CardDescription>
                Voluntary Product Accessibility Template preparation and certification
              </CardDescription>
            </CardHeader>
            <CardContent>
              <ul className="space-y-2 mb-6">
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  Section 508 evaluation
                </li>
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  WCAG 2.1 AA assessment
                </li>
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  ACR report preparation
                </li>
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  Remediation guidance
                </li>
              </ul>
              <Button 
                onClick={handleScheduleConsultation}
                variant="outline"
                className="w-full btn-secondary"
                data-testid="button-vpat-consultation"
              >
                Get VPAT Assessment
              </Button>
            </CardContent>
          </Card>

          <Card className="service-card">
            <CardHeader>
              <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center mb-4">
                <Gavel className="h-6 w-6 text-primary" aria-hidden="true" />
              </div>
              <CardTitle>Compliance Support</CardTitle>
              <CardDescription>
                Ongoing support for federal compliance requirements and regulations
              </CardDescription>
            </CardHeader>
            <CardContent>
              <ul className="space-y-2 mb-6">
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  FAR/DFARS compliance
                </li>
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  Agency-specific requirements
                </li>
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  Proposal preparation support
                </li>
                <li className="flex items-center text-sm">
                  <CheckCircle className="h-4 w-4 text-accent mr-2" aria-hidden="true" />
                  Contract management
                </li>
              </ul>
              <Button 
                onClick={handleScheduleConsultation}
                variant="outline"
                className="w-full btn-secondary"
                data-testid="button-compliance-consultation"
              >
                Discuss Requirements
              </Button>
            </CardContent>
          </Card>
        </div>

        {/* Compliance Requirements */}
        <div className="mb-20">
          <h2 className="text-3xl font-bold text-foreground mb-8 text-center">
            Key Compliance Requirements
          </h2>
          <div className="grid md:grid-cols-2 gap-6">
            {complianceRequirements.map((requirement, index) => (
              <Card key={index} className="border-l-4 border-l-primary">
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <CardTitle className="text-lg">{requirement.title}</CardTitle>
                    <Badge 
                      variant={requirement.urgency === "Critical" ? "destructive" : "secondary"}
                      className="ml-2"
                    >
                      {requirement.urgency}
                    </Badge>
                  </div>
                  <CardDescription>{requirement.description}</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="flex items-center text-sm text-muted-foreground">
                    <Clock className="h-4 w-4 mr-2" aria-hidden="true" />
                    Deadline: {requirement.deadline}
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* SAM Registration Process */}
        <div className="mb-20 bg-muted/30 rounded-2xl p-8 lg:p-12">
          <h2 className="text-3xl font-bold text-foreground mb-8 text-center">
            SAM.gov Registration Process
          </h2>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8">
            {samRegistrationSteps.map((step, index) => (
              <div key={step.step} className="text-center">
                <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
                  <span className="text-2xl font-bold text-primary">{step.step}</span>
                </div>
                <h3 className="text-lg font-semibold text-foreground mb-2">{step.title}</h3>
                <p className="text-muted-foreground mb-4 text-sm">
                  {step.description}
                </p>
                <div className="inline-flex items-center text-xs text-accent bg-accent/10 px-2 py-1 rounded-full">
                  <Clock className="h-3 w-3 mr-1" aria-hidden="true" />
                  {step.timeframe}
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* 2025 Updates */}
        <div className="mb-20">
          <Card className="border-l-4 border-l-accent bg-accent/5">
            <CardHeader>
              <div className="flex items-center">
                <AlertTriangle className="h-6 w-6 text-accent mr-3" aria-hidden="true" />
                <CardTitle className="text-xl">2025 Compliance Updates</CardTitle>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <h4 className="font-semibold text-foreground mb-2">New SAM.gov Requirements (Effective November 2024)</h4>
                  <p className="text-muted-foreground text-sm">
                    Enhanced pre-award registration requirements clarify that offerors must be registered 
                    at the point of offer submission, not just at contract award.
                  </p>
                </div>
                <div>
                  <h4 className="font-semibold text-foreground mb-2">WCAG 2.2 Migration</h4>
                  <p className="text-muted-foreground text-sm">
                    Federal agencies are beginning to require WCAG 2.2 compliance in addition to 
                    Section 508 standards. We help ensure your products meet the latest requirements.
                  </p>
                </div>
                <div>
                  <h4 className="font-semibold text-foreground mb-2">Enhanced Documentation Requirements</h4>
                  <p className="text-muted-foreground text-sm">
                    New testing methodologies and VPAT requirements based on the DHS Trusted Tester 
                    program and harmonized testing processes.
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Resources */}
        <div className="mb-20">
          <h2 className="text-3xl font-bold text-foreground mb-8 text-center">
            Government Contracting Resources
          </h2>
          <div className="grid md:grid-cols-3 gap-6">
            <Card className="service-card">
              <CardHeader>
                <CardTitle className="text-lg">SAM.gov Portal</CardTitle>
                <CardDescription>Official government registration system</CardDescription>
              </CardHeader>
              <CardContent>
                <Button 
                  variant="outline" 
                  className="w-full"
                  onClick={() => window.open('https://sam.gov', '_blank')}
                  data-testid="link-sam-gov"
                >
                  Visit SAM.gov
                  <ExternalLink className="ml-2 h-4 w-4" aria-hidden="true" />
                </Button>
              </CardContent>
            </Card>

            <Card className="service-card">
              <CardHeader>
                <CardTitle className="text-lg">Section 508 Guidelines</CardTitle>
                <CardDescription>Official accessibility standards</CardDescription>
              </CardHeader>
              <CardContent>
                <Button 
                  variant="outline" 
                  className="w-full"
                  onClick={() => window.open('https://section508.gov', '_blank')}
                  data-testid="link-section508"
                >
                  Visit Section508.gov
                  <ExternalLink className="ml-2 h-4 w-4" aria-hidden="true" />
                </Button>
              </CardContent>
            </Card>

            <Card className="service-card">
              <CardHeader>
                <CardTitle className="text-lg">WCAG Guidelines</CardTitle>
                <CardDescription>Web accessibility standards</CardDescription>
              </CardHeader>
              <CardContent>
                <Button 
                  variant="outline" 
                  className="w-full"
                  onClick={() => window.open('https://www.w3.org/WAI/WCAG21/quickref/', '_blank')}
                  data-testid="link-wcag"
                >
                  View WCAG 2.1
                  <ExternalLink className="ml-2 h-4 w-4" aria-hidden="true" />
                </Button>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Call to Action */}
        <div className="text-center bg-primary/5 rounded-2xl p-12">
          <h2 className="text-3xl font-bold text-foreground mb-4">
            Ready to Start Your Government Contracting Journey?
          </h2>
          <p className="text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Our experts are ready to help you navigate the complex requirements and 
            position your business for federal contracting success.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button 
              onClick={handleScheduleConsultation}
              className="btn-primary"
              size="lg"
              data-testid="button-schedule-consultation-cta"
            >
              <CalendarCheck className="mr-2 h-5 w-5" aria-hidden="true" />
              Schedule Free Consultation
            </Button>
            <Button 
              onClick={handleSAMRegistration}
              variant="outline"
              className="btn-secondary"
              size="lg"
              data-testid="button-sam-registration-cta"
            >
              <Shield className="mr-2 h-5 w-5" aria-hidden="true" />
              Start SAM Registration
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
}
