import { useState } from "react";
import ContactForm from "@/components/forms/contact-form";
import ConsultationForm from "@/components/forms/consultation-form";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { 
  Phone, 
  Mail, 
  MapPin, 
  Clock, 
  Calendar,
  AlertTriangle,
  MessageSquare,
  Users
} from "lucide-react";

export default function Contact() {
  const [activeTab, setActiveTab] = useState("contact");
  
  const handleEmergencyContact = () => {
    // In a real application, this would open an emergency contact form or call
    window.location.href = "tel:+17038297277";
  };

  return (
    <div className="py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <h1 className="text-4xl lg:text-5xl font-bold text-foreground mb-6">
            Contact Us
          </h1>
          <p className="text-xl text-muted-foreground max-w-3xl mx-auto">
            Ready to get started? Contact us today for a consultation. We'll help you navigate 
            government contracting requirements and optimize your financial operations.
          </p>
        </div>

        <div className="grid lg:grid-cols-2 gap-12">
          {/* Contact Forms */}
          <div>
            <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
              <TabsList className="grid w-full grid-cols-2">
                <TabsTrigger value="contact" data-testid="tab-contact">Send Message</TabsTrigger>
                <TabsTrigger value="consultation" data-testid="tab-consultation">Schedule Consultation</TabsTrigger>
              </TabsList>
              
              <TabsContent value="contact" className="mt-6">
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center">
                      <MessageSquare className="mr-3 h-5 w-5 text-primary" aria-hidden="true" />
                      Send us a Message
                    </CardTitle>
                    <CardDescription>
                      Get in touch with our team for general inquiries or questions about our services.
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <ContactForm />
                  </CardContent>
                </Card>
              </TabsContent>
              
              <TabsContent value="consultation" className="mt-6">
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center">
                      <Calendar className="mr-3 h-5 w-5 text-primary" aria-hidden="true" />
                      Schedule a Consultation
                    </CardTitle>
                    <CardDescription>
                      Book a personalized consultation to discuss your specific accounting and compliance needs.
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <ConsultationForm />
                  </CardContent>
                </Card>
              </TabsContent>
            </Tabs>
          </div>
          
          {/* Contact Information & Additional Options */}
          <div className="space-y-8">
            {/* Contact Details */}
            <Card className="service-card">
              <CardHeader>
                <CardTitle className="flex items-center">
                  <Phone className="mr-3 h-5 w-5 text-primary" aria-hidden="true" />
                  Contact Information
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="flex items-start space-x-4">
                  <Phone className="h-5 w-5 text-primary mt-1" aria-hidden="true" />
                  <div>
                    <p className="font-medium text-foreground">Phone</p>
                    <p className="text-muted-foreground">
                      <a 
                        href="tel:+17038297277" 
                        className="hover:text-primary transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring"
                        data-testid="link-phone"
                      >
                        (703) 829-7277 ext 101
                      </a>
                    </p>
                    <p className="text-sm text-accent mt-1">
                      💬 Text preferred for quicker response
                    </p>
                  </div>
                </div>
                
                <div className="flex items-start space-x-4">
                  <Mail className="h-5 w-5 text-primary mt-1" aria-hidden="true" />
                  <div>
                    <p className="font-medium text-foreground">Email</p>
                    <p className="text-muted-foreground">
                      <a 
                        href="mailto:tmhglobal@gmail.com" 
                        className="hover:text-primary transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring"
                        data-testid="link-email"
                      >
                        tmhglobal@gmail.com
                      </a>
                    </p>
                  </div>
                </div>
                
                <div className="flex items-start space-x-4">
                  <MapPin className="h-5 w-5 text-primary mt-1" aria-hidden="true" />
                  <div>
                    <p className="font-medium text-foreground">Address</p>
                    <p className="text-muted-foreground">
                      4196 Merchant Plaza Suite 634<br />
                      Woodbridge, VA 22192
                    </p>
                  </div>
                </div>
                
                <div className="flex items-start space-x-4">
                  <Clock className="h-5 w-5 text-primary mt-1" aria-hidden="true" />
                  <div>
                    <p className="font-medium text-foreground">Business Hours</p>
                    <p className="text-muted-foreground">
                      Monday, Wednesday - Thursday: 10:00 AM - 8:00 PM<br />
                      Tuesday: Closed<br />
                      Friday: Closed<br />
                      Saturday & Sunday: Closed
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>
            
            {/* Free Consultation - Only show when not on consultation tab */}
            {activeTab !== "consultation" && (
              <Card className="border-primary/20 bg-primary/5">
                <CardHeader>
                  <CardTitle className="flex items-center text-primary">
                    <Calendar className="mr-3 h-5 w-5" aria-hidden="true" />
                    Free Initial Consultation
                  </CardTitle>
                  <CardDescription>
                    Book a complimentary 30-minute consultation to discuss your accounting and 
                    government contracting needs.
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <ul className="space-y-2 text-sm text-muted-foreground mb-6">
                    <li>• Assess your current financial processes</li>
                    <li>• Identify government contracting opportunities</li>
                    <li>• Review compliance requirements</li>
                    <li>• Discuss our service offerings</li>
                  </ul>
                  <Button 
                    className="w-full btn-primary"
                    onClick={() => setActiveTab("consultation")}
                    data-testid="button-free-consultation"
                  >
                    <Calendar className="mr-2 h-4 w-4" aria-hidden="true" />
                    Schedule Free Consultation
                  </Button>
                </CardContent>
              </Card>
            )}
            
            {/* Emergency Contact */}
            <Alert className="border-destructive/50 bg-destructive/5">
              <AlertTriangle className="h-4 w-4 text-destructive" aria-hidden="true" />
              <AlertDescription>
                <div className="space-y-3">
                  <p className="font-medium text-destructive">
                    Urgent Compliance Issue?
                  </p>
                  <p className="text-sm text-muted-foreground">
                    If you have an urgent government compliance deadline or critical 
                    financial issue, contact us immediately for priority assistance.
                  </p>
                  <Button 
                    variant="destructive"
                    size="sm"
                    onClick={handleEmergencyContact}
                    className="w-full"
                    data-testid="button-emergency-contact"
                  >
                    <Phone className="mr-2 h-4 w-4" aria-hidden="true" />
                    Emergency Contact
                  </Button>
                </div>
              </AlertDescription>
            </Alert>

            {/* SMS Communication */}
            <Card className="service-card">
              <CardHeader>
                <CardTitle className="flex items-center">
                  <MessageSquare className="mr-3 h-5 w-5 text-accent" aria-hidden="true" />
                  SMS Communication
                </CardTitle>
                <CardDescription>
                  Receive important updates via secure text messaging
                </CardDescription>
              </CardHeader>
              <CardContent>
                <p className="text-sm text-muted-foreground mb-4">
                  Stay informed about deadlines, document requests, and important 
                  compliance updates through our secure SMS service.
                </p>
                <Button 
                  variant="outline"
                  className="w-full btn-secondary"
                  data-testid="button-sms-signup"
                >
                  <MessageSquare className="mr-2 h-4 w-4" aria-hidden="true" />
                  Enable SMS Updates
                </Button>
              </CardContent>
            </Card>

            {/* Team Access */}
            <Card className="service-card">
              <CardHeader>
                <CardTitle className="flex items-center">
                  <Users className="mr-3 h-5 w-5 text-accent" aria-hidden="true" />
                  Meet Our Team
                </CardTitle>
                <CardDescription>
                  Connect with our specialized experts
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-3 text-sm">
                  <div>
                    <p className="font-medium text-foreground">Sarah Johnson, CPA</p>
                    <p className="text-muted-foreground">Senior Accounting Manager</p>
                  </div>
                  <div>
                    <p className="font-medium text-foreground">Michael Chen</p>
                    <p className="text-muted-foreground">Government Contracting Specialist</p>
                  </div>
                  <div>
                    <p className="font-medium text-foreground">Dr. Amanda Rodriguez</p>
                    <p className="text-muted-foreground">Compliance & Accessibility Expert</p>
                  </div>
                </div>
                <Button 
                  variant="outline"
                  className="w-full mt-4 btn-secondary"
                  data-testid="button-team-directory"
                >
                  <Users className="mr-2 h-4 w-4" aria-hidden="true" />
                  View Team Directory
                </Button>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Map or Additional Info Section */}
        <div className="mt-16 bg-muted/30 rounded-2xl p-8">
          <div className="text-center">
            <h2 className="text-2xl font-bold text-foreground mb-4">
              Serving Businesses Nationwide
            </h2>
            <p className="text-muted-foreground max-w-2xl mx-auto">
              While our headquarters are in Washington, DC, we provide accounting and 
              government contracting services to businesses across the United States. 
              Virtual consultations and remote services are available.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}
