import { useState } from "react";
import { CalendarSetup } from "@/components/calendar-setup";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Calendar } from "@/components/ui/calendar";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { 
  Video,
  Calendar as CalendarIcon,
  Clock,
  User,
  Phone,
  Mail,
  FileText,
  CheckCircle,
  Calendar as CalIcon,
  Users,
  MessageSquare,
  Star,
  ExternalLink,
  Plus,
  Filter,
  Download,
  Share
} from "lucide-react";
import { format, addDays, isSameDay, isAfter, isBefore } from "date-fns";

// Backend consultation data structure
interface Consultation {
  id: string;
  firstName: string;
  lastName: string;
  email: string;
  phone?: string;
  companyName?: string;
  serviceType: string;
  preferredDate: string;
  preferredTime: string;
  message?: string;
  calendarEventId?: string;
  status?: 'scheduled' | 'in-progress' | 'completed' | 'cancelled' | 'rescheduled';
  createdAt: string;
}

// Helper function to get client display name
const getClientName = (consultation: Consultation) => 
  `${consultation.firstName} ${consultation.lastName}`;

// Helper function to get scheduled date/time
const getScheduledDateTime = (consultation: Consultation) => 
  `${consultation.preferredDate}T${consultation.preferredTime}:00`;

export default function Consultation() {
  const [selectedDate, setSelectedDate] = useState<Date>(new Date());
  const [showScheduleDialog, setShowScheduleDialog] = useState(false);
  const [activeTab, setActiveTab] = useState("upcoming");
  const [filterCategory, setFilterCategory] = useState<string>("all");
  const [newConsultation, setNewConsultation] = useState({
    title: "",
    description: "",
    category: "business-development", // Match our working serviceTypes
    duration: "60",
    meetingType: "phone",
    priority: "medium",
    scheduledTime: "14:00", // 2:00 PM
    email: "",
    phone: "",
    clientName: ""
  });
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch consultations
  const { data: consultations = [], isLoading: consultationsLoading } = useQuery({
    queryKey: ["/api/consultations", filterCategory],
    queryFn: () => {
      const url = filterCategory === "all" 
        ? "/api/consultations" 
        : `/api/consultations?category=${filterCategory}`;
      return apiRequest("GET", url).then(res => res.json());
    }
  });

  // Fetch available time slots for selected date
  const { data: availableSlots = [], isLoading: slotsLoading } = useQuery({
    queryKey: ["/api/consultations/availability", format(selectedDate, 'yyyy-MM-dd')],
    queryFn: () => apiRequest("GET", `/api/consultations/availability?date=${format(selectedDate, 'yyyy-MM-dd')}`).then(res => res.json()),
    enabled: showScheduleDialog
  });

  // Current user
  const { data: user } = useQuery({
    queryKey: ["/api/user"],
    retry: false,
  });

  // Schedule consultation mutation
  const scheduleConsultationMutation = useMutation({
    mutationFn: async (consultationData: any) => {
      // Transform data to match our working API schema
      const transformedData = {
        firstName: (user as any)?.firstName || consultationData.clientName.split(' ')[0] || '',
        lastName: (user as any)?.lastName || consultationData.clientName.split(' ').slice(1).join(' ') || '',
        email: (user as any)?.email || consultationData.email || '',
        phone: consultationData.phone || '',
        companyName: '', // Optional
        serviceType: consultationData.category, // Map category to serviceType
        preferredDate: format(selectedDate, 'yyyy-MM-dd'),
        preferredTime: consultationData.scheduledTime,
        message: consultationData.description || consultationData.title || 'Consultation request'
      };
      
      const response = await apiRequest("POST", "/api/consultations", transformedData);
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Consultation Scheduled",
        description: "Your consultation has been scheduled successfully. You'll receive a confirmation email with meeting details.",
      });
      setShowScheduleDialog(false);
      setNewConsultation({
        title: "",
        description: "",
        category: "general",
        duration: "60",
        meetingType: "phone",
        priority: "medium",
        scheduledTime: "09:00",
        email: "",
        phone: "",
        clientName: ""
      });
      queryClient.invalidateQueries({ queryKey: ["/api/consultations"] });
    },
    onError: (error: any) => {
      toast({
        title: "Scheduling Failed",
        description: error.message || "Failed to schedule consultation.",
        variant: "destructive",
      });
    }
  });

  // Cancel consultation mutation
  const cancelConsultationMutation = useMutation({
    mutationFn: async (consultationId: string) => {
      const response = await apiRequest("PUT", `/api/consultations/${consultationId}/cancel`);
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Consultation Cancelled",
        description: "The consultation has been cancelled successfully.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/consultations"] });
    }
  });

  // Reschedule consultation mutation
  const rescheduleConsultationMutation = useMutation({
    mutationFn: async ({ consultationId, newDate, newTime }: { consultationId: string; newDate: Date; newTime: string }) => {
      const response = await apiRequest("PUT", `/api/consultations/${consultationId}/reschedule`, {
        scheduledDate: format(newDate, 'yyyy-MM-dd') + 'T' + newTime + ':00'
      });
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Consultation Rescheduled",
        description: "The consultation has been rescheduled successfully.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/consultations"] });
    }
  });

  const handleScheduleConsultation = () => {
    if (!newConsultation.title.trim()) {
      toast({
        title: "Title Required",
        description: "Please enter a consultation title.",
        variant: "destructive",
      });
      return;
    }

    if (!newConsultation.description.trim()) {
      toast({
        title: "Description Required",
        description: "Please describe what you'd like to discuss.",
        variant: "destructive",
      });
      return;
    }

    scheduleConsultationMutation.mutate(newConsultation);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'scheduled': return 'bg-blue-500';
      case 'in-progress': return 'bg-green-500';
      case 'completed': return 'bg-gray-500';
      case 'cancelled': return 'bg-red-500';
      case 'rescheduled': return 'bg-yellow-500';
      default: return 'bg-gray-500';
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'urgent': return 'border-red-500 bg-red-50';
      case 'high': return 'border-orange-500 bg-orange-50';
      case 'medium': return 'border-yellow-500 bg-yellow-50';
      case 'low': return 'border-blue-500 bg-blue-50';
      default: return 'border-gray-500 bg-gray-50';
    }
  };

  const getCategoryIcon = (serviceType: string) => {
    switch (serviceType) {
      case 'compliance': return <CheckCircle className="w-4 h-4" />;
      case 'tax': return <FileText className="w-4 h-4" />;
      case 'hr-services': return <Users className="w-4 h-4" />;
      case 'business-development': return <Star className="w-4 h-4" />;
      case 'government-contracting': return <FileText className="w-4 h-4" />;
      default: return <MessageSquare className="w-4 h-4" />;
    }
  };

  const upcomingConsultations = consultations.filter((c: Consultation) => 
    (!c.status || c.status === 'scheduled') && isAfter(new Date(getScheduledDateTime(c)), new Date())
  );

  const todayConsultations = consultations.filter((c: Consultation) => 
    isSameDay(new Date(getScheduledDateTime(c)), new Date()) && c.status !== 'cancelled'
  );

  const pastConsultations = consultations.filter((c: Consultation) => 
    c.status === 'completed' || isBefore(new Date(getScheduledDateTime(c)), new Date())
  );

  return (
    <div className="container mx-auto px-4 py-8 space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight" data-testid="page-title">
            One-on-One Consultations
          </h1>
          <p className="text-muted-foreground mt-2" data-testid="page-description">
            Schedule personalized business consultations with TMH Global experts
          </p>
        </div>
        <Dialog open={showScheduleDialog} onOpenChange={setShowScheduleDialog}>
          <DialogTrigger asChild>
            <Button className="flex items-center gap-2" data-testid="schedule-consultation-button">
              <Plus className="w-4 h-4" />
              Schedule Consultation
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl max-h-[90vh]">
            <DialogHeader>
              <DialogTitle>Schedule New Consultation</DialogTitle>
              <DialogDescription>
                Book a one-on-one session with our business experts
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-6 overflow-y-auto max-h-[calc(90vh-120px)] pr-2">
              {/* Client Information */}
              <div className="grid gap-4">
                <Label className="text-base font-semibold">📝 Your Information</Label>
                <div>
                  <Label htmlFor="clientName">Your Full Name</Label>
                  <Input
                    id="clientName"
                    value={newConsultation.clientName}
                    onChange={(e) => setNewConsultation({...newConsultation, clientName: e.target.value})}
                    placeholder="John Smith"
                    data-testid="consultation-client-name-input"
                    required
                  />
                </div>
                
                <div>
                  <Label htmlFor="title">Consultation Title</Label>
                  <Input
                    id="title"
                    value={newConsultation.title}
                    onChange={(e) => setNewConsultation({...newConsultation, title: e.target.value})}
                    placeholder="Section 508 Compliance Review"
                    data-testid="consultation-title-input"
                  />
                </div>
                
                <div>
                  <Label htmlFor="description">What would you like to discuss?</Label>
                  <Textarea
                    id="description"
                    value={newConsultation.description}
                    onChange={(e) => setNewConsultation({...newConsultation, description: e.target.value})}
                    placeholder="I need help reviewing our website for Section 508 compliance requirements..."
                    rows={3}
                    data-testid="consultation-description-input"
                  />
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="email">Email Address</Label>
                    <Input
                      id="email"
                      type="email"
                      value={newConsultation.email}
                      onChange={(e) => setNewConsultation({...newConsultation, email: e.target.value})}
                      placeholder="your.email@company.com"
                      data-testid="consultation-email-input"
                      required
                    />
                  </div>
                  
                  <div>
                    <Label htmlFor="phone">Phone Number</Label>
                    <Input
                      id="phone"
                      type="tel"
                      value={newConsultation.phone}
                      onChange={(e) => setNewConsultation({...newConsultation, phone: e.target.value})}
                      placeholder="(555) 123-4567"
                      data-testid="consultation-phone-input"
                      required
                    />
                  </div>
                </div>
              </div>

              {/* Consultation Details */}
              <div className="grid gap-4">
                <Label className="text-base font-semibold">💼 Consultation Details</Label>
                
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="category">Service Category</Label>
                    <Select 
                      value={newConsultation.category} 
                      onValueChange={(value) => setNewConsultation({...newConsultation, category: value})}
                    >
                      <SelectTrigger data-testid="consultation-category-select">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="general-consultation">General Consultation</SelectItem>
                        <SelectItem value="bookkeeping">Bookkeeping Services</SelectItem>
                        <SelectItem value="hr-services">HR Services</SelectItem>
                        <SelectItem value="cfo-services">CFO Services</SelectItem>
                        <SelectItem value="auditing">Auditing</SelectItem>
                        <SelectItem value="business-development">Business Development</SelectItem>
                        <SelectItem value="tax-preparation">Tax Preparation</SelectItem>
                        <SelectItem value="sam-registration">S.A.M. Registration</SelectItem>
                        <SelectItem value="government-contracting">Government Contracting</SelectItem>
                        <SelectItem value="compliance-assessment">Compliance Assessment</SelectItem>
                        <SelectItem value="vpat-documentation">VPAT Documentation</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <Label htmlFor="duration">Duration</Label>
                    <Select 
                      value={newConsultation.duration} 
                      onValueChange={(value) => setNewConsultation({...newConsultation, duration: value})}
                    >
                      <SelectTrigger data-testid="consultation-duration-select">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="30">30 minutes</SelectItem>
                        <SelectItem value="60">1 hour</SelectItem>
                        <SelectItem value="90">1.5 hours</SelectItem>
                        <SelectItem value="120">2 hours</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="meeting-type">Meeting Type</Label>
                    <Select 
                      value={newConsultation.meetingType} 
                      onValueChange={(value) => setNewConsultation({...newConsultation, meetingType: value})}
                    >
                      <SelectTrigger data-testid="consultation-meeting-type-select">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="phone">Phone Call</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <Label htmlFor="priority">Priority</Label>
                    <Select 
                      value={newConsultation.priority} 
                      onValueChange={(value) => setNewConsultation({...newConsultation, priority: value})}
                    >
                      <SelectTrigger data-testid="consultation-priority-select">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="low">Low</SelectItem>
                        <SelectItem value="medium">Medium</SelectItem>
                        <SelectItem value="high">High</SelectItem>
                        <SelectItem value="urgent">Urgent</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>
              </div>

              {/* Date and Time Selection */}
              <div>
                <Label className="text-base font-semibold">Choose Your Consultation Date & Time</Label>
                <p className="text-sm text-muted-foreground mb-3">
                  Select a date starting tomorrow. We're closed on Tuesdays, Saturdays, and Sundays.
                </p>
                <div className="flex gap-4 mt-2">
                  <div className="space-y-2">
                    <Label className="text-sm font-medium">📅 Select Date</Label>
                    <Calendar
                      mode="single"
                      selected={selectedDate}
                      onSelect={(date) => date && setSelectedDate(date)}
                      disabled={(date) => isBefore(date, addDays(new Date(), 1))}
                      className="rounded-md border"
                    />
                  </div>
                  
                  <div className="flex-1 space-y-2">
                    <Label className="text-sm font-medium">🕒 Available Time Slots</Label>
                    {slotsLoading ? (
                      <div className="text-center py-4">Loading available times...</div>
                    ) : availableSlots.length === 0 ? (
                      <div className="text-center py-4 text-muted-foreground">
                        No available slots for this date
                      </div>
                    ) : (
                      <div className="grid grid-cols-2 gap-2 max-h-32 overflow-y-auto">
                        {availableSlots.map((slot: {time: string, available: boolean}) => (
                          <Button
                            key={slot.time}
                            variant={newConsultation.scheduledTime === slot.time ? "default" : "outline"}
                            size="sm"
                            onClick={() => setNewConsultation({...newConsultation, scheduledTime: slot.time})}
                            disabled={!slot.available}
                            data-testid={`time-slot-${slot.time}`}
                          >
                            {slot.time}
                          </Button>
                        ))}
                      </div>
                    )}
                  </div>
                </div>
              </div>

              <div className="flex gap-2">
                <Button
                  onClick={handleScheduleConsultation}
                  disabled={scheduleConsultationMutation.isPending}
                  className="flex-1"
                  data-testid="submit-consultation"
                >
                  {scheduleConsultationMutation.isPending ? "Scheduling..." : "Schedule Consultation"}
                </Button>
                <Button
                  variant="outline"
                  onClick={() => setShowScheduleDialog(false)}
                  className="flex-1"
                >
                  Cancel
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      </div>

      {/* Quick Stats */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Today's Sessions</CardTitle>
            <CalendarIcon className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="today-sessions-count">
              {todayConsultations.length}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Upcoming</CardTitle>
            <Clock className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="upcoming-sessions-count">
              {upcomingConsultations.length}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Completed</CardTitle>
            <CheckCircle className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="completed-sessions-count">
              {pastConsultations.length}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Average Rating</CardTitle>
            <Star className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="average-rating">
              4.8
            </div>
            <p className="text-xs text-muted-foreground">
              Based on client feedback
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Filter */}
      <div className="flex items-center gap-4">
        <Label htmlFor="category-filter">Filter by Category:</Label>
        <Select value={filterCategory} onValueChange={setFilterCategory}>
          <SelectTrigger className="w-[200px]" data-testid="consultation-category-filter">
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">All Categories</SelectItem>
            <SelectItem value="general">General Business</SelectItem>
            <SelectItem value="compliance">Compliance & Accessibility</SelectItem>
            <SelectItem value="tax">Tax Preparation</SelectItem>
            <SelectItem value="hr">HR Management</SelectItem>
            <SelectItem value="business-development">Business Development</SelectItem>
            <SelectItem value="government-contracting">Government Contracting</SelectItem>
          </SelectContent>
        </Select>
      </div>

      {/* Google Calendar Integration */}
      <CalendarSetup />

      {/* Consultations List */}
      <div className="space-y-6">
        {/* Today's Consultations */}
        {todayConsultations.length > 0 && (
          <div>
            <h2 className="text-2xl font-bold mb-4">Today's Consultations</h2>
            <div className="grid gap-4">
              {todayConsultations.map((consultation: Consultation) => (
                <Card key={consultation.id} className="border-l-4 border-l-green-500" data-testid={`today-consultation-${consultation.id}`}>
                  <CardHeader>
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <CardTitle className="text-lg flex items-center gap-2">
                          {getCategoryIcon(consultation.serviceType)}
                          {consultation.message || 'Consultation Session'}
                        </CardTitle>
                        <CardDescription className="mt-1">
                          {format(new Date(getScheduledDateTime(consultation)), "p")} • {getClientName(consultation)}
                        </CardDescription>
                      </div>
                      <div className="flex items-center gap-2">
                        <Badge className="bg-blue-500 text-white">
                          {consultation.serviceType}
                        </Badge>
                        <Badge className="bg-green-500 text-white">
                          {consultation.status || 'scheduled'}
                        </Badge>
                      </div>
                    </div>
                  </CardHeader>
                  <CardContent>
                    <p className="text-sm text-muted-foreground mb-4">
                      {consultation.message || 'No additional details provided'}
                    </p>
                    <div className="flex items-center justify-between">
                      <div className="text-sm">
                        <p><strong>Client:</strong> {getClientName(consultation)}</p>
                        <p><strong>Service:</strong> {consultation.serviceType}</p>
                      </div>
                      <div className="flex gap-2">
                        {false && (
                          <Button 
                            size="sm"
                            onClick={() => window.open(consultation.zoomJoinUrl!, '_blank')}
                            data-testid={`join-consultation-${consultation.id}`}
                          >
                            <Video className="w-4 h-4 mr-2" />
                            Join Meeting
                          </Button>
                        )}
                        <Button 
                          size="sm" 
                          variant="outline"
                          onClick={() => rescheduleConsultationMutation.mutate({
                            consultationId: consultation.id,
                            newDate: addDays(new Date(), 1),
                            newTime: "10:00"
                          })}
                          data-testid={`reschedule-consultation-${consultation.id}`}
                        >
                          Reschedule
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        )}

        {/* Upcoming Consultations */}
        <div>
          <h2 className="text-2xl font-bold mb-4">Upcoming Consultations</h2>
          {consultationsLoading ? (
            <Card>
              <CardContent className="pt-6">
                <div className="text-center">Loading consultations...</div>
              </CardContent>
            </Card>
          ) : upcomingConsultations.length === 0 ? (
            <Card>
              <CardContent className="text-center py-12">
                <CalendarIcon className="w-12 h-12 mx-auto text-muted-foreground mb-4" />
                <h3 className="text-lg font-semibold mb-2">No Upcoming Consultations</h3>
                <p className="text-muted-foreground mb-4">
                  Schedule your first consultation to get personalized business guidance
                </p>
                <Button onClick={() => setShowScheduleDialog(true)}>
                  Schedule Consultation
                </Button>
              </CardContent>
            </Card>
          ) : (
            <div className="grid gap-4">
              {upcomingConsultations.map((consultation: Consultation) => (
                <Card key={consultation.id} className={`border-l-4 ${getPriorityColor(consultation.priority)}`} data-testid={`upcoming-consultation-${consultation.id}`}>
                  <CardHeader>
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <CardTitle className="text-lg flex items-center gap-2">
                          {getCategoryIcon(consultation.serviceType)}
                          {(consultation.message || 'Consultation Session')}
                        </CardTitle>
                        <CardDescription className="mt-1">
                          {format(new Date(getScheduledDateTime(consultation)), "PPP 'at' p")} • 60 minutes
                        </CardDescription>
                      </div>
                      <div className="flex items-center gap-2">
                        <Badge className="bg-blue-500 text-white">
                          {consultation.serviceType}
                        </Badge>
                        <Badge className="bg-green-500 text-white">
                          {consultation.status || 'scheduled'}
                        </Badge>
                      </div>
                    </div>
                  </CardHeader>
                  <CardContent>
                    <p className="text-sm text-muted-foreground mb-4">
                      {consultation.message || 'No additional details provided'}
                    </p>
                    <div className="flex items-center justify-between">
                      <div className="text-sm">
                        <p><strong>Client:</strong> {getClientName(consultation)}</p>
                        <p><strong>Service:</strong> {consultation.serviceType}</p>
                      </div>
                      <div className="flex gap-2">
                        <Button 
                          size="sm" 
                          variant="outline"
                          onClick={() => cancelConsultationMutation.mutate(consultation.id)}
                          data-testid={`cancel-consultation-${consultation.id}`}
                        >
                          Cancel
                        </Button>
                        <Button 
                          size="sm" 
                          variant="outline"
                          data-testid={`reschedule-consultation-${consultation.id}`}
                        >
                          Reschedule
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
        </div>

        {/* Past Consultations */}
        {pastConsultations.length > 0 && (
          <div>
            <h2 className="text-2xl font-bold mb-4">Past Consultations</h2>
            <div className="grid gap-4">
              {pastConsultations.slice(0, 5).map((consultation: Consultation) => (
                <Card key={consultation.id} className="border-l-4 border-l-gray-500" data-testid={`past-consultation-${consultation.id}`}>
                  <CardHeader>
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <CardTitle className="text-lg flex items-center gap-2">
                          {getCategoryIcon(consultation.serviceType)}
                          {(consultation.message || 'Consultation Session')}
                        </CardTitle>
                        <CardDescription className="mt-1">
                          {format(new Date(getScheduledDateTime(consultation)), "PPP")} • 60 minutes
                        </CardDescription>
                      </div>
                      <div className="flex items-center gap-2">
                        {consultation.rating && (
                          <div className="flex items-center gap-1">
                            <Star className="w-4 h-4 fill-yellow-400 text-yellow-400" />
                            <span className="text-sm">{consultation.rating}</span>
                          </div>
                        )}
                        <Badge className={`${getStatusColor(consultation.status)} text-white`}>
                          {consultation.status}
                        </Badge>
                      </div>
                    </div>
                  </CardHeader>
                  <CardContent>
                    {consultation.outcome && (
                      <div className="bg-green-50 border-l-4 border-l-green-400 p-3 rounded mb-4">
                        <p className="text-sm text-green-700">
                          <strong>Outcome:</strong> {consultation.outcome}
                        </p>
                      </div>
                    )}
                    <div className="flex items-center justify-between">
                      <div className="text-sm">
                        <p><strong>Client:</strong> {getClientName(consultation)}</p>
                      </div>
                      <div className="flex gap-2">
                        <Button size="sm" variant="outline">
                          <Download className="w-4 h-4 mr-2" />
                          Summary
                        </Button>
                        <Button size="sm" variant="outline">
                          <Share className="w-4 h-4 mr-2" />
                          Share
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        )}
      </div>
    </div>
  );
}