import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Progress } from "@/components/ui/progress";
import { useToast } from "@/hooks/use-toast";
import { 
  Shield, 
  AlertTriangle, 
  CheckCircle, 
  Eye, 
  Search, 
  FileText, 
  BarChart3,
  TrendingUp,
  AlertCircle,
  RefreshCw,
  Download,
  Settings,
  Clock,
  Target,
  Zap
} from "lucide-react";
import { format, isAfter } from "date-fns";

interface ComplianceScan {
  id: string;
  url: string;
  status: 'pending' | 'running' | 'completed' | 'failed';
  score: number;
  issuesFound: number;
  criticalIssues: number;
  warningIssues: number;
  passedChecks: number;
  totalChecks: number;
  scanDate: string;
  scanDuration?: number;
  userAgent: string;
  viewport: string;
}

interface ComplianceIssue {
  id: string;
  scanId: string;
  severity: 'critical' | 'serious' | 'moderate' | 'minor';
  rule: string;
  description: string;
  element: string;
  xpath: string;
  recommendation: string;
  wcagLevel: 'A' | 'AA' | 'AAA';
  section508: boolean;
}

interface ComplianceAlert {
  id: string;
  type: 'new_critical' | 'threshold_exceeded' | 'compliance_degraded' | 'scan_failed';
  message: string;
  url?: string;
  severity: 'high' | 'medium' | 'low';
  isRead: boolean;
  createdAt: string;
}

export default function ComplianceMonitoring() {
  const [activeTab, setActiveTab] = useState("overview");
  const [newScanUrl, setNewScanUrl] = useState("");
  const [filterSeverity, setFilterSeverity] = useState<string>("all");
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch compliance scans
  const { data: scans = [], isLoading: scansLoading } = useQuery({
    queryKey: ["/api/compliance/scans"],
    queryFn: () => apiRequest("GET", "/api/compliance/scans").then(res => res.json()),
    refetchInterval: 30000 // Refresh every 30 seconds
  });

  // Fetch compliance issues
  const { data: issues = [], isLoading: issuesLoading } = useQuery({
    queryKey: ["/api/compliance/issues", filterSeverity],
    queryFn: () => {
      const url = filterSeverity === "all" 
        ? "/api/compliance/issues" 
        : `/api/compliance/issues?severity=${filterSeverity}`;
      return apiRequest("GET", url).then(res => res.json());
    }
  });

  // Fetch compliance alerts
  const { data: alerts = [], isLoading: alertsLoading } = useQuery({
    queryKey: ["/api/compliance/alerts"],
    queryFn: () => apiRequest("GET", "/api/compliance/alerts").then(res => res.json())
  });

  // Start new scan mutation
  const scanMutation = useMutation({
    mutationFn: async (url: string) => {
      const response = await apiRequest("POST", "/api/compliance/scan", { url });
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Scan Started",
        description: "Accessibility scan has been initiated. Results will appear shortly.",
      });
      setNewScanUrl("");
      queryClient.invalidateQueries({ queryKey: ["/api/compliance/scans"] });
    },
    onError: (error: any) => {
      toast({
        title: "Scan Failed",
        description: error.message || "Failed to start accessibility scan.",
        variant: "destructive",
      });
    }
  });

  // Mark alert as read mutation
  const markAlertReadMutation = useMutation({
    mutationFn: async (alertId: string) => {
      const response = await apiRequest("PUT", `/api/compliance/alerts/${alertId}/read`);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/compliance/alerts"] });
    }
  });

  const handleStartScan = () => {
    if (!newScanUrl.trim()) {
      toast({
        title: "URL Required",
        description: "Please enter a URL to scan for accessibility compliance.",
        variant: "destructive",
      });
      return;
    }

    // Validate URL format
    try {
      new URL(newScanUrl);
      scanMutation.mutate(newScanUrl);
    } catch {
      toast({
        title: "Invalid URL",
        description: "Please enter a valid URL (including http:// or https://).",
        variant: "destructive",
      });
    }
  };

  const getComplianceScore = () => {
    if (scans.length === 0) return 0;
    const totalScore = scans.reduce((sum: number, scan: ComplianceScan) => sum + scan.score, 0);
    return Math.round(totalScore / scans.length);
  };

  const getTotalIssues = () => {
    return issues.length;
  };

  const getCriticalIssues = () => {
    return issues.filter((issue: ComplianceIssue) => issue.severity === 'critical').length;
  };

  const getUnreadAlerts = () => {
    return alerts.filter((alert: ComplianceAlert) => !alert.isRead).length;
  };

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case 'critical': return 'text-red-600 bg-red-50 border-red-200';
      case 'serious': return 'text-orange-600 bg-orange-50 border-orange-200';
      case 'moderate': return 'text-yellow-600 bg-yellow-50 border-yellow-200';
      case 'minor': return 'text-blue-600 bg-blue-50 border-blue-200';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  const getScanStatusColor = (status: string) => {
    switch (status) {
      case 'completed': return 'text-green-600 bg-green-50';
      case 'running': return 'text-blue-600 bg-blue-50';
      case 'failed': return 'text-red-600 bg-red-50';
      default: return 'text-gray-600 bg-gray-50';
    }
  };

  return (
    <div className="container mx-auto px-4 py-8 space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight" data-testid="page-title">
            Section 508 Compliance Monitoring
          </h1>
          <p className="text-muted-foreground mt-2" data-testid="page-description">
            Automated accessibility scanning and compliance tracking for government contracting
          </p>
        </div>
        <Button 
          onClick={() => queryClient.invalidateQueries({ queryKey: ["/api/compliance"] })}
          variant="outline"
          className="flex items-center gap-2"
          data-testid="refresh-data"
        >
          <RefreshCw className="w-4 h-4" />
          Refresh
        </Button>
      </div>

      {/* Overview Cards */}
      <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Compliance Score</CardTitle>
            <Shield className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="compliance-score">
              {getComplianceScore()}%
            </div>
            <Progress value={getComplianceScore()} className="mt-2" />
            <p className="text-xs text-muted-foreground mt-2">
              WCAG 2.1 AA compliance average
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Issues</CardTitle>
            <AlertTriangle className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="total-issues">
              {getTotalIssues()}
            </div>
            <p className="text-xs text-muted-foreground mt-2">
              Across all scanned websites
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Critical Issues</CardTitle>
            <AlertCircle className="h-4 w-4 text-red-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600" data-testid="critical-issues">
              {getCriticalIssues()}
            </div>
            <p className="text-xs text-muted-foreground mt-2">
              Require immediate attention
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Unread Alerts</CardTitle>
            <Eye className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="unread-alerts">
              {getUnreadAlerts()}
            </div>
            <p className="text-xs text-muted-foreground mt-2">
              New compliance notifications
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Main Content */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList>
          <TabsTrigger value="overview" data-testid="tab-overview">Overview</TabsTrigger>
          <TabsTrigger value="scans" data-testid="tab-scans">Recent Scans</TabsTrigger>
          <TabsTrigger value="issues" data-testid="tab-issues">Issues</TabsTrigger>
          <TabsTrigger value="alerts" data-testid="tab-alerts">
            Alerts {getUnreadAlerts() > 0 && (
              <Badge variant="destructive" className="ml-2 px-1 py-0 text-xs">
                {getUnreadAlerts()}
              </Badge>
            )}
          </TabsTrigger>
        </TabsList>

        <TabsContent value="overview" className="space-y-6">
          {/* New Scan */}
          <Card>
            <CardHeader>
              <CardTitle>Start New Accessibility Scan</CardTitle>
              <CardDescription>
                Scan any website for Section 508 and WCAG 2.1 AA compliance
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="flex gap-4">
                <div className="flex-1">
                  <Label htmlFor="scan-url">Website URL</Label>
                  <Input
                    id="scan-url"
                    value={newScanUrl}
                    onChange={(e) => setNewScanUrl(e.target.value)}
                    placeholder="https://yourwebsite.com"
                    data-testid="scan-url-input"
                  />
                </div>
                <div className="flex items-end">
                  <Button 
                    onClick={handleStartScan}
                    disabled={scanMutation.isPending}
                    data-testid="start-scan-button"
                  >
                    {scanMutation.isPending ? (
                      <>
                        <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                        Scanning...
                      </>
                    ) : (
                      <>
                        <Search className="w-4 h-4 mr-2" />
                        Start Scan
                      </>
                    )}
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Recent Scans Summary */}
          <Card>
            <CardHeader>
              <CardTitle>Recent Scan Results</CardTitle>
            </CardHeader>
            <CardContent>
              {scansLoading ? (
                <div className="text-center py-8">
                  <RefreshCw className="w-8 h-8 animate-spin mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">Loading scan results...</p>
                </div>
              ) : scans.length === 0 ? (
                <div className="text-center py-8">
                  <Shield className="w-12 h-12 mx-auto text-muted-foreground mb-4" />
                  <h3 className="text-lg font-semibold mb-2">No scans yet</h3>
                  <p className="text-muted-foreground">
                    Start your first accessibility scan to monitor compliance
                  </p>
                </div>
              ) : (
                <div className="space-y-4">
                  {scans.slice(0, 5).map((scan: ComplianceScan) => (
                    <div key={scan.id} className="flex items-center justify-between p-4 border rounded-lg" data-testid={`scan-${scan.id}`}>
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-1">
                          <p className="font-medium">{scan.url}</p>
                          <Badge className={getScanStatusColor(scan.status)}>
                            {scan.status}
                          </Badge>
                        </div>
                        <p className="text-sm text-muted-foreground">
                          {format(new Date(scan.scanDate), "MMM dd, yyyy 'at' h:mm a")}
                        </p>
                      </div>
                      <div className="text-right">
                        <p className="text-2xl font-bold">{scan.score}%</p>
                        <p className="text-sm text-muted-foreground">
                          {scan.issuesFound} issues found
                        </p>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="scans" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>All Accessibility Scans</CardTitle>
              <CardDescription>
                Detailed results from all website accessibility scans
              </CardDescription>
            </CardHeader>
            <CardContent>
              {scansLoading ? (
                <div className="text-center py-8">
                  <RefreshCw className="w-8 h-8 animate-spin mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">Loading scans...</p>
                </div>
              ) : (
                <div className="space-y-4">
                  {scans.map((scan: ComplianceScan) => (
                    <Card key={scan.id} className="border-l-4 border-l-blue-500" data-testid={`detailed-scan-${scan.id}`}>
                      <CardHeader>
                        <div className="flex items-center justify-between">
                          <div>
                            <CardTitle className="text-lg">{scan.url}</CardTitle>
                            <CardDescription>
                              Scanned {format(new Date(scan.scanDate), "PPP 'at' p")}
                              {scan.scanDuration && ` • ${scan.scanDuration}s duration`}
                            </CardDescription>
                          </div>
                          <Badge className={getScanStatusColor(scan.status)}>
                            {scan.status}
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent>
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
                          <div className="text-center">
                            <p className="text-3xl font-bold text-green-600">{scan.score}%</p>
                            <p className="text-sm text-muted-foreground">Compliance Score</p>
                          </div>
                          <div className="text-center">
                            <p className="text-2xl font-bold">{scan.passedChecks}/{scan.totalChecks}</p>
                            <p className="text-sm text-muted-foreground">Checks Passed</p>
                          </div>
                          <div className="text-center">
                            <p className="text-2xl font-bold text-red-600">{scan.criticalIssues}</p>
                            <p className="text-sm text-muted-foreground">Critical Issues</p>
                          </div>
                          <div className="text-center">
                            <p className="text-2xl font-bold text-orange-600">{scan.issuesFound}</p>
                            <p className="text-sm text-muted-foreground">Total Issues</p>
                          </div>
                        </div>
                        <div className="flex gap-2">
                          <Button size="sm" variant="outline">
                            <Eye className="w-4 h-4 mr-2" />
                            View Details
                          </Button>
                          <Button size="sm" variant="outline">
                            <Download className="w-4 h-4 mr-2" />
                            Export Report
                          </Button>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="issues" className="space-y-6">
          {/* Filter */}
          <Card>
            <CardHeader>
              <CardTitle>Accessibility Issues</CardTitle>
              <CardDescription>
                Filter and review accessibility issues found across all scans
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="flex gap-4 mb-6">
                <div className="flex-1">
                  <Label htmlFor="severity-filter">Filter by Severity</Label>
                  <Select value={filterSeverity} onValueChange={setFilterSeverity}>
                    <SelectTrigger data-testid="severity-filter">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Severities</SelectItem>
                      <SelectItem value="critical">Critical</SelectItem>
                      <SelectItem value="serious">Serious</SelectItem>
                      <SelectItem value="moderate">Moderate</SelectItem>
                      <SelectItem value="minor">Minor</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              {issuesLoading ? (
                <div className="text-center py-8">
                  <RefreshCw className="w-8 h-8 animate-spin mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">Loading issues...</p>
                </div>
              ) : issues.length === 0 ? (
                <div className="text-center py-8">
                  <CheckCircle className="w-12 h-12 mx-auto text-green-500 mb-4" />
                  <h3 className="text-lg font-semibold mb-2">No issues found</h3>
                  <p className="text-muted-foreground">
                    {filterSeverity === "all" 
                      ? "All scanned websites are accessibility compliant"
                      : `No ${filterSeverity} issues found`
                    }
                  </p>
                </div>
              ) : (
                <div className="space-y-4">
                  {issues.map((issue: ComplianceIssue) => (
                    <Card key={issue.id} className={`border-l-4 ${getSeverityColor(issue.severity)}`} data-testid={`issue-${issue.id}`}>
                      <CardHeader>
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <CardTitle className="text-lg">{issue.rule}</CardTitle>
                            <CardDescription className="mt-1">
                              {issue.description}
                            </CardDescription>
                          </div>
                          <div className="flex flex-col items-end gap-2">
                            <Badge className={getSeverityColor(issue.severity)}>
                              {issue.severity}
                            </Badge>
                            <div className="flex gap-1">
                              <Badge variant="outline" className="text-xs">
                                WCAG {issue.wcagLevel}
                              </Badge>
                              {issue.section508 && (
                                <Badge variant="outline" className="text-xs">
                                  Section 508
                                </Badge>
                              )}
                            </div>
                          </div>
                        </div>
                      </CardHeader>
                      <CardContent>
                        <div className="space-y-3">
                          <div>
                            <Label className="text-sm font-medium">Element</Label>
                            <code className="block p-2 bg-gray-100 rounded text-sm mt-1">
                              {issue.element}
                            </code>
                          </div>
                          <div>
                            <Label className="text-sm font-medium">Recommendation</Label>
                            <p className="text-sm text-muted-foreground mt-1">
                              {issue.recommendation}
                            </p>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="alerts" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Compliance Alerts</CardTitle>
              <CardDescription>
                Real-time notifications about accessibility compliance issues
              </CardDescription>
            </CardHeader>
            <CardContent>
              {alertsLoading ? (
                <div className="text-center py-8">
                  <RefreshCw className="w-8 h-8 animate-spin mx-auto text-muted-foreground mb-4" />
                  <p className="text-muted-foreground">Loading alerts...</p>
                </div>
              ) : alerts.length === 0 ? (
                <div className="text-center py-8">
                  <CheckCircle className="w-12 h-12 mx-auto text-green-500 mb-4" />
                  <h3 className="text-lg font-semibold mb-2">No alerts</h3>
                  <p className="text-muted-foreground">
                    All your websites are maintaining good accessibility compliance
                  </p>
                </div>
              ) : (
                <div className="space-y-4">
                  {alerts.map((alert: ComplianceAlert) => (
                    <Alert 
                      key={alert.id} 
                      className={`${!alert.isRead ? 'border-l-4 border-l-blue-500 bg-blue-50/50' : ''}`}
                      data-testid={`alert-${alert.id}`}
                    >
                      <AlertTriangle className="h-4 w-4" />
                      <AlertDescription>
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <p className="font-medium">{alert.message}</p>
                            {alert.url && (
                              <p className="text-sm text-muted-foreground mt-1">
                                Website: {alert.url}
                              </p>
                            )}
                            <p className="text-sm text-muted-foreground mt-1">
                              {format(new Date(alert.createdAt), "MMM dd, yyyy 'at' h:mm a")}
                            </p>
                          </div>
                          <div className="flex items-center gap-2">
                            <Badge variant={
                              alert.severity === 'high' ? 'destructive' : 
                              alert.severity === 'medium' ? 'default' : 'secondary'
                            }>
                              {alert.severity}
                            </Badge>
                            {!alert.isRead && (
                              <Button 
                                size="sm" 
                                variant="outline"
                                onClick={() => markAlertReadMutation.mutate(alert.id)}
                                data-testid={`mark-read-${alert.id}`}
                              >
                                Mark Read
                              </Button>
                            )}
                          </div>
                        </div>
                      </AlertDescription>
                    </Alert>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}