import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { Separator } from "@/components/ui/separator";
import { Heart, MessageCircle, Share2, Crown, Lock, Users, TrendingUp } from "lucide-react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { insertPostSchema, type InsertPost, type Post, type User } from "@shared/schema";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

interface PostWithUser extends Post {
  user: User;
  userLiked?: boolean;
}

export default function Community() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [, setLocation] = useLocation();
  const [showCreatePost, setShowCreatePost] = useState(false);

  // Fetch community posts
  const { data: posts = [], isLoading: postsLoading } = useQuery<PostWithUser[]>({
    queryKey: ["/api/posts"],
  });

  // Fetch current user
  const { data: currentUser } = useQuery<User>({
    queryKey: ["/api/user"],
  });

  // Create post form
  const form = useForm<InsertPost>({
    resolver: zodResolver(insertPostSchema),
    defaultValues: {
      title: "",
      content: "",
      category: "general",
      isPublic: true,
      isPremiumOnly: false,
    },
  });

  // Create post mutation
  const createPostMutation = useMutation({
    mutationFn: (data: InsertPost) => apiRequest("POST", "/api/posts", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/posts"] });
      form.reset();
      setShowCreatePost(false);
      toast({
        title: "Post Created",
        description: "Your insight has been shared with the community.",
      });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  // Like post mutation
  const likeMutation = useMutation({
    mutationFn: (postId: string) => apiRequest("POST", `/api/posts/${postId}/like`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/posts"] });
    },
  });

  const onSubmit = (data: InsertPost) => {
    createPostMutation.mutate(data);
  };

  const handleLike = (postId: string) => {
    likeMutation.mutate(postId);
  };

  const getCategoryColor = (category: string) => {
    const colors = {
      "sam-registration": "bg-blue-100 text-blue-800",
      "compliance": "bg-green-100 text-green-800", 
      "contracts": "bg-purple-100 text-purple-800",
      "training": "bg-orange-100 text-orange-800",
      "general": "bg-gray-100 text-gray-800",
    };
    return colors[category as keyof typeof colors] || colors.general;
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 to-blue-50">
      {/* Header */}
      <div className="bg-white border-b shadow-sm">
        <div className="max-w-6xl mx-auto px-6 py-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <div className="bg-gradient-to-r from-blue-600 to-indigo-600 p-3 rounded-lg">
                <Users className="h-6 w-6 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold text-gray-900">TMH Global Community</h1>
                <p className="text-gray-600">Exclusive insights and expertise for government contractors</p>
              </div>
            </div>
            
            {/* Premium Badge */}
            <div className="flex items-center gap-3">
              <Badge variant="secondary" className="px-3 py-1 bg-gradient-to-r from-yellow-100 to-yellow-200 text-yellow-800 border-yellow-300">
                <Crown className="h-4 w-4 mr-1" />
                Premium Community
              </Badge>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-6xl mx-auto px-6 py-8">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-6">
            
            {/* Create Post Section */}
            {!showCreatePost ? (
              <Card className="border-0 shadow-sm bg-white/80 backdrop-blur">
                <CardContent className="p-6">
                  <div className="flex items-center gap-4">
                    <Avatar>
                      <AvatarFallback className="bg-blue-600 text-white">
                        {currentUser?.username?.[0]?.toUpperCase() || "U"}
                      </AvatarFallback>
                    </Avatar>
                    <Button 
                      variant="outline" 
                      className="flex-1 justify-start text-gray-500 bg-gray-50 hover:bg-gray-100"
                      onClick={() => setShowCreatePost(true)}
                      data-testid="button-create-post"
                    >
                      Share your government contracting insights...
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ) : (
              <Card className="border-0 shadow-md bg-white">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <TrendingUp className="h-5 w-5 text-blue-600" />
                    Share Your Expertise
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
                      <FormField
                        control={form.control}
                        name="title"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Title</FormLabel>
                            <FormControl>
                              <Input 
                                placeholder="What's your insight about?" 
                                {...field} 
                                data-testid="input-post-title"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="content"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Content</FormLabel>
                            <FormControl>
                              <Textarea 
                                placeholder="Share your knowledge and experience with fellow contractors..."
                                className="min-h-[120px] resize-none"
                                {...field}
                                data-testid="textarea-post-content"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="category"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Category</FormLabel>
                            <Select onValueChange={field.onChange} defaultValue={field.value}>
                              <FormControl>
                                <SelectTrigger data-testid="select-post-category">
                                  <SelectValue placeholder="Select category" />
                                </SelectTrigger>
                              </FormControl>
                              <SelectContent>
                                <SelectItem value="general">General Discussion</SelectItem>
                                <SelectItem value="sam-registration">SAM Registration</SelectItem>
                                <SelectItem value="compliance">Compliance</SelectItem>
                                <SelectItem value="contracts">Contract Management</SelectItem>
                                <SelectItem value="training">Training & Development</SelectItem>
                              </SelectContent>
                            </Select>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <div className="flex items-center justify-between pt-4">
                        <div className="flex items-center gap-4">
                          <FormField
                            control={form.control}
                            name="isPremiumOnly"
                            render={({ field }) => (
                              <div className="flex items-center gap-2">
                                <input
                                  type="checkbox"
                                  checked={field.value || false}
                                  onChange={field.onChange}
                                  className="w-4 h-4 text-yellow-600"
                                  data-testid="checkbox-premium-only"
                                />
                                <label className="text-sm flex items-center gap-1">
                                  <Lock className="h-4 w-4 text-yellow-600" />
                                  Premium Members Only
                                </label>
                              </div>
                            )}
                          />
                        </div>

                        <div className="flex gap-2">
                          <Button
                            type="button"
                            variant="outline"
                            onClick={() => setShowCreatePost(false)}
                            data-testid="button-cancel-post"
                          >
                            Cancel
                          </Button>
                          <Button 
                            type="submit"
                            disabled={createPostMutation.isPending}
                            data-testid="button-submit-post"
                          >
                            {createPostMutation.isPending ? "Posting..." : "Share Insight"}
                          </Button>
                        </div>
                      </div>
                    </form>
                  </Form>
                </CardContent>
              </Card>
            )}

            {/* Posts Feed */}
            <div className="space-y-4">
              {postsLoading ? (
                <div className="space-y-4">
                  {[1, 2, 3].map((i) => (
                    <Card key={i} className="border-0 shadow-sm">
                      <CardContent className="p-6">
                        <div className="animate-pulse">
                          <div className="flex items-center gap-3 mb-4">
                            <div className="w-10 h-10 bg-gray-200 rounded-full"></div>
                            <div className="space-y-2">
                              <div className="w-32 h-4 bg-gray-200 rounded"></div>
                              <div className="w-24 h-3 bg-gray-200 rounded"></div>
                            </div>
                          </div>
                          <div className="space-y-2">
                            <div className="w-full h-4 bg-gray-200 rounded"></div>
                            <div className="w-3/4 h-4 bg-gray-200 rounded"></div>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              ) : posts.length === 0 ? (
                <Card className="border-0 shadow-sm bg-white/80">
                  <CardContent className="p-12 text-center">
                    <Users className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                    <h3 className="text-lg font-semibold text-gray-900 mb-2">Welcome to TMH Global Community</h3>
                    <p className="text-gray-600 mb-4">
                      Be the first to share your government contracting insights with fellow professionals.
                    </p>
                    <Button onClick={() => setShowCreatePost(true)} data-testid="button-first-post">
                      Share Your First Insight
                    </Button>
                  </CardContent>
                </Card>
              ) : (
                posts.map((post) => (
                  <Card key={post.id} className="border-0 shadow-sm bg-white hover:shadow-md transition-shadow" data-testid={`card-post-${post.id}`}>
                    <CardContent className="p-6">
                      {/* Post Header */}
                      <div className="flex items-start justify-between mb-4">
                        <div className="flex items-center gap-3">
                          <Avatar>
                            <AvatarFallback className="bg-blue-600 text-white">
                              {post.user?.username?.[0]?.toUpperCase() || "U"}
                            </AvatarFallback>
                          </Avatar>
                          <div>
                            <div className="flex items-center gap-2">
                              <p className="font-semibold text-gray-900" data-testid={`text-author-${post.id}`}>
                                {post.user?.username || "Unknown User"}
                              </p>
                              {post.isPremiumOnly && (
                                <Lock className="h-4 w-4 text-yellow-600" />
                              )}
                            </div>
                            <p className="text-sm text-gray-500">
                              {post.createdAt ? new Date(post.createdAt).toLocaleDateString() : "Unknown date"}
                            </p>
                          </div>
                        </div>
                        <Badge className={getCategoryColor(post.category || "general")}>
                          {post.category}
                        </Badge>
                      </div>

                      {/* Post Content */}
                      <div className="mb-4">
                        <h3 className="font-semibold text-gray-900 mb-2" data-testid={`text-title-${post.id}`}>
                          {post.title}
                        </h3>
                        <p className="text-gray-700 leading-relaxed" data-testid={`text-content-${post.id}`}>
                          {post.content}
                        </p>
                      </div>

                      <Separator className="my-4" />

                      {/* Post Actions */}
                      <div className="flex items-center gap-6">
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => handleLike(post.id)}
                          className={`gap-2 ${post.userLiked ? 'text-red-600' : 'text-gray-600'}`}
                          data-testid={`button-like-${post.id}`}
                        >
                          <Heart className={`h-4 w-4 ${post.userLiked ? 'fill-current' : ''}`} />
                          {post.likesCount}
                        </Button>
                        
                        <Button variant="ghost" size="sm" className="gap-2 text-gray-600" data-testid={`button-comment-${post.id}`}>
                          <MessageCircle className="h-4 w-4" />
                          {post.commentsCount}
                        </Button>
                        
                        <Button variant="ghost" size="sm" className="gap-2 text-gray-600" data-testid={`button-share-${post.id}`}>
                          <Share2 className="h-4 w-4" />
                          Share
                        </Button>
                      </div>
                    </CardContent>
                  </Card>
                ))
              )}
            </div>
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            
            {/* Premium Training Access */}
            <Card className="border-0 shadow-sm bg-gradient-to-br from-yellow-50 to-orange-50">
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-yellow-800">
                  <Crown className="h-5 w-5" />
                  Premium Training
                </CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-sm text-yellow-700 mb-4">
                  Access exclusive advanced training content, webinars, and certification programs.
                </p>
                <Button 
                  className="w-full bg-gradient-to-r from-yellow-600 to-orange-600 hover:from-yellow-700 hover:to-orange-700" 
                  onClick={() => setLocation("/premium")}
                  data-testid="button-upgrade-premium"
                >
                  Upgrade to Premium
                </Button>
              </CardContent>
            </Card>

            {/* Community Stats */}
            <Card className="border-0 shadow-sm bg-white/80">
              <CardHeader>
                <CardTitle className="text-gray-900">Community Insights</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <span className="text-sm text-gray-600">Total Posts</span>
                  <span className="font-semibold text-gray-900" data-testid="text-total-posts">{posts.length}</span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm text-gray-600">Active Members</span>
                  <span className="font-semibold text-gray-900">127</span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm text-gray-600">Premium Members</span>
                  <span className="font-semibold text-yellow-600">45</span>
                </div>
              </CardContent>
            </Card>

            {/* Quick Categories */}
            <Card className="border-0 shadow-sm bg-white/80">
              <CardHeader>
                <CardTitle className="text-gray-900">Popular Topics</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  {[
                    { name: "SAM Registration", count: 23 },
                    { name: "Compliance", count: 18 },
                    { name: "Contract Management", count: 15 },
                    { name: "Training", count: 12 },
                  ].map((topic) => (
                    <div key={topic.name} className="flex items-center justify-between p-2 rounded hover:bg-gray-50 cursor-pointer">
                      <span className="text-sm text-gray-700">{topic.name}</span>
                      <Badge variant="secondary" className="text-xs">{topic.count}</Badge>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}