import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { useQuery } from "@tanstack/react-query";
import { useState } from "react";
import { useToast } from "@/hooks/use-toast";
import { 
  Lock, 
  Video, 
  MessageSquare, 
  Upload, 
  Play, 
  Calendar,
  FileText,
  Download,
  Eye,
  Clock,
  Users,
  Shield,
  X
} from "lucide-react";

export default function ClientPortal() {
  const [selectedCategory, setSelectedCategory] = useState<string>("all");
  const [selectedVideo, setSelectedVideo] = useState<any>(null);
  const { toast } = useToast();

  // Fetch training videos
  const { data: trainingVideos, isLoading: videosLoading } = useQuery({
    queryKey: ["/api/training-videos"],
  });

  // Mock data for demonstration
  const clientDocuments = [
    { id: 1, name: "Monthly Financial Report - December 2024", type: "PDF", date: "2024-12-15", category: "Financial" },
    { id: 2, name: "Tax Documents Package 2024", type: "ZIP", date: "2024-12-10", category: "Tax" },
    { id: 3, name: "SAM Registration Certificate", type: "PDF", date: "2024-12-05", category: "Compliance" }
  ];

  const recentMessages = [
    { id: 1, from: "TMH Global Team", subject: "Monthly Report Ready", date: "2024-12-15", unread: true },
    { id: 2, from: "Sarah Johnson, CPA", subject: "Tax Planning Meeting", date: "2024-12-12", unread: false },
    { id: 3, from: "Compliance Team", subject: "SAM Renewal Reminder", date: "2024-12-10", unread: true }
  ];

  const videoCategories = ["all", "Government Contracting", "SAM Registration", "Compliance", "Financial"];

  const filteredVideos = (trainingVideos as any[])?.filter((video: any) => 
    selectedCategory === "all" || video.category === selectedCategory
  ) || [];

  const handleWatchVideo = (video: any) => {
    setSelectedVideo(video);
  };

  const handleViewDocument = (doc: any) => {
    toast({
      title: "Document Viewer",
      description: `Opening: ${doc.name}`,
    });
  };

  const handleDownloadDocument = (doc: any) => {
    toast({
      title: "Download Started",
      description: `Downloading: ${doc.name}`,
    });
  };

  // Extract YouTube video ID from URL
  const getYouTubeVideoId = (url: string): string | null => {
    const patterns = [
      /(?:youtube\.com\/watch\?v=|youtu\.be\/)([^&\n?#]+)/,
      /youtube\.com\/embed\/([^&\n?#]+)/,
    ];
    for (const pattern of patterns) {
      const match = url.match(pattern);
      if (match) return match[1];
    }
    return null;
  };

  const handleReadMessage = (message: any) => {
    toast({
      title: "Message",
      description: `Opening: ${message.subject}`,
    });
  };

  return (
    <div className="py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <h1 className="text-4xl lg:text-5xl font-bold text-foreground mb-6">
            Client Portal
          </h1>
          <p className="text-xl text-muted-foreground max-w-3xl mx-auto">
            Access your financial documents, training materials, and communicate with our team 
            through our secure, WCAG 2.1 AA compliant portal.
          </p>
        </div>

        {/* Portal Features Overview */}
        <div className="grid lg:grid-cols-4 gap-8 mb-16">
          <Card className="service-card text-center">
            <CardHeader>
              <Lock className="h-12 w-12 text-primary mx-auto mb-4" aria-hidden="true" />
              <CardTitle className="text-lg">Secure Access</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground">
                Encrypted document storage with role-based access controls
              </p>
            </CardContent>
          </Card>

          <Card className="service-card text-center">
            <CardHeader>
              <Video className="h-12 w-12 text-accent mx-auto mb-4" aria-hidden="true" />
              <CardTitle className="text-lg">Training Library</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground">
                Accessible video content with captions and transcripts
              </p>
            </CardContent>
          </Card>

          <Card className="service-card text-center">
            <CardHeader>
              <MessageSquare className="h-12 w-12 text-accent mx-auto mb-4" aria-hidden="true" />
              <CardTitle className="text-lg">Communication</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground">
                Secure messaging and SMS integration with our team
              </p>
            </CardContent>
          </Card>

          <Card className="service-card text-center">
            <CardHeader>
              <Upload className="h-12 w-12 text-accent mx-auto mb-4" aria-hidden="true" />
              <CardTitle className="text-lg">File Management</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground">
                Secure file upload and document management system
              </p>
            </CardContent>
          </Card>
        </div>

        {/* Login Notice */}
        <Alert className="mb-12 border-primary bg-primary/5">
          <Shield className="h-4 w-4" aria-hidden="true" />
          <AlertDescription>
            <strong>Client Portal Access:</strong> This is a demonstration of our client portal features. 
            Actual portal access requires secure authentication. Contact us to set up your account.
          </AlertDescription>
        </Alert>

        {/* Training Videos Section */}
        <div className="mb-16">
          <div className="flex items-center justify-between mb-8">
            <h2 className="text-3xl font-bold text-foreground">Training Video Library</h2>
            <div className="flex gap-2 flex-wrap">
              {videoCategories.map((category) => (
                <Button
                  key={category}
                  variant={selectedCategory === category ? "default" : "outline"}
                  size="sm"
                  onClick={() => setSelectedCategory(category)}
                  className="capitalize"
                  data-testid={`filter-${category}`}
                >
                  {category}
                </Button>
              ))}
            </div>
          </div>

          {videosLoading ? (
            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
              {[1, 2, 3].map((i) => (
                <Card key={i} className="animate-pulse">
                  <CardHeader>
                    <div className="h-4 bg-muted rounded w-3/4 mb-2"></div>
                    <div className="h-3 bg-muted rounded w-1/2"></div>
                  </CardHeader>
                  <CardContent>
                    <div className="h-32 bg-muted rounded mb-4"></div>
                    <div className="h-8 bg-muted rounded"></div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : (
            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
              {filteredVideos.map((video: any) => (
                <Card key={video.id} className="service-card">
                  <CardHeader>
                    <div className="flex items-center justify-between">
                      <Badge variant="secondary" className="text-xs">
                        {video.category}
                      </Badge>
                      <Clock className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                    </div>
                    <CardTitle className="text-lg">{video.title}</CardTitle>
                    <CardDescription className="text-sm">
                      {video.description}
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="aspect-video bg-muted rounded-lg flex items-center justify-center mb-4">
                      <Play className="h-12 w-12 text-primary" aria-hidden="true" />
                    </div>
                    <div className="space-y-2">
                      <Button 
                        className="w-full btn-primary"
                        onClick={() => handleWatchVideo(video)}
                        data-testid={`play-video-${video.id}`}
                      >
                        <Play className="mr-2 h-4 w-4" aria-hidden="true" />
                        Watch Video
                      </Button>
                      <div className="flex gap-2">
                        <Button 
                          variant="outline" 
                          size="sm" 
                          className="flex-1"
                          onClick={() => toast({ title: "Captions", description: "Displaying captions for " + video.title })}
                          data-testid={`captions-${video.id}`}
                        >
                          <FileText className="mr-1 h-3 w-3" aria-hidden="true" />
                          Captions
                        </Button>
                        <Button 
                          variant="outline" 
                          size="sm" 
                          className="flex-1"
                          onClick={() => toast({ title: "Transcript Download", description: "Downloading transcript for " + video.title })}
                          data-testid={`transcript-${video.id}`}
                        >
                          <Download className="mr-1 h-3 w-3" aria-hidden="true" />
                          Transcript
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
        </div>

        {/* Document Library */}
        <div className="mb-16">
          <h2 className="text-3xl font-bold text-foreground mb-8">Document Library</h2>
          <div className="space-y-4">
            {clientDocuments.map((doc) => (
              <Card key={doc.id} className="hover:shadow-md transition-shadow">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4">
                      <FileText className="h-8 w-8 text-primary" aria-hidden="true" />
                      <div>
                        <h3 className="font-semibold text-foreground">{doc.name}</h3>
                        <div className="flex items-center space-x-4 text-sm text-muted-foreground">
                          <span>{doc.type}</span>
                          <span>{doc.date}</span>
                          <Badge variant="outline" className="text-xs">
                            {doc.category}
                          </Badge>
                        </div>
                      </div>
                    </div>
                    <div className="flex gap-2">
                      <Button 
                        variant="outline" 
                        size="sm"
                        onClick={() => handleViewDocument(doc)}
                        data-testid={`view-document-${doc.id}`}
                      >
                        <Eye className="mr-1 h-4 w-4" aria-hidden="true" />
                        View
                      </Button>
                      <Button 
                        variant="outline" 
                        size="sm"
                        onClick={() => handleDownloadDocument(doc)}
                        data-testid={`download-document-${doc.id}`}
                      >
                        <Download className="mr-1 h-4 w-4" aria-hidden="true" />
                        Download
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Recent Messages */}
        <div className="mb-16">
          <h2 className="text-3xl font-bold text-foreground mb-8">Recent Messages</h2>
          <div className="space-y-4">
            {recentMessages.map((message) => (
              <Card key={message.id} className="hover:shadow-md transition-shadow">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4">
                      <MessageSquare className="h-8 w-8 text-accent" aria-hidden="true" />
                      <div>
                        <div className="flex items-center space-x-2">
                          <h3 className="font-semibold text-foreground">{message.subject}</h3>
                          {message.unread && (
                            <Badge variant="destructive" className="text-xs">
                              New
                            </Badge>
                          )}
                        </div>
                        <div className="flex items-center space-x-4 text-sm text-muted-foreground">
                          <span>From: {message.from}</span>
                          <span>{message.date}</span>
                        </div>
                      </div>
                    </div>
                    <Button 
                      variant="outline" 
                      size="sm"
                      onClick={() => handleReadMessage(message)}
                      data-testid={`read-message-${message.id}`}
                    >
                      <Eye className="mr-1 h-4 w-4" aria-hidden="true" />
                      Read
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* File Upload Section */}
        <div className="mb-16">
          <Card className="service-card">
            <CardHeader>
              <CardTitle className="flex items-center">
                <Upload className="mr-3 h-6 w-6 text-primary" aria-hidden="true" />
                Secure File Upload
              </CardTitle>
              <CardDescription>
                Upload documents, receipts, and other materials securely through our encrypted system.
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="border-2 border-dashed border-border rounded-lg p-8 text-center">
                <Upload className="h-12 w-12 text-muted-foreground mx-auto mb-4" aria-hidden="true" />
                <p className="text-lg font-medium text-foreground mb-2">Drop files here or click to upload</p>
                <p className="text-sm text-muted-foreground mb-4">
                  Supports PDF, DOC, XLS, JPG, PNG files up to 10MB
                </p>
                <Button 
                  className="btn-primary"
                  data-testid="button-file-upload"
                >
                  Select Files
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Portal Access CTA */}
        <div className="text-center bg-muted/30 rounded-2xl p-12">
          <h2 className="text-3xl font-bold text-foreground mb-4">
            Need Access to the Client Portal?
          </h2>
          <p className="text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Contact us to set up your secure client portal account with personalized access 
            to your financial documents and training materials.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button 
              className="btn-primary"
              size="lg"
              onClick={() => window.location.href = "/contact?service=client-portal-access"}
              data-testid="button-request-access"
            >
              <Users className="mr-2 h-5 w-5" aria-hidden="true" />
              Request Portal Access
            </Button>
            <Button 
              variant="outline"
              className="btn-secondary"
              size="lg"
              data-testid="button-portal-demo"
            >
              <Calendar className="mr-2 h-5 w-5" aria-hidden="true" />
              Schedule Demo
            </Button>
          </div>
        </div>
      </div>

      {/* Video Player Dialog */}
      <Dialog open={!!selectedVideo} onOpenChange={() => setSelectedVideo(null)}>
        <DialogContent className="max-w-4xl">
          <DialogHeader>
            <DialogTitle>{selectedVideo?.title}</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <div className="aspect-video bg-black rounded-lg overflow-hidden">
              {selectedVideo?.videoUrl && (() => {
                const youtubeId = getYouTubeVideoId(selectedVideo.videoUrl);
                if (youtubeId) {
                  return (
                    <iframe
                      className="w-full h-full"
                      src={`https://www.youtube.com/embed/${youtubeId}`}
                      title={selectedVideo.title}
                      frameBorder="0"
                      allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                      allowFullScreen
                    />
                  );
                } else if (selectedVideo.videoUrl.includes('zoom.us')) {
                  return (
                    <iframe
                      className="w-full h-full"
                      src={selectedVideo.videoUrl}
                      title={selectedVideo.title}
                      frameBorder="0"
                      allow="microphone; camera"
                      allowFullScreen
                    />
                  );
                } else {
                  return (
                    <video className="w-full h-full" controls>
                      <source src={selectedVideo.videoUrl} />
                      Your browser does not support the video tag.
                    </video>
                  );
                }
              })()}
            </div>
            <div className="flex gap-2">
              <Button 
                variant="outline" 
                className="flex-1"
                onClick={() => toast({ title: "Captions", description: "Captions enabled for " + selectedVideo?.title })}
              >
                <FileText className="mr-2 h-4 w-4" />
                Toggle Captions
              </Button>
              <Button 
                variant="outline" 
                className="flex-1"
                onClick={() => toast({ title: "Download", description: "Downloading transcript for " + selectedVideo?.title })}
              >
                <Download className="mr-2 h-4 w-4" />
                Download Transcript
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
