import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Video, Plus, Youtube, Calendar, Pencil, Trash2 } from "lucide-react";
import type { TrainingVideo } from "@shared/schema";

export default function AdminVideos() {
  const { toast } = useToast();
  const [isAdding, setIsAdding] = useState(false);
  const [editingId, setEditingId] = useState<string | null>(null);
  const [formData, setFormData] = useState({
    title: "",
    description: "",
    videoUrl: "",
    category: "",
    requiredTier: "free",
    isPublic: true,
  });

  const { data: videos, isLoading } = useQuery<TrainingVideo[]>({
    queryKey: ["/api/training-videos"],
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.title || !formData.videoUrl || !formData.category) {
      toast({
        title: "Missing Information",
        description: "Please fill in all required fields",
        variant: "destructive",
      });
      return;
    }

    try {
      if (editingId) {
        await apiRequest("PATCH", `/api/training-videos/${editingId}`, formData);
        toast({
          title: "Video Updated",
          description: "Training video has been updated successfully",
        });
      } else {
        await apiRequest("POST", "/api/training-videos", formData);
        toast({
          title: "Video Added",
          description: "Training video has been added successfully",
        });
      }
      
      queryClient.invalidateQueries({ queryKey: ["/api/training-videos"] });

      setFormData({
        title: "",
        description: "",
        videoUrl: "",
        category: "",
        requiredTier: "free",
        isPublic: true,
      });
      setIsAdding(false);
      setEditingId(null);
    } catch (error: any) {
      console.error("Error saving video:", error);
      toast({
        title: "Error",
        description: error?.message || "Failed to save video. Please try again.",
        variant: "destructive",
      });
    }
  };

  const startEdit = (video: TrainingVideo) => {
    setFormData({
      title: video.title,
      description: video.description || "",
      videoUrl: video.videoUrl,
      category: video.category,
      requiredTier: video.requiredTier,
      isPublic: video.isPublic,
    });
    setEditingId(video.id);
    setIsAdding(true);
  };

  const handleDelete = async (id: string, title: string) => {
    if (!confirm(`Are you sure you want to delete "${title}"? This action cannot be undone.`)) {
      return;
    }

    try {
      await apiRequest("DELETE", `/api/training-videos/${id}`);
      
      queryClient.invalidateQueries({ queryKey: ["/api/training-videos"] });
      
      toast({
        title: "Video Deleted",
        description: "Training video has been deleted successfully",
      });
    } catch (error: any) {
      console.error("Error deleting video:", error);
      toast({
        title: "Error",
        description: error?.message || "Failed to delete video. Please try again.",
        variant: "destructive",
      });
    }
  };

  const cancelEdit = () => {
    setFormData({
      title: "",
      description: "",
      videoUrl: "",
      category: "",
      requiredTier: "free",
      isPublic: true,
    });
    setIsAdding(false);
    setEditingId(null);
  };

  const getVideoType = (url: string) => {
    if (url.includes('youtube.com') || url.includes('youtu.be')) return 'YouTube';
    if (url.includes('zoom.us')) return 'Zoom';
    return 'Video';
  };

  return (
    <div className="min-h-screen bg-gradient-to-b from-background to-muted/20">
      <div className="container mx-auto px-4 py-12 max-w-7xl">
        <div className="mb-8">
          <h1 className="text-4xl font-bold text-foreground mb-3">
            Training Video Management
          </h1>
          <p className="text-xl text-muted-foreground">
            Add and manage your YouTube videos and Zoom recordings
          </p>
        </div>

        {/* Add/Edit Video Form */}
        {isAdding ? (
          <Card className="mb-8">
            <CardHeader>
              <CardTitle>{editingId ? "Edit Training Video" : "Add New Training Video"}</CardTitle>
              <CardDescription>
                Paste the URL from YouTube or your Zoom cloud recording
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                  <Label htmlFor="title">Video Title *</Label>
                  <Input
                    id="title"
                    value={formData.title}
                    onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                    placeholder="e.g., Introduction to Government Contracting"
                    data-testid="input-video-title"
                  />
                </div>

                <div>
                  <Label htmlFor="videoUrl">Video URL * (YouTube or Zoom)</Label>
                  <Input
                    id="videoUrl"
                    value={formData.videoUrl}
                    onChange={(e) => setFormData({ ...formData, videoUrl: e.target.value })}
                    placeholder="https://www.youtube.com/watch?v=... or https://zoom.us/rec/..."
                    data-testid="input-video-url"
                  />
                  {formData.videoUrl && (
                    <p className="text-sm text-muted-foreground mt-1">
                      Type: {getVideoType(formData.videoUrl)}
                    </p>
                  )}
                </div>

                <div>
                  <Label htmlFor="description">Description</Label>
                  <Textarea
                    id="description"
                    value={formData.description}
                    onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                    placeholder="Brief description of the video content..."
                    data-testid="input-video-description"
                  />
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="category">Category *</Label>
                    <Input
                      id="category"
                      value={formData.category}
                      onChange={(e) => setFormData({ ...formData, category: e.target.value })}
                      placeholder="e.g., Government Contracting, Bookkeeping"
                      data-testid="input-video-category"
                    />
                  </div>

                  <div>
                    <Label htmlFor="tier">Access Level</Label>
                    <Select
                      value={formData.requiredTier}
                      onValueChange={(value) => setFormData({ ...formData, requiredTier: value })}
                    >
                      <SelectTrigger id="tier" data-testid="select-video-tier">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="free">Free (All Users)</SelectItem>
                        <SelectItem value="starter">Starter Tier</SelectItem>
                        <SelectItem value="professional">Professional Tier</SelectItem>
                        <SelectItem value="premium">Premium Tier Only</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>

                <div className="flex gap-3 pt-4">
                  <Button type="submit" className="btn-primary" data-testid="button-save-video">
                    {editingId ? (
                      <>
                        <Pencil className="mr-2 h-4 w-4" />
                        Update Video
                      </>
                    ) : (
                      <>
                        <Plus className="mr-2 h-4 w-4" />
                        Add Video
                      </>
                    )}
                  </Button>
                  <Button
                    type="button"
                    variant="outline"
                    onClick={cancelEdit}
                    data-testid="button-cancel-video"
                  >
                    Cancel
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>
        ) : (
          <Button
            onClick={() => setIsAdding(true)}
            className="btn-primary mb-8"
            size="lg"
            data-testid="button-add-new-video"
          >
            <Plus className="mr-2 h-5 w-5" />
            Add New Video
          </Button>
        )}

        {/* Video List */}
        <div>
          <h2 className="text-2xl font-bold text-foreground mb-4">
            Current Training Videos ({videos?.length || 0})
          </h2>
          
          {isLoading ? (
            <p className="text-muted-foreground">Loading videos...</p>
          ) : videos && videos.length > 0 ? (
            <div className="grid gap-4">
              {videos.map((video) => (
                <Card key={video.id}>
                  <CardContent className="p-6">
                    <div className="flex items-start gap-4">
                      <div className="p-3 bg-primary/10 rounded-lg">
                        {video.videoUrl.includes('youtube') ? (
                          <Youtube className="h-6 w-6 text-primary" />
                        ) : video.videoUrl.includes('zoom') ? (
                          <Calendar className="h-6 w-6 text-primary" />
                        ) : (
                          <Video className="h-6 w-6 text-primary" />
                        )}
                      </div>
                      <div className="flex-1">
                        <h3 className="text-lg font-semibold text-foreground mb-1">
                          {video.title}
                        </h3>
                        <p className="text-sm text-muted-foreground mb-2">
                          {video.description}
                        </p>
                        <div className="flex gap-4 text-sm text-muted-foreground">
                          <span>Category: {video.category}</span>
                          <span>•</span>
                          <span>Access: {video.requiredTier}</span>
                          <span>•</span>
                          <span>{getVideoType(video.videoUrl)}</span>
                        </div>
                        <p className="text-xs text-muted-foreground mt-2 break-all">
                          URL: {video.videoUrl}
                        </p>
                      </div>
                      <div className="flex gap-2">
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => startEdit(video)}
                          data-testid={`button-edit-video-${video.id}`}
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => handleDelete(video.id, video.title)}
                          data-testid={`button-delete-video-${video.id}`}
                        >
                          <Trash2 className="h-4 w-4 text-destructive" />
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : (
            <Card>
              <CardContent className="p-12 text-center">
                <Video className="h-16 w-16 text-muted-foreground mx-auto mb-4" />
                <h3 className="text-xl font-semibold text-foreground mb-2">
                  No Training Videos Yet
                </h3>
                <p className="text-muted-foreground mb-6">
                  Add your first training video from YouTube or Zoom
                </p>
                <Button
                  onClick={() => setIsAdding(true)}
                  className="btn-primary"
                  data-testid="button-add-first-video"
                >
                  <Plus className="mr-2 h-4 w-4" />
                  Add Your First Video
                </Button>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </div>
  );
}
