import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Progress } from "@/components/ui/progress";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Switch } from "@/components/ui/switch";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { 
  Video, 
  Mic, 
  MicOff, 
  Camera, 
  CameraOff,
  ScreenShare,
  Users,
  MessageCircle,
  BarChart3,
  Hand,
  Grid3X3,
  Settings,
  Download,
  Upload,
  Play,
  Square,
  Volume2,
  VolumeX,
  Monitor,
  UserPlus,
  HelpCircle,
  FileText,
  ThumbsUp,
  ThumbsDown,
  Share,
  StopCircle
} from "lucide-react";

interface Poll {
  id: string;
  question: string;
  options: string[];
  votes: Record<string, number>;
  isActive: boolean;
  createdBy: string;
  sessionId: string;
}

interface QAQuestion {
  id: string;
  question: string;
  askedBy: string;
  askedByName: string;
  answer?: string;
  answeredBy?: string;
  status: 'pending' | 'answered' | 'dismissed';
  upvotes: number;
  timestamp: string;
  sessionId: string;
}

interface BreakoutRoom {
  id: string;
  name: string;
  participants: string[];
  maxParticipants: number;
  sessionId: string;
  status: 'active' | 'inactive';
  topic?: string;
}

interface InteractiveTrainingSessionProps {
  sessionId: string;
  isInstructor: boolean;
  currentUserId: string;
}

export default function InteractiveTrainingSession({ 
  sessionId, 
  isInstructor, 
  currentUserId 
}: InteractiveTrainingSessionProps) {
  const [activeTab, setActiveTab] = useState("main");
  const [isRecording, setIsRecording] = useState(false);
  const [isMuted, setIsMuted] = useState(false);
  const [isVideoOff, setIsVideoOff] = useState(false);
  const [isScreenSharing, setIsScreenSharing] = useState(false);
  const [newPollQuestion, setNewPollQuestion] = useState("");
  const [newPollOptions, setNewPollOptions] = useState(["", ""]);
  const [newQAQuestion, setNewQAQuestion] = useState("");
  const [showCreatePoll, setShowCreatePoll] = useState(false);
  const [showBreakoutSetup, setShowBreakoutSetup] = useState(false);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch session polls
  const { data: polls = [], isLoading: pollsLoading } = useQuery({
    queryKey: ["/api/training/polls", sessionId],
    queryFn: () => apiRequest("GET", `/api/training/polls/${sessionId}`).then(res => res.json()),
    refetchInterval: 5000 // Refresh every 5 seconds
  });

  // Fetch Q&A questions
  const { data: qaQuestions = [], isLoading: qaLoading } = useQuery({
    queryKey: ["/api/training/qa", sessionId],
    queryFn: () => apiRequest("GET", `/api/training/qa/${sessionId}`).then(res => res.json()),
    refetchInterval: 3000 // Refresh every 3 seconds
  });

  // Fetch breakout rooms
  const { data: breakoutRooms = [], isLoading: roomsLoading } = useQuery({
    queryKey: ["/api/training/breakouts", sessionId],
    queryFn: () => apiRequest("GET", `/api/training/breakouts/${sessionId}`).then(res => res.json()),
    refetchInterval: 5000
  });

  // Create poll mutation
  const createPollMutation = useMutation({
    mutationFn: async (pollData: { question: string; options: string[] }) => {
      const response = await apiRequest("POST", `/api/training/polls/${sessionId}`, {
        ...pollData,
        createdBy: currentUserId
      });
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Poll Created",
        description: "Your poll is now live for participants to vote.",
      });
      setShowCreatePoll(false);
      setNewPollQuestion("");
      setNewPollOptions(["", ""]);
      queryClient.invalidateQueries({ queryKey: ["/api/training/polls", sessionId] });
    },
    onError: (error: any) => {
      toast({
        title: "Poll Creation Failed",
        description: error.message || "Failed to create poll.",
        variant: "destructive",
      });
    }
  });

  // Vote on poll mutation
  const voteOnPollMutation = useMutation({
    mutationFn: async ({ pollId, optionIndex }: { pollId: string; optionIndex: number }) => {
      const response = await apiRequest("POST", `/api/training/polls/${pollId}/vote`, {
        optionIndex,
        userId: currentUserId
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/training/polls", sessionId] });
    }
  });

  // Submit Q&A question mutation
  const submitQuestionMutation = useMutation({
    mutationFn: async (question: string) => {
      const response = await apiRequest("POST", `/api/training/qa/${sessionId}`, {
        question,
        askedBy: currentUserId
      });
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Question Submitted",
        description: "Your question has been submitted and will be answered soon.",
      });
      setNewQAQuestion("");
      queryClient.invalidateQueries({ queryKey: ["/api/training/qa", sessionId] });
    }
  });

  // Answer question mutation (instructor only)
  const answerQuestionMutation = useMutation({
    mutationFn: async ({ questionId, answer }: { questionId: string; answer: string }) => {
      const response = await apiRequest("PUT", `/api/training/qa/${questionId}/answer`, {
        answer,
        answeredBy: currentUserId
      });
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/training/qa", sessionId] });
    }
  });

  // Create breakout rooms mutation
  const createBreakoutMutation = useMutation({
    mutationFn: async (roomData: { name: string; maxParticipants: number; topic?: string }) => {
      const response = await apiRequest("POST", `/api/training/breakouts/${sessionId}`, roomData);
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Breakout Room Created",
        description: "Participants can now join the breakout room.",
      });
      setShowBreakoutSetup(false);
      queryClient.invalidateQueries({ queryKey: ["/api/training/breakouts", sessionId] });
    }
  });

  const handleCreatePoll = () => {
    if (!newPollQuestion.trim()) {
      toast({
        title: "Question Required",
        description: "Please enter a poll question.",
        variant: "destructive",
      });
      return;
    }

    const validOptions = newPollOptions.filter(option => option.trim());
    if (validOptions.length < 2) {
      toast({
        title: "Options Required",
        description: "Please provide at least 2 poll options.",
        variant: "destructive",
      });
      return;
    }

    createPollMutation.mutate({
      question: newPollQuestion,
      options: validOptions
    });
  };

  const handleSubmitQuestion = () => {
    if (!newQAQuestion.trim()) {
      toast({
        title: "Question Required",
        description: "Please enter your question.",
        variant: "destructive",
      });
      return;
    }

    submitQuestionMutation.mutate(newQAQuestion);
  };

  const addPollOption = () => {
    setNewPollOptions([...newPollOptions, ""]);
  };

  const updatePollOption = (index: number, value: string) => {
    const updated = [...newPollOptions];
    updated[index] = value;
    setNewPollOptions(updated);
  };

  const removePollOption = (index: number) => {
    if (newPollOptions.length > 2) {
      setNewPollOptions(newPollOptions.filter((_, i) => i !== index));
    }
  };

  return (
    <div className="space-y-6">
      {/* Session Controls */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Video className="w-5 h-5" />
            Session Controls
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap gap-4">
            {/* Audio/Video Controls */}
            <div className="flex gap-2">
              <Button
                variant={isMuted ? "destructive" : "outline"}
                size="sm"
                onClick={() => setIsMuted(!isMuted)}
                data-testid="toggle-mute"
              >
                {isMuted ? <MicOff className="w-4 h-4" /> : <Mic className="w-4 h-4" />}
                {isMuted ? "Unmute" : "Mute"}
              </Button>
              
              <Button
                variant={isVideoOff ? "destructive" : "outline"}
                size="sm"
                onClick={() => setIsVideoOff(!isVideoOff)}
                data-testid="toggle-video"
              >
                {isVideoOff ? <CameraOff className="w-4 h-4" /> : <Camera className="w-4 h-4" />}
                {isVideoOff ? "Start Video" : "Stop Video"}
              </Button>
            </div>

            {/* Screen Sharing */}
            {isInstructor && (
              <Button
                variant={isScreenSharing ? "default" : "outline"}
                size="sm"
                onClick={() => setIsScreenSharing(!isScreenSharing)}
                data-testid="toggle-screen-share"
              >
                <ScreenShare className="w-4 h-4 mr-2" />
                {isScreenSharing ? "Stop Sharing" : "Share Screen"}
              </Button>
            )}

            {/* Recording */}
            {isInstructor && (
              <Button
                variant={isRecording ? "destructive" : "outline"}
                size="sm"
                onClick={() => setIsRecording(!isRecording)}
                data-testid="toggle-recording"
              >
                {isRecording ? <StopCircle className="w-4 h-4 mr-2" /> : <Record className="w-4 h-4 mr-2" />}
                {isRecording ? "Stop Recording" : "Start Recording"}
              </Button>
            )}

            {/* Participants */}
            <Button variant="outline" size="sm">
              <Users className="w-4 h-4 mr-2" />
              Participants (12)
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Interactive Features Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="grid w-full grid-cols-5">
          <TabsTrigger value="main" data-testid="tab-main">Main Session</TabsTrigger>
          <TabsTrigger value="polls" data-testid="tab-polls">Polls</TabsTrigger>
          <TabsTrigger value="qa" data-testid="tab-qa">Q&A</TabsTrigger>
          <TabsTrigger value="breakouts" data-testid="tab-breakouts">Breakout Rooms</TabsTrigger>
          <TabsTrigger value="resources" data-testid="tab-resources">Resources</TabsTrigger>
        </TabsList>

        <TabsContent value="main" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Main Training Session</CardTitle>
              <CardDescription>
                Primary video conference and presentation area
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="aspect-video bg-gray-900 rounded-lg flex items-center justify-center text-white">
                <div className="text-center">
                  <Video className="w-16 h-16 mx-auto mb-4 opacity-50" />
                  <p className="text-lg">Main Video Conference</p>
                  <p className="text-sm opacity-75">Zoom integration would appear here</p>
                </div>
              </div>
              
              {/* Chat Section */}
              <div className="mt-6">
                <h4 className="font-semibold mb-3 flex items-center gap-2">
                  <MessageCircle className="w-4 h-4" />
                  Session Chat
                </h4>
                <div className="border rounded-lg p-4 h-32 overflow-y-auto bg-gray-50">
                  <div className="space-y-2 text-sm">
                    <p><span className="font-medium">Instructor:</span> Welcome everyone to today's session!</p>
                    <p><span className="font-medium">John D:</span> Thanks for having us</p>
                    <p><span className="font-medium">Sarah M:</span> Looking forward to learning about compliance</p>
                  </div>
                </div>
                <div className="mt-2 flex gap-2">
                  <Input placeholder="Type your message..." className="flex-1" />
                  <Button size="sm">Send</Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="polls" className="space-y-6">
          {/* Create Poll (Instructor Only) */}
          {isInstructor && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  Create Interactive Poll
                  <Dialog open={showCreatePoll} onOpenChange={setShowCreatePoll}>
                    <DialogTrigger asChild>
                      <Button size="sm" data-testid="create-poll-button">
                        <BarChart3 className="w-4 h-4 mr-2" />
                        New Poll
                      </Button>
                    </DialogTrigger>
                    <DialogContent>
                      <DialogHeader>
                        <DialogTitle>Create New Poll</DialogTitle>
                        <DialogDescription>
                          Create an interactive poll for your training participants
                        </DialogDescription>
                      </DialogHeader>
                      <div className="space-y-4">
                        <div>
                          <Label htmlFor="poll-question">Poll Question</Label>
                          <Input
                            id="poll-question"
                            value={newPollQuestion}
                            onChange={(e) => setNewPollQuestion(e.target.value)}
                            placeholder="What is your experience level with Section 508?"
                            data-testid="poll-question-input"
                          />
                        </div>
                        
                        <div>
                          <Label>Poll Options</Label>
                          <div className="space-y-2">
                            {newPollOptions.map((option, index) => (
                              <div key={index} className="flex gap-2">
                                <Input
                                  value={option}
                                  onChange={(e) => updatePollOption(index, e.target.value)}
                                  placeholder={`Option ${index + 1}`}
                                  data-testid={`poll-option-${index}`}
                                />
                                {newPollOptions.length > 2 && (
                                  <Button
                                    variant="outline"
                                    size="sm"
                                    onClick={() => removePollOption(index)}
                                  >
                                    Remove
                                  </Button>
                                )}
                              </div>
                            ))}
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={addPollOption}
                              className="w-full"
                            >
                              Add Option
                            </Button>
                          </div>
                        </div>
                        
                        <div className="flex gap-2">
                          <Button
                            onClick={handleCreatePoll}
                            disabled={createPollMutation.isPending}
                            className="flex-1"
                            data-testid="submit-poll"
                          >
                            {createPollMutation.isPending ? "Creating..." : "Create Poll"}
                          </Button>
                          <Button
                            variant="outline"
                            onClick={() => setShowCreatePoll(false)}
                            className="flex-1"
                          >
                            Cancel
                          </Button>
                        </div>
                      </div>
                    </DialogContent>
                  </Dialog>
                </CardTitle>
              </CardHeader>
            </Card>
          )}

          {/* Active Polls */}
          <div className="space-y-4">
            {pollsLoading ? (
              <Card>
                <CardContent className="pt-6">
                  <div className="text-center">Loading polls...</div>
                </CardContent>
              </Card>
            ) : polls.length === 0 ? (
              <Card>
                <CardContent className="pt-6">
                  <div className="text-center">
                    <BarChart3 className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                    <h3 className="text-lg font-semibold mb-2">No Active Polls</h3>
                    <p className="text-gray-600">
                      {isInstructor 
                        ? "Create a poll to engage with your participants"
                        : "Wait for the instructor to create polls"
                      }
                    </p>
                  </div>
                </CardContent>
              </Card>
            ) : (
              polls.map((poll: Poll) => (
                <Card key={poll.id} className="border-l-4 border-l-blue-500" data-testid={`poll-${poll.id}`}>
                  <CardHeader>
                    <CardTitle className="text-lg">{poll.question}</CardTitle>
                    <CardDescription>
                      {poll.isActive ? "Active poll - cast your vote!" : "Poll closed"}
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-3">
                      {poll.options.map((option, index) => {
                        const votes = poll.votes[index] || 0;
                        const totalVotes = Object.values(poll.votes).reduce((sum: number, v: number) => sum + v, 0);
                        const percentage = totalVotes > 0 ? (votes / totalVotes) * 100 : 0;
                        
                        return (
                          <div key={index} className="space-y-2">
                            <div className="flex items-center justify-between">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => voteOnPollMutation.mutate({ pollId: poll.id, optionIndex: index })}
                                disabled={!poll.isActive || voteOnPollMutation.isPending}
                                className="flex-1 justify-start"
                                data-testid={`vote-option-${index}`}
                              >
                                <ThumbsUp className="w-4 h-4 mr-2" />
                                {option}
                              </Button>
                              <Badge variant="secondary">
                                {votes} votes ({percentage.toFixed(0)}%)
                              </Badge>
                            </div>
                            <Progress value={percentage} className="h-2" />
                          </div>
                        );
                      })}
                    </div>
                  </CardContent>
                </Card>
              ))
            )}
          </div>
        </TabsContent>

        <TabsContent value="qa" className="space-y-6">
          {/* Submit Question */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <HelpCircle className="w-5 h-5" />
                Ask a Question
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <Textarea
                  value={newQAQuestion}
                  onChange={(e) => setNewQAQuestion(e.target.value)}
                  placeholder="Type your question about the training content..."
                  rows={3}
                  data-testid="qa-question-input"
                />
                <Button
                  onClick={handleSubmitQuestion}
                  disabled={submitQuestionMutation.isPending}
                  data-testid="submit-question"
                >
                  {submitQuestionMutation.isPending ? "Submitting..." : "Submit Question"}
                </Button>
              </div>
            </CardContent>
          </Card>

          {/* Q&A Questions */}
          <div className="space-y-4">
            {qaLoading ? (
              <Card>
                <CardContent className="pt-6">
                  <div className="text-center">Loading questions...</div>
                </CardContent>
              </Card>
            ) : qaQuestions.length === 0 ? (
              <Card>
                <CardContent className="pt-6">
                  <div className="text-center">
                    <MessageCircle className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                    <h3 className="text-lg font-semibold mb-2">No Questions Yet</h3>
                    <p className="text-gray-600">
                      Be the first to ask a question about the training content
                    </p>
                  </div>
                </CardContent>
              </Card>
            ) : (
              qaQuestions.map((question: QAQuestion) => (
                <Card key={question.id} className={`border-l-4 ${
                  question.status === 'answered' ? 'border-l-green-500' : 
                  question.status === 'pending' ? 'border-l-yellow-500' : 'border-l-gray-500'
                }`} data-testid={`qa-question-${question.id}`}>
                  <CardHeader>
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <CardTitle className="text-lg">{question.question}</CardTitle>
                        <CardDescription>
                          Asked by {question.askedByName} • {new Date(question.timestamp).toLocaleString()}
                        </CardDescription>
                      </div>
                      <div className="flex items-center gap-2">
                        <Badge variant={
                          question.status === 'answered' ? 'default' : 
                          question.status === 'pending' ? 'secondary' : 'destructive'
                        }>
                          {question.status}
                        </Badge>
                        <Button variant="ghost" size="sm">
                          <ThumbsUp className="w-4 h-4 mr-1" />
                          {question.upvotes}
                        </Button>
                      </div>
                    </div>
                  </CardHeader>
                  {question.answer && (
                    <CardContent>
                      <div className="bg-green-50 border-l-4 border-l-green-400 p-4 rounded">
                        <p className="font-medium text-green-800 mb-2">Answer:</p>
                        <p className="text-green-700">{question.answer}</p>
                      </div>
                    </CardContent>
                  )}
                </Card>
              ))
            )}
          </div>
        </TabsContent>

        <TabsContent value="breakouts" className="space-y-6">
          {/* Create Breakout Room (Instructor Only) */}
          {isInstructor && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  Breakout Room Management
                  <Dialog open={showBreakoutSetup} onOpenChange={setShowBreakoutSetup}>
                    <DialogTrigger asChild>
                      <Button size="sm" data-testid="create-breakout-button">
                        <Grid3X3 className="w-4 h-4 mr-2" />
                        Create Room
                      </Button>
                    </DialogTrigger>
                    <DialogContent>
                      <DialogHeader>
                        <DialogTitle>Create Breakout Room</DialogTitle>
                        <DialogDescription>
                          Set up small group collaboration spaces
                        </DialogDescription>
                      </DialogHeader>
                      <div className="space-y-4">
                        <div>
                          <Label htmlFor="room-name">Room Name</Label>
                          <Input
                            id="room-name"
                            placeholder="Section 508 Discussion Group"
                            data-testid="room-name-input"
                          />
                        </div>
                        <div>
                          <Label htmlFor="max-participants">Maximum Participants</Label>
                          <Select>
                            <SelectTrigger data-testid="max-participants-select">
                              <SelectValue placeholder="Select size" />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem value="3">3 participants</SelectItem>
                              <SelectItem value="5">5 participants</SelectItem>
                              <SelectItem value="8">8 participants</SelectItem>
                              <SelectItem value="10">10 participants</SelectItem>
                            </SelectContent>
                          </Select>
                        </div>
                        <div>
                          <Label htmlFor="room-topic">Discussion Topic (Optional)</Label>
                          <Textarea
                            id="room-topic"
                            placeholder="Discuss real-world Section 508 compliance challenges"
                            rows={2}
                            data-testid="room-topic-input"
                          />
                        </div>
                        <div className="flex gap-2">
                          <Button className="flex-1" data-testid="create-room-submit">
                            Create Room
                          </Button>
                          <Button
                            variant="outline"
                            className="flex-1"
                            onClick={() => setShowBreakoutSetup(false)}
                          >
                            Cancel
                          </Button>
                        </div>
                      </div>
                    </DialogContent>
                  </Dialog>
                </CardTitle>
              </CardHeader>
            </Card>
          )}

          {/* Breakout Rooms List */}
          <div className="space-y-4">
            {roomsLoading ? (
              <Card>
                <CardContent className="pt-6">
                  <div className="text-center">Loading breakout rooms...</div>
                </CardContent>
              </Card>
            ) : breakoutRooms.length === 0 ? (
              <Card>
                <CardContent className="pt-6">
                  <div className="text-center">
                    <Grid3X3 className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                    <h3 className="text-lg font-semibold mb-2">No Breakout Rooms</h3>
                    <p className="text-gray-600">
                      {isInstructor 
                        ? "Create breakout rooms for small group discussions"
                        : "Wait for the instructor to create breakout rooms"
                      }
                    </p>
                  </div>
                </CardContent>
              </Card>
            ) : (
              breakoutRooms.map((room: BreakoutRoom) => (
                <Card key={room.id} className="border-l-4 border-l-purple-500" data-testid={`breakout-room-${room.id}`}>
                  <CardHeader>
                    <div className="flex items-center justify-between">
                      <div>
                        <CardTitle className="text-lg">{room.name}</CardTitle>
                        <CardDescription>
                          {room.participants.length}/{room.maxParticipants} participants
                          {room.topic && ` • Topic: ${room.topic}`}
                        </CardDescription>
                      </div>
                      <div className="flex items-center gap-2">
                        <Badge variant={room.status === 'active' ? 'default' : 'secondary'}>
                          {room.status}
                        </Badge>
                        <Button size="sm" data-testid={`join-room-${room.id}`}>
                          <UserPlus className="w-4 h-4 mr-2" />
                          Join Room
                        </Button>
                      </div>
                    </div>
                  </CardHeader>
                  <CardContent>
                    <Progress 
                      value={(room.participants.length / room.maxParticipants) * 100} 
                      className="h-2"
                    />
                  </CardContent>
                </Card>
              ))
            )}
          </div>
        </TabsContent>

        <TabsContent value="resources" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileText className="w-5 h-5" />
                Session Resources
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {/* Shared Files */}
                <div>
                  <h4 className="font-semibold mb-3">Shared Documents</h4>
                  <div className="space-y-2">
                    <div className="flex items-center justify-between p-3 border rounded-lg">
                      <div className="flex items-center gap-3">
                        <FileText className="w-5 h-5 text-blue-500" />
                        <div>
                          <p className="font-medium">Section 508 Compliance Checklist</p>
                          <p className="text-sm text-gray-600">PDF • 2.3 MB</p>
                        </div>
                      </div>
                      <Button size="sm" variant="outline">
                        <Download className="w-4 h-4 mr-2" />
                        Download
                      </Button>
                    </div>
                    
                    <div className="flex items-center justify-between p-3 border rounded-lg">
                      <div className="flex items-center gap-3">
                        <FileText className="w-5 h-5 text-green-500" />
                        <div>
                          <p className="font-medium">WCAG 2.1 Quick Reference</p>
                          <p className="text-sm text-gray-600">PDF • 1.8 MB</p>
                        </div>
                      </div>
                      <Button size="sm" variant="outline">
                        <Download className="w-4 h-4 mr-2" />
                        Download
                      </Button>
                    </div>
                  </div>
                </div>

                {/* Upload Resources (Instructor Only) */}
                {isInstructor && (
                  <div>
                    <h4 className="font-semibold mb-3">Share Resources</h4>
                    <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                      <Upload className="w-8 h-8 mx-auto text-gray-400 mb-2" />
                      <p className="text-gray-600 mb-2">Drag and drop files here, or click to browse</p>
                      <Button variant="outline" size="sm">
                        Choose Files
                      </Button>
                    </div>
                  </div>
                )}

                {/* Quick Links */}
                <div>
                  <h4 className="font-semibold mb-3">Quick Links</h4>
                  <div className="space-y-2">
                    <Button variant="outline" className="w-full justify-start">
                      <Share className="w-4 h-4 mr-2" />
                      Section 508.gov Official Website
                    </Button>
                    <Button variant="outline" className="w-full justify-start">
                      <Share className="w-4 h-4 mr-2" />
                      WCAG 2.1 Guidelines
                    </Button>
                    <Button variant="outline" className="w-full justify-start">
                      <Share className="w-4 h-4 mr-2" />
                      TMH Global Compliance Tools
                    </Button>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}