import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { 
  Check, 
  X, 
  Star,
  Calculator,
  Users,
  TrendingUp,
  Search,
  Rocket,
  FileText,
  ArrowRight,
  Shield,
  Crown,
  Zap
} from "lucide-react";

interface ServiceFeature {
  name: string;
  basic: boolean | string;
  standard: boolean | string;
  premium: boolean | string;
  description?: string;
}

interface ServiceComparison {
  id: string;
  name: string;
  icon: typeof Calculator;
  description: string;
  basicPrice: string;
  standardPrice: string;
  premiumPrice: string;
  features: ServiceFeature[];
}

const serviceComparisons: ServiceComparison[] = [
  {
    id: "bookkeeping",
    name: "Bookkeeping Services",
    icon: Calculator,
    description: "Professional financial record keeping and management",
    basicPrice: "$500/mo",
    standardPrice: "$850/mo",
    premiumPrice: "$1,500/mo",
    features: [
      { name: "Monthly Reconciliation", basic: true, standard: true, premium: true },
      { name: "Financial Statements", basic: "Basic", standard: "Advanced", premium: "Real-time" },
      { name: "Expense Tracking", basic: true, standard: true, premium: true },
      { name: "QuickBooks Setup", basic: true, standard: true, premium: true },
      { name: "Support Level", basic: "Email", standard: "Phone + Email", premium: "Dedicated Manager" },
      { name: "Reconciliation Frequency", basic: "Monthly", standard: "Weekly", premium: "Real-time" },
      { name: "Custom Reporting", basic: false, standard: true, premium: true },
      { name: "Tax Prep Assistance", basic: false, standard: true, premium: true },
      { name: "Government Compliance", basic: false, standard: false, premium: true },
      { name: "SOC 2 Assistance", basic: false, standard: false, premium: true }
    ]
  },
  {
    id: "hr",
    name: "HR Services",
    icon: Users,
    description: "Comprehensive human resources support",
    basicPrice: "Custom",
    standardPrice: "Custom",
    premiumPrice: "Custom",
    features: [
      { name: "Payroll Processing", basic: true, standard: true, premium: true },
      { name: "Benefits Administration", basic: "Basic", standard: "Full", premium: "Full + Federal" },
      { name: "Employee Handbook", basic: "Template", standard: "Custom", premium: "Federal Compliant" },
      { name: "Support Level", basic: "Email", standard: "Phone + Email", premium: "HR Specialist" },
      { name: "Performance Management", basic: false, standard: true, premium: true },
      { name: "Training Programs", basic: false, standard: "Custom", premium: "Federal + Custom" },
      { name: "Recruitment Help", basic: false, standard: true, premium: true },
      { name: "Federal Compliance", basic: false, standard: false, premium: true },
      { name: "Security Clearance Tracking", basic: false, standard: false, premium: true },
      { name: "EEO-1 Reporting", basic: false, standard: false, premium: true }
    ]
  },
  {
    id: "cfo",
    name: "CFO Services",
    icon: TrendingUp,
    description: "Strategic financial leadership and planning",
    basicPrice: "$2,500/mo",
    standardPrice: "$4,500/mo",
    premiumPrice: "$7,500/mo",
    features: [
      { name: "Financial Reviews", basic: "Monthly", standard: "Weekly", premium: "On-demand" },
      { name: "Budget Planning", basic: true, standard: true, premium: true },
      { name: "Cash Flow Management", basic: true, standard: true, premium: true },
      { name: "Strategy Sessions", basic: "Quarterly", standard: "Monthly", premium: "Weekly" },
      { name: "Board Presentations", basic: false, standard: true, premium: true },
      { name: "Investment Planning", basic: false, standard: true, premium: true },
      { name: "Risk Management", basic: false, standard: true, premium: true },
      { name: "CFO Access", basic: "Email", standard: "Direct", premium: "24/7 Direct" },
      { name: "Government Compliance", basic: false, standard: false, premium: true },
      { name: "DCAA-Ready Systems", basic: false, standard: false, premium: true }
    ]
  }
];

const renderFeatureValue = (value: boolean | string) => {
  if (value === true) {
    return <Check className="h-4 w-4 text-green-600" />;
  }
  if (value === false) {
    return <X className="h-4 w-4 text-gray-400" />;
  }
  return <span className="text-sm text-foreground">{value}</span>;
};

interface ServiceComparisonProps {
  selectedServices?: string[];
  onSelectService?: (serviceId: string, tier: string) => void;
}

export default function ServiceComparison({ selectedServices = [], onSelectService }: ServiceComparisonProps) {
  const [activeComparison, setActiveComparison] = useState(serviceComparisons[0].id);
  const [selectedTiers, setSelectedTiers] = useState<Record<string, string>>({});

  const handleSelectTier = (serviceId: string, tier: string) => {
    setSelectedTiers(prev => ({ ...prev, [serviceId]: tier }));
    onSelectService?.(serviceId, tier);
  };

  const getCurrentService = () => {
    return serviceComparisons.find(s => s.id === activeComparison) || serviceComparisons[0];
  };

  const service = getCurrentService();

  return (
    <div className="bg-background">
      <div className="text-center mb-8">
        <h3 className="text-2xl font-bold text-foreground mb-2">Compare Service Packages</h3>
        <p className="text-muted-foreground">Choose the right service level for your business needs</p>
      </div>

      <Tabs value={activeComparison} onValueChange={setActiveComparison} className="w-full">
        <TabsList className="grid w-full grid-cols-3 mb-8">
          {serviceComparisons.map((service) => {
            const Icon = service.icon;
            return (
              <TabsTrigger 
                key={service.id} 
                value={service.id}
                className="flex flex-col items-center p-4 h-auto"
                data-testid={`comparison-tab-${service.id}`}
              >
                <Icon className="h-5 w-5 mb-2" />
                <span className="text-sm font-medium">{service.name}</span>
              </TabsTrigger>
            );
          })}
        </TabsList>

        {serviceComparisons.map((service) => (
          <TabsContent key={service.id} value={service.id}>
            <Card>
              <CardHeader className="text-center">
                <div className="flex items-center justify-center mb-4">
                  <service.icon className="h-8 w-8 text-primary mr-3" />
                  <div>
                    <CardTitle className="text-2xl">{service.name}</CardTitle>
                    <p className="text-muted-foreground mt-2">{service.description}</p>
                  </div>
                </div>
              </CardHeader>

              <CardContent>
                <div className="overflow-x-auto">
                  <table className="w-full">
                    <thead>
                      <tr className="border-b">
                        <th className="text-left py-4 px-4 font-semibold">Features</th>
                        <th className="text-center py-4 px-4">
                          <div className="flex flex-col items-center">
                            <Shield className="h-6 w-6 text-primary mb-2" />
                            <span className="font-semibold">Essential</span>
                            <span className="text-sm text-muted-foreground">{service.basicPrice}</span>
                          </div>
                        </th>
                        <th className="text-center py-4 px-4 relative">
                          <Badge className="absolute -top-2 left-1/2 transform -translate-x-1/2 bg-primary">
                            <Star className="h-3 w-3 mr-1" />
                            Popular
                          </Badge>
                          <div className="flex flex-col items-center mt-4">
                            <Star className="h-6 w-6 text-primary mb-2" />
                            <span className="font-semibold">Professional</span>
                            <span className="text-sm text-muted-foreground">{service.standardPrice}</span>
                          </div>
                        </th>
                        <th className="text-center py-4 px-4 relative">
                          <Badge className="absolute -top-2 left-1/2 transform -translate-x-1/2 bg-accent">
                            <Shield className="h-3 w-3 mr-1" />
                            Gov Ready
                          </Badge>
                          <div className="flex flex-col items-center mt-4">
                            <Crown className="h-6 w-6 text-primary mb-2" />
                            <span className="font-semibold">Enterprise</span>
                            <span className="text-sm text-muted-foreground">{service.premiumPrice}</span>
                          </div>
                        </th>
                      </tr>
                    </thead>
                    <tbody>
                      {service.features.map((feature, index) => (
                        <tr key={index} className="border-b hover:bg-muted/50">
                          <td className="py-3 px-4 font-medium">{feature.name}</td>
                          <td className="py-3 px-4 text-center">
                            {renderFeatureValue(feature.basic)}
                          </td>
                          <td className="py-3 px-4 text-center">
                            {renderFeatureValue(feature.standard)}
                          </td>
                          <td className="py-3 px-4 text-center">
                            {renderFeatureValue(feature.premium)}
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>

                <div className="grid md:grid-cols-3 gap-4 mt-8">
                  {['basic', 'standard', 'premium'].map((tier) => {
                    const isSelected = selectedTiers[service.id] === tier;
                    const tierName = tier === 'basic' ? 'Essential' : 
                                   tier === 'standard' ? 'Professional' : 'Enterprise';
                    
                    return (
                      <Button
                        key={tier}
                        variant={isSelected ? "default" : "outline"}
                        className={`w-full ${isSelected ? 'btn-primary' : ''}`}
                        onClick={() => handleSelectTier(service.id, tier)}
                        data-testid={`button-select-comparison-${service.id}-${tier}`}
                      >
                        {isSelected ? (
                          <>
                            <Check className="mr-2 h-4 w-4" />
                            Selected: {tierName}
                          </>
                        ) : (
                          <>
                            Select {tierName}
                            <ArrowRight className="ml-2 h-4 w-4" />
                          </>
                        )}
                      </Button>
                    );
                  })}
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        ))}
      </Tabs>

      {Object.keys(selectedTiers).length > 0 && (
        <div className="mt-8 p-6 bg-primary/5 rounded-lg border border-primary/20">
          <div className="text-center">
            <h4 className="text-lg font-semibold text-foreground mb-2">Your Selected Services</h4>
            <div className="flex flex-wrap gap-2 justify-center mb-4">
              {Object.entries(selectedTiers).map(([serviceId, tier]) => {
                const service = serviceComparisons.find(s => s.id === serviceId);
                const tierName = tier === 'basic' ? 'Essential' : 
                               tier === 'standard' ? 'Professional' : 'Enterprise';
                return (
                  <Badge key={serviceId} variant="outline" className="px-3 py-1">
                    {service?.name} - {tierName}
                  </Badge>
                );
              })}
            </div>
            <Button 
              className="btn-primary"
              data-testid="button-contact-selected-services"
            >
              <Zap className="mr-2 h-4 w-4" />
              Get Quote for Selected Services
            </Button>
          </div>
        </div>
      )}
    </div>
  );
}