import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { 
  Check, 
  Star, 
  ArrowRight,
  Shield,
  Crown,
  Zap
} from "lucide-react";

interface PricingTier {
  id: string;
  name: string;
  icon: typeof Shield;
  price: string;
  billing: string;
  description: string;
  features: string[];
  popular?: boolean;
  governmentReady?: boolean;
}

interface PricingTiersProps {
  serviceId: string;
  serviceTitle: string;
  onSelectTier: (serviceId: string, tier: string) => void;
}

const getPricingTiers = (serviceId: string): PricingTier[] => {
  const tiersByService: Record<string, PricingTier[]> = {
    bookkeeping: [
      {
        id: "basic",
        name: "Essential",
        icon: Shield,
        price: "$500",
        billing: "per month",
        description: "Core bookkeeping for small businesses",
        features: [
          "Monthly reconciliation",
          "Basic financial statements",
          "Expense tracking",
          "Email support",
          "QuickBooks setup"
        ]
      },
      {
        id: "standard",
        name: "Professional",
        icon: Star,
        price: "$850",
        billing: "per month",
        description: "Comprehensive bookkeeping with enhanced reporting",
        features: [
          "Everything in Essential",
          "Weekly reconciliation",
          "Advanced financial analysis",
          "Phone support",
          "Custom reporting",
          "Tax preparation assistance"
        ],
        popular: true
      },
      {
        id: "premium",
        name: "Enterprise",
        icon: Crown,
        price: "$1,500",
        billing: "per month",
        description: "Full-service bookkeeping with compliance support",
        features: [
          "Everything in Professional",
          "Real-time reconciliation",
          "Government compliance reporting",
          "Dedicated account manager",
          "Priority support",
          "SOC 2 compliance assistance"
        ],
        governmentReady: true
      }
    ],
    hr: [
      {
        id: "basic",
        name: "Starter",
        icon: Shield,
        price: "Custom",
        billing: "per employee",
        description: "Essential HR services for growing teams",
        features: [
          "Payroll processing",
          "Basic benefits administration",
          "Employee handbook template",
          "Email support",
          "Compliance alerts"
        ]
      },
      {
        id: "standard",
        name: "Growth",
        icon: Star,
        price: "Custom",
        billing: "per employee",
        description: "Comprehensive HR with performance management",
        features: [
          "Everything in Starter",
          "Performance management system",
          "Custom training programs",
          "Recruitment assistance",
          "Phone support",
          "HR analytics"
        ],
        popular: true
      },
      {
        id: "premium",
        name: "Federal Ready",
        icon: Crown,
        price: "Custom",
        billing: "per employee",
        description: "Government contractor HR compliance",
        features: [
          "Everything in Growth",
          "Federal compliance monitoring",
          "Security clearance tracking",
          "EEO-1 reporting",
          "Dedicated HR specialist",
          "Government contract labor standards"
        ],
        governmentReady: true
      }
    ],
    cfo: [
      {
        id: "basic",
        name: "Advisory",
        icon: Shield,
        price: "$2,500",
        billing: "per month",
        description: "Strategic financial guidance",
        features: [
          "Monthly financial reviews",
          "Budget planning",
          "Cash flow management",
          "Quarterly strategy sessions",
          "Email support"
        ]
      },
      {
        id: "standard",
        name: "Executive",
        icon: Star,
        price: "$4,500",
        billing: "per month",
        description: "Comprehensive CFO services",
        features: [
          "Everything in Advisory",
          "Weekly financial analysis",
          "Board presentation prep",
          "Investment planning",
          "Risk management",
          "Direct access to CFO"
        ],
        popular: true
      },
      {
        id: "premium",
        name: "Government CFO",
        icon: Crown,
        price: "$7,500",
        billing: "per month",
        description: "Specialized government contracting CFO",
        features: [
          "Everything in Executive",
          "Government compliance oversight",
          "Contract financial management",
          "DCAA-ready accounting systems",
          "Proposal cost analysis",
          "Federal audit support"
        ],
        governmentReady: true
      }
    ],
    auditing: [
      {
        id: "basic",
        name: "Review",
        icon: Shield,
        price: "$5,000",
        billing: "per project",
        description: "Basic financial review and assessment",
        features: [
          "Financial statement review",
          "Basic internal controls assessment",
          "Written report with findings",
          "Email support",
          "30-day follow-up"
        ]
      },
      {
        id: "standard",
        name: "Comprehensive",
        icon: Star,
        price: "$12,000",
        billing: "per project",
        description: "Full independent audit",
        features: [
          "Everything in Review",
          "Complete financial audit",
          "Detailed risk assessment",
          "Management letter",
          "Compliance testing",
          "Phone support throughout"
        ],
        popular: true
      },
      {
        id: "premium",
        name: "Government Ready",
        icon: Crown,
        price: "$25,000",
        billing: "per project",
        description: "Government contractor audit readiness",
        features: [
          "Everything in Comprehensive",
          "DCAA compliance audit",
          "Government contract requirements",
          "Federal audit preparation",
          "Ongoing audit support",
          "Priority response team"
        ],
        governmentReady: true
      }
    ],
    "business-development": [
      {
        id: "basic",
        name: "Planning",
        icon: Shield,
        price: "$1,500",
        billing: "per month",
        description: "Strategic business planning",
        features: [
          "Business plan development",
          "Market research",
          "Financial modeling",
          "Monthly strategy review",
          "Email support"
        ]
      },
      {
        id: "standard",
        name: "Growth",
        icon: Star,
        price: "$3,500",
        billing: "per month",
        description: "Comprehensive growth strategy",
        features: [
          "Everything in Planning",
          "Competitive analysis",
          "Partnership development",
          "Investor preparation",
          "Weekly progress reviews",
          "Direct consultant access"
        ],
        popular: true
      },
      {
        id: "premium",
        name: "Federal Market",
        icon: Crown,
        price: "$6,500",
        billing: "per month",
        description: "Government contracting business development",
        features: [
          "Everything in Growth",
          "Federal market analysis",
          "Proposal strategy development",
          "GSA schedule assistance",
          "Government partnership facilitation",
          "Contract opportunity tracking"
        ],
        governmentReady: true
      }
    ],
    tax: [
      {
        id: "basic",
        name: "Individual",
        icon: Shield,
        price: "$200",
        billing: "per return",
        description: "Individual tax preparation",
        features: [
          "Standard tax return preparation",
          "Basic deduction optimization",
          "E-filing included",
          "Email support",
          "Amendment support (if needed)"
        ]
      },
      {
        id: "standard",
        name: "Business",
        icon: Star,
        price: "$750",
        billing: "per return",
        description: "Business tax preparation and planning",
        features: [
          "Everything in Individual",
          "Business tax returns",
          "Quarterly estimated payments",
          "Tax planning strategy",
          "Phone support",
          "Multi-state filing"
        ],
        popular: true
      },
      {
        id: "premium",
        name: "Government Contractor",
        icon: Crown,
        price: "$1,500",
        billing: "per return",
        description: "Specialized government contractor tax services",
        features: [
          "Everything in Business",
          "Government contract tax implications",
          "Federal compliance reporting",
          "Contract cost accounting",
          "IRS representation",
          "Year-round tax support"
        ],
        governmentReady: true
      }
    ]
  };

  return tiersByService[serviceId] || [];
};

export default function PricingTiers({ serviceId, serviceTitle, onSelectTier }: PricingTiersProps) {
  const [selectedTier, setSelectedTier] = useState<string | null>(null);
  const tiers = getPricingTiers(serviceId);

  if (tiers.length === 0) return null;

  const handleSelectTier = (tierId: string) => {
    setSelectedTier(tierId);
    onSelectTier(serviceId, tierId);
  };

  return (
    <div className="mt-8">
      <div className="text-center mb-8">
        <h3 className="text-2xl font-bold text-foreground mb-2">Choose Your {serviceTitle} Package</h3>
        <p className="text-muted-foreground">Select the service level that best fits your business needs</p>
      </div>

      <div className="grid md:grid-cols-3 gap-6">
        {tiers.map((tier) => {
          const Icon = tier.icon;
          const isSelected = selectedTier === tier.id;
          
          return (
            <Card 
              key={tier.id}
              className={`relative transition-all duration-300 cursor-pointer ${
                isSelected ? 'ring-2 ring-primary shadow-lg scale-105' : 'hover:shadow-md'
              } ${tier.popular ? 'border-primary' : ''} ${tier.governmentReady ? 'border-accent' : ''}`}
              onClick={() => handleSelectTier(tier.id)}
              data-testid={`pricing-tier-${serviceId}-${tier.id}`}
            >
              {tier.popular && (
                <div className="absolute -top-3 left-1/2 transform -translate-x-1/2">
                  <Badge className="bg-primary text-primary-foreground px-3 py-1">
                    <Star className="h-3 w-3 mr-1" />
                    Most Popular
                  </Badge>
                </div>
              )}
              
              {tier.governmentReady && (
                <div className="absolute -top-3 right-4">
                  <Badge className="bg-accent text-accent-foreground px-3 py-1">
                    <Shield className="h-3 w-3 mr-1" />
                    Gov Ready
                  </Badge>
                </div>
              )}

              <CardHeader className="text-center pb-4">
                <div className="w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Icon className="h-8 w-8 text-primary" aria-hidden="true" />
                </div>
                <CardTitle className="text-xl font-bold text-foreground">{tier.name}</CardTitle>
                <div className="mt-4">
                  <div className="text-3xl font-bold text-primary">
                    {tier.price}
                    <span className="text-lg font-normal text-muted-foreground">/{tier.billing}</span>
                  </div>
                </div>
                <p className="text-sm text-muted-foreground mt-2">{tier.description}</p>
              </CardHeader>

              <CardContent>
                <ul className="space-y-3 mb-6">
                  {tier.features.map((feature, index) => (
                    <li key={index} className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-3 mt-0.5 flex-shrink-0" aria-hidden="true" />
                      <span className="text-sm text-muted-foreground">{feature}</span>
                    </li>
                  ))}
                </ul>

                <Button 
                  onClick={() => handleSelectTier(tier.id)}
                  variant={isSelected ? "default" : "outline"}
                  className={`w-full ${isSelected ? 'btn-primary' : ''}`}
                  data-testid={`button-select-${serviceId}-${tier.id}`}
                >
                  {isSelected ? (
                    <>
                      <Check className="mr-2 h-4 w-4" />
                      Selected
                    </>
                  ) : (
                    <>
                      Select {tier.name}
                      <ArrowRight className="ml-2 h-4 w-4" />
                    </>
                  )}
                </Button>
              </CardContent>
            </Card>
          );
        })}
      </div>

      {selectedTier && (
        <div className="mt-8 p-6 bg-primary/5 rounded-lg border border-primary/20">
          <div className="text-center">
            <h4 className="text-lg font-semibold text-foreground mb-2">
              Great choice! You've selected the {tiers.find(t => t.id === selectedTier)?.name} package.
            </h4>
            <p className="text-muted-foreground mb-4">
              Ready to get started? Contact us to discuss your specific needs and timeline.
            </p>
            <Button 
              className="btn-primary"
              data-testid={`button-contact-selected-${serviceId}`}
            >
              <Zap className="mr-2 h-4 w-4" />
              Contact Us About This Package
            </Button>
          </div>
        </div>
      )}
    </div>
  );
}