import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { 
  CheckCircle2,
  Circle,
  AlertTriangle,
  Shield,
  FileText,
  Users,
  Clock,
  Info
} from "lucide-react";

interface ComplianceItem {
  id: string;
  title: string;
  description: string;
  urgency: "Critical" | "High" | "Medium" | "Low";
  timeframe: string;
  completed?: boolean;
  requiresAssistance?: boolean;
}

interface ComplianceCategory {
  id: string;
  title: string;
  description: string;
  icon: typeof Shield;
  items: ComplianceItem[];
}

const complianceCategories: ComplianceCategory[] = [
  {
    id: "sam-registration",
    title: "SAM.gov Registration",
    description: "System for Award Management registration requirements",
    icon: Shield,
    items: [
      {
        id: "duns-number",
        title: "DUNS Number Verification",
        description: "Obtain and verify your Data Universal Numbering System identifier",
        urgency: "Critical",
        timeframe: "1-2 business days",
        completed: true
      },
      {
        id: "banking-info",
        title: "Banking Information Setup",
        description: "Configure secure banking details for federal payments",
        urgency: "Critical",
        timeframe: "1 business day",
        completed: true
      },
      {
        id: "sam-registration",
        title: "Complete SAM Registration",
        description: "Submit comprehensive business profile and certifications",
        urgency: "Critical",
        timeframe: "3-5 business days",
        completed: false,
        requiresAssistance: true
      },
      {
        id: "annual-renewal",
        title: "Annual Registration Renewal",
        description: "Maintain active SAM status with yearly updates",
        urgency: "High",
        timeframe: "Ongoing annually",
        completed: false
      }
    ]
  },
  {
    id: "accessibility",
    title: "Accessibility Compliance",
    description: "Section 508 and WCAG requirements for ICT products",
    icon: Users,
    items: [
      {
        id: "section-508",
        title: "Section 508 Compliance",
        description: "Ensure all ICT products meet federal accessibility standards",
        urgency: "Critical",
        timeframe: "Ongoing",
        completed: false,
        requiresAssistance: true
      },
      {
        id: "wcag-compliance",
        title: "WCAG 2.1 AA Compliance",
        description: "Meet Web Content Accessibility Guidelines Level AA",
        urgency: "High",
        timeframe: "Project timeline",
        completed: false,
        requiresAssistance: true
      },
      {
        id: "vpat-documentation",
        title: "VPAT Documentation",
        description: "Create Voluntary Product Accessibility Template",
        urgency: "High",
        timeframe: "2-3 business days",
        completed: false,
        requiresAssistance: true
      },
      {
        id: "accessibility-testing",
        title: "Accessibility Testing",
        description: "Conduct comprehensive accessibility testing and validation",
        urgency: "Medium",
        timeframe: "1-2 weeks",
        completed: false
      }
    ]
  },
  {
    id: "documentation",
    title: "Required Documentation",
    description: "Essential paperwork and certifications for federal contracting",
    icon: FileText,
    items: [
      {
        id: "capability-statement",
        title: "Capability Statement",
        description: "Develop comprehensive business capability overview",
        urgency: "High",
        timeframe: "3-5 business days",
        completed: false,
        requiresAssistance: true
      },
      {
        id: "past-performance",
        title: "Past Performance Documentation",
        description: "Compile previous work examples and client references",
        urgency: "Medium",
        timeframe: "1-2 weeks",
        completed: false
      },
      {
        id: "certifications",
        title: "Business Certifications",
        description: "Obtain relevant small business or minority certifications",
        urgency: "Medium",
        timeframe: "4-8 weeks",
        completed: false
      },
      {
        id: "insurance-bonding",
        title: "Insurance and Bonding",
        description: "Secure appropriate insurance coverage and bonding capacity",
        urgency: "High",
        timeframe: "1-2 weeks",
        completed: false
      }
    ]
  }
];

const getUrgencyColor = (urgency: ComplianceItem['urgency']) => {
  switch (urgency) {
    case 'Critical': return 'bg-red-100 text-red-800 border-red-200';
    case 'High': return 'bg-orange-100 text-orange-800 border-orange-200';
    case 'Medium': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
    case 'Low': return 'bg-green-100 text-green-800 border-green-200';
  }
};

interface ComplianceChecklistProps {
  serviceId?: string;
  showPersonalized?: boolean;
  onRequestAssistance?: (itemId: string) => void;
}

export default function ComplianceChecklist({ 
  serviceId, 
  showPersonalized = false, 
  onRequestAssistance 
}: ComplianceChecklistProps) {
  const [checkedItems, setCheckedItems] = useState<Record<string, boolean>>({});

  const handleToggleItem = (itemId: string) => {
    setCheckedItems(prev => ({
      ...prev,
      [itemId]: !prev[itemId]
    }));
  };

  const getCompletedCount = (category: ComplianceCategory) => {
    return category.items.filter(item => 
      item.completed || checkedItems[item.id]
    ).length;
  };

  const getTotalProgress = () => {
    const total = complianceCategories.reduce((sum, cat) => sum + cat.items.length, 0);
    const completed = complianceCategories.reduce((sum, cat) => 
      sum + getCompletedCount(cat), 0
    );
    return Math.round((completed / total) * 100);
  };

  return (
    <div className="bg-muted/30 rounded-2xl p-8">
      <div className="text-center mb-8">
        <h3 className="text-2xl font-bold text-foreground mb-2">
          Government Contracting Compliance Checklist
        </h3>
        <p className="text-muted-foreground mb-4">
          Track your progress through essential federal contracting requirements
        </p>
        
        {showPersonalized && (
          <div className="max-w-md mx-auto">
            <div className="flex items-center justify-between text-sm text-muted-foreground mb-2">
              <span>Overall Progress</span>
              <span>{getTotalProgress()}% Complete</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-3">
              <div 
                className="bg-gradient-to-r from-primary to-accent h-3 rounded-full transition-all duration-500"
                style={{ width: `${getTotalProgress()}%` }}
              ></div>
            </div>
          </div>
        )}
      </div>

      <Tabs defaultValue={complianceCategories[0].id} className="w-full">
        <TabsList className="grid w-full grid-cols-3 mb-8">
          {complianceCategories.map((category) => {
            const Icon = category.icon;
            const completed = getCompletedCount(category);
            const total = category.items.length;
            
            return (
              <TabsTrigger 
                key={category.id} 
                value={category.id}
                className="flex flex-col items-center p-4 h-auto"
                data-testid={`compliance-tab-${category.id}`}
              >
                <Icon className="h-5 w-5 mb-2" />
                <span className="text-sm font-medium">{category.title}</span>
                <span className="text-xs text-muted-foreground">
                  {completed}/{total} Complete
                </span>
              </TabsTrigger>
            );
          })}
        </TabsList>

        {complianceCategories.map((category) => (
          <TabsContent key={category.id} value={category.id}>
            <Card>
              <CardHeader>
                <div className="flex items-center">
                  <category.icon className="h-6 w-6 text-primary mr-3" />
                  <div>
                    <CardTitle className="text-xl">{category.title}</CardTitle>
                    <p className="text-muted-foreground text-sm mt-1">
                      {category.description}
                    </p>
                  </div>
                </div>
              </CardHeader>
              
              <CardContent>
                <div className="space-y-4">
                  {category.items.map((item) => {
                    const isCompleted = item.completed || checkedItems[item.id];
                    
                    return (
                      <div 
                        key={item.id}
                        className={`p-4 rounded-lg border transition-all ${
                          isCompleted 
                            ? 'bg-green-50 border-green-200' 
                            : 'bg-card border-border hover:border-primary/30'
                        }`}
                      >
                        <div className="flex items-start space-x-4">
                          <button
                            onClick={() => handleToggleItem(item.id)}
                            className="mt-1 transition-colors"
                            data-testid={`compliance-item-${item.id}`}
                          >
                            {isCompleted ? (
                              <CheckCircle2 className="h-5 w-5 text-green-600" />
                            ) : (
                              <Circle className="h-5 w-5 text-gray-400 hover:text-primary" />
                            )}
                          </button>
                          
                          <div className="flex-1">
                            <div className="flex items-start justify-between mb-2">
                              <h4 className={`font-semibold ${
                                isCompleted ? 'text-green-800 line-through' : 'text-foreground'
                              }`}>
                                {item.title}
                              </h4>
                              <div className="flex items-center space-x-2">
                                <Badge variant="outline" className={getUrgencyColor(item.urgency)}>
                                  {item.urgency}
                                </Badge>
                                <Badge variant="outline">
                                  <Clock className="h-3 w-3 mr-1" />
                                  {item.timeframe}
                                </Badge>
                              </div>
                            </div>
                            
                            <p className={`text-sm mb-3 ${
                              isCompleted ? 'text-green-600' : 'text-muted-foreground'
                            }`}>
                              {item.description}
                            </p>
                            
                            {item.requiresAssistance && !isCompleted && (
                              <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
                                <div className="flex items-center text-blue-700">
                                  <Info className="h-4 w-4 mr-2" />
                                  <span className="text-sm font-medium">
                                    Professional assistance recommended
                                  </span>
                                </div>
                                <Button 
                                  size="sm" 
                                  variant="outline"
                                  onClick={() => onRequestAssistance?.(item.id)}
                                  data-testid={`button-assistance-${item.id}`}
                                >
                                  Get Help
                                </Button>
                              </div>
                            )}
                            
                            {isCompleted && (
                              <div className="flex items-center text-green-600 text-sm">
                                <CheckCircle2 className="h-4 w-4 mr-2" />
                                Completed - Great job!
                              </div>
                            )}
                          </div>
                        </div>
                      </div>
                    );
                  })}
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        ))}
      </Tabs>

      {showPersonalized && (
        <div className="mt-8 text-center">
          <Button 
            className="btn-primary"
            data-testid="button-compliance-consultation"
          >
            <Shield className="mr-2 h-4 w-4" />
            Schedule Compliance Consultation
          </Button>
          <p className="text-sm text-muted-foreground mt-2">
            Get personalized guidance on your compliance requirements
          </p>
        </div>
      )}
    </div>
  );
}