import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { 
  CheckCircle2,
  Clock,
  Circle,
  Play,
  ArrowRight,
  FileCheck,
  Building,
  Shield,
  RefreshCw
} from "lucide-react";

interface SAMStep {
  id: number;
  title: string;
  description: string;
  detailedDescription: string;
  timeframe: string;
  requirements: string[];
  icon: typeof Building;
  status: 'completed' | 'in-progress' | 'pending';
}

const samSteps: SAMStep[] = [
  {
    id: 1,
    title: "Initial Assessment",
    description: "Business structure review and documentation gathering",
    detailedDescription: "We review your business structure and help gather all required documentation to ensure a smooth registration process.",
    timeframe: "1-2 business days",
    requirements: [
      "DUNS number verification",
      "Tax identification documents",
      "Business licenses and registrations",
      "Banking information",
      "Key personnel identification"
    ],
    icon: Building,
    status: 'completed'
  },
  {
    id: 2,
    title: "Registration Submission",
    description: "Complete SAM.gov registration with accurate business data",
    detailedDescription: "Our experts handle the complex SAM.gov registration process, ensuring all information is accurate and complete.",
    timeframe: "3-5 business days",
    requirements: [
      "Complete business profile",
      "Banking details verification",
      "Points of contact setup",
      "NAICS code selection",
      "Capability statements"
    ],
    icon: FileCheck,
    status: 'in-progress'
  },
  {
    id: 3,
    title: "Compliance Verification",
    description: "Ensure representations and certifications are complete",
    detailedDescription: "We verify all compliance requirements and certifications are properly completed for your specific business type.",
    timeframe: "2-3 business days",
    requirements: [
      "Representations and certifications",
      "Small business certifications",
      "Cybersecurity compliance",
      "Financial capability review",
      "Past performance documentation"
    ],
    icon: Shield,
    status: 'pending'
  },
  {
    id: 4,
    title: "Ongoing Maintenance",
    description: "Annual renewals and compliance monitoring",
    detailedDescription: "We provide ongoing support to maintain your active SAM status and ensure compliance with evolving requirements.",
    timeframe: "Ongoing annually",
    requirements: [
      "Annual renewal management",
      "Compliance monitoring",
      "Update notifications",
      "Regulatory change alerts",
      "Ongoing support"
    ],
    icon: RefreshCw,
    status: 'pending'
  }
];

interface SAMProgressTrackerProps {
  clientMode?: boolean;
  onGetStarted?: () => void;
}

export default function SAMProgressTracker({ clientMode = false, onGetStarted }: SAMProgressTrackerProps) {
  const [selectedStep, setSelectedStep] = useState<number | null>(null);
  
  const getStatusIcon = (status: SAMStep['status']) => {
    switch (status) {
      case 'completed':
        return <CheckCircle2 className="h-6 w-6 text-green-600" />;
      case 'in-progress':
        return <Play className="h-6 w-6 text-blue-600" />;
      case 'pending':
        return <Circle className="h-6 w-6 text-gray-400" />;
    }
  };

  const getStatusColor = (status: SAMStep['status']) => {
    switch (status) {
      case 'completed':
        return 'bg-green-100 border-green-300 text-green-800';
      case 'in-progress':
        return 'bg-blue-100 border-blue-300 text-blue-800';
      case 'pending':
        return 'bg-gray-100 border-gray-300 text-gray-600';
    }
  };

  const getProgressWidth = () => {
    const completedSteps = samSteps.filter(step => step.status === 'completed').length;
    const inProgressSteps = samSteps.filter(step => step.status === 'in-progress').length;
    return ((completedSteps + (inProgressSteps * 0.5)) / samSteps.length) * 100;
  };

  return (
    <div className="bg-muted/30 rounded-2xl p-8 lg:p-12">
      <div className="text-center mb-8">
        <h2 className="text-3xl font-bold text-foreground mb-4">
          {clientMode ? "Your SAM Registration Progress" : "SAM.gov Registration Process"}
        </h2>
        <p className="text-muted-foreground text-lg max-w-3xl mx-auto">
          Our streamlined process ensures your business is properly registered and compliant 
          with all federal contracting requirements.
        </p>
        
        {clientMode && (
          <div className="mt-6">
            <div className="w-full bg-gray-200 rounded-full h-3 mb-2">
              <div 
                className="bg-gradient-to-r from-primary to-accent h-3 rounded-full transition-all duration-1000 ease-out"
                style={{ width: `${getProgressWidth()}%` }}
              ></div>
            </div>
            <p className="text-sm text-muted-foreground">
              Overall Progress: {Math.round(getProgressWidth())}% Complete
            </p>
          </div>
        )}
      </div>

      <div className="grid lg:grid-cols-4 gap-8">
        {samSteps.map((step, index) => {
          const Icon = step.icon;
          const isSelected = selectedStep === step.id;
          
          return (
            <div key={step.id} className="relative">
              {/* Progress line */}
              {index < samSteps.length - 1 && (
                <div className="hidden lg:block absolute top-16 left-full w-full h-0.5 bg-gray-200 transform -translate-x-1/2 z-0">
                  <div 
                    className={`h-full transition-all duration-700 ${
                      step.status === 'completed' ? 'bg-green-500' : 
                      step.status === 'in-progress' ? 'bg-blue-500 w-1/2' : 'bg-gray-200'
                    }`}
                  ></div>
                </div>
              )}
              
              <Card 
                className={`service-card cursor-pointer transition-all duration-300 relative z-10 ${
                  isSelected ? 'ring-2 ring-primary shadow-lg scale-105' : ''
                } ${getStatusColor(step.status)}`}
                onClick={() => setSelectedStep(isSelected ? null : step.id)}
                data-testid={`sam-step-${step.id}`}
              >
                <CardHeader className="text-center pb-4">
                  <div className="w-16 h-16 bg-background rounded-full flex items-center justify-center mx-auto mb-4 shadow-md">
                    <Icon className="h-8 w-8 text-primary" aria-hidden="true" />
                  </div>
                  <div className="flex items-center justify-center mb-2">
                    {getStatusIcon(step.status)}
                    <span className="ml-2 text-2xl font-bold text-primary">{step.id}</span>
                  </div>
                  <CardTitle className="text-lg font-semibold text-foreground">
                    {step.title}
                  </CardTitle>
                </CardHeader>
                
                <CardContent className="text-center">
                  <p className="text-muted-foreground mb-4 text-sm">
                    {isSelected ? step.detailedDescription : step.description}
                  </p>
                  
                  <Badge variant="outline" className="mb-4">
                    <Clock className="h-3 w-3 mr-1" aria-hidden="true" />
                    {step.timeframe}
                  </Badge>
                  
                  {isSelected && (
                    <div className="mt-4 pt-4 border-t border-border">
                      <h4 className="font-semibold text-foreground mb-2">Key Requirements:</h4>
                      <ul className="text-xs text-muted-foreground space-y-1 text-left">
                        {step.requirements.map((req, idx) => (
                          <li key={idx} className="flex items-center">
                            <ArrowRight className="h-3 w-3 mr-2 text-primary flex-shrink-0" />
                            {req}
                          </li>
                        ))}
                      </ul>
                    </div>
                  )}
                  
                  {clientMode && step.status === 'in-progress' && (
                    <Button 
                      size="sm" 
                      className="mt-4 btn-primary"
                      data-testid={`button-continue-step-${step.id}`}
                    >
                      Continue Step
                    </Button>
                  )}
                </CardContent>
              </Card>
            </div>
          );
        })}
      </div>
      
      {!clientMode && onGetStarted && (
        <div className="text-center mt-12">
          <Button 
            onClick={onGetStarted}
            size="lg"
            className="btn-primary"
            data-testid="button-start-sam-registration"
          >
            <Shield className="mr-2 h-5 w-5" aria-hidden="true" />
            Start Your SAM Registration
          </Button>
          <p className="text-sm text-muted-foreground mt-2">
            Get started with professional SAM registration assistance
          </p>
        </div>
      )}
    </div>
  );
}