import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { Link, useLocation } from "wouter";
import { Menu } from "lucide-react";

// TMH Global primary logo imported
import logoImage from "@assets/images/logo-primary.jpg";

const navigationItems = [
  { href: "/", label: "Home" },
  { href: "/services", label: "Services" },
  { href: "/government-contracting", label: "Government Contracting" },
  { href: "/community", label: "Community" },
  { href: "/training", label: "Training" },
  { href: "/live-training", label: "Live Training" },
  { href: "/consultation", label: "Consultation" },
  { href: "/pricing", label: "Pricing" },
  { href: "/premium", label: "Premium" },
  { href: "/monitoring", label: "Analytics" },
  { href: "/compliance-monitoring", label: "Compliance" },
  { href: "/sms-campaigns", label: "SMS Campaigns" },
  { href: "/client-portal", label: "Client Portal" },
  { href: "/settings", label: "Settings" },
  { href: "/contact", label: "Contact" }
];

export default function Navigation() {
  const [location] = useLocation();
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);

  const isActive = (href: string) => {
    if (href === "/") {
      return location === "/";
    }
    return location.startsWith(href);
  };

  return (
    <header className="bg-card shadow-sm border-b border-border sticky top-0 z-50" role="banner">
      <nav className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8" aria-label="Main navigation">
        <div className="flex justify-between items-center h-16">
          {/* Logo */}
          <div className="flex items-center">
            <Link href="/" className="flex-shrink-0 focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 rounded">
              {/* TMH Global Logo */}
              <img 
                src={logoImage} 
                alt="TMH Global LLC Logo" 
                className="h-12 w-12 object-contain"
                data-testid="company-logo"
              />
              
              {/* Text fallback - hidden when logo loads */}
              {/* <h1 className="text-xl font-bold text-primary">
                TMH Global LLC
              </h1> */}
            </Link>
          </div>
          
          {/* Desktop Navigation */}
          <div className="hidden md:block">
            <div className="ml-10 flex items-baseline space-x-4">
              {navigationItems.map((item) => (
                <Link
                  key={item.href}
                  href={item.href}
                  className={`px-3 py-2 rounded-md text-sm font-medium transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 ${
                    isActive(item.href)
                      ? "text-primary bg-primary/10"
                      : "text-muted-foreground hover:text-primary"
                  }`}
                  data-testid={`nav-link-${item.label.toLowerCase().replace(/\s+/g, '-')}`}
                >
                  {item.label}
                </Link>
              ))}
              <Link href="/contact">
                <Button 
                  className="btn-primary ml-4"
                  data-testid="nav-button-schedule"
                >
                  Schedule Consultation
                </Button>
              </Link>
            </div>
          </div>
          
          {/* Mobile menu button */}
          <div className="md:hidden">
            <Sheet open={mobileMenuOpen} onOpenChange={setMobileMenuOpen}>
              <SheetTrigger asChild>
                <Button 
                  variant="ghost" 
                  size="sm"
                  className="focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring"
                  aria-label="Open navigation menu"
                  data-testid="mobile-menu-button"
                >
                  <Menu className="h-5 w-5" aria-hidden="true" />
                </Button>
              </SheetTrigger>
              <SheetContent side="right" className="w-[300px] sm:w-[350px]">
                <div className="flex flex-col space-y-4 mt-8">
                  {navigationItems.map((item) => (
                    <Link
                      key={item.href}
                      href={item.href}
                      onClick={() => setMobileMenuOpen(false)}
                      className={`block px-3 py-2 rounded-md text-base font-medium transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring ${
                        isActive(item.href)
                          ? "text-primary bg-primary/10"
                          : "text-muted-foreground hover:text-primary"
                      }`}
                      data-testid={`mobile-nav-link-${item.label.toLowerCase().replace(/\s+/g, '-')}`}
                    >
                      {item.label}
                    </Link>
                  ))}
                  <div className="pt-4">
                    <Link href="/contact" onClick={() => setMobileMenuOpen(false)}>
                      <Button 
                        className="w-full btn-primary"
                        data-testid="mobile-nav-button-schedule"
                      >
                        Schedule Consultation
                      </Button>
                    </Link>
                  </div>
                </div>
              </SheetContent>
            </Sheet>
          </div>
        </div>
      </nav>
    </header>
  );
}
