import { useState, useEffect } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { insertContactSchema, type InsertContact } from "@shared/schema";
import { Loader2, Send } from "lucide-react";

const serviceOptions = [
  { value: "bookkeeping", label: "Bookkeeping" },
  { value: "hr", label: "HR Services" },
  { value: "cfo", label: "CFO Services" },
  { value: "auditing", label: "Auditing" },
  { value: "business-development", label: "Business Development" },
  { value: "tax-preparation", label: "Tax Preparation" },
  { value: "government-contracting", label: "Government Contracting" },
  { value: "sam-registration", label: "S.A.M. Registration" },
  { value: "client-portal-access", label: "Client Portal Access" },
];

export default function ContactForm() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isSubmitting, setIsSubmitting] = useState(false);

  // Read URL parameters before form initialization
  const getInitialServiceInterest = () => {
    if (typeof window === 'undefined') return "";
    const urlParams = new URLSearchParams(window.location.search);
    const serviceParam = urlParams.get('service');
    return serviceParam && serviceOptions.some(opt => opt.value === serviceParam) 
      ? serviceParam 
      : "";
  };

  const form = useForm<InsertContact>({
    resolver: zodResolver(insertContactSchema),
    defaultValues: {
      firstName: "",
      lastName: "",
      email: "",
      phone: "",
      serviceInterest: getInitialServiceInterest(),
      message: "",
      isUrgent: false,
      smsOptIn: false,
    },
  });

  const contactMutation = useMutation({
    mutationFn: async (data: InsertContact) => {
      return apiRequest("POST", "/api/contact", data);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/contacts"] });
      toast({
        title: "Message Sent Successfully",
        description: "Thank you for contacting us. We'll get back to you within 24 hours.",
      });
      form.reset();
      setIsSubmitting(false);
    },
    onError: (error: any) => {
      toast({
        title: "Error Sending Message",
        description: error.message || "Please try again later or contact us directly.",
        variant: "destructive",
      });
      setIsSubmitting(false);
    },
  });

  const onSubmit = async (data: InsertContact) => {
    setIsSubmitting(true);
    contactMutation.mutate(data);
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6" noValidate>
        {/* Name Fields */}
        <div className="grid md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="firstName"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-foreground">
                  First Name *
                </FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    className="form-input"
                    placeholder="Enter your first name"
                    disabled={isSubmitting}
                    data-testid="input-first-name"
                    aria-describedby={form.formState.errors.firstName ? "first-name-error" : undefined}
                  />
                </FormControl>
                <FormMessage id="first-name-error" />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="lastName"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-foreground">
                  Last Name *
                </FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    className="form-input"
                    placeholder="Enter your last name"
                    disabled={isSubmitting}
                    data-testid="input-last-name"
                    aria-describedby={form.formState.errors.lastName ? "last-name-error" : undefined}
                  />
                </FormControl>
                <FormMessage id="last-name-error" />
              </FormItem>
            )}
          />
        </div>

        {/* Email Field */}
        <FormField
          control={form.control}
          name="email"
          render={({ field }) => (
            <FormItem>
              <FormLabel className="text-foreground">
                Email Address *
              </FormLabel>
              <FormControl>
                <Input
                  {...field}
                  type="email"
                  className="form-input"
                  placeholder="Enter your email address"
                  disabled={isSubmitting}
                  data-testid="input-email"
                  aria-describedby={form.formState.errors.email ? "email-error" : undefined}
                />
              </FormControl>
              <FormMessage id="email-error" />
            </FormItem>
          )}
        />

        {/* Phone Field */}
        <FormField
          control={form.control}
          name="phone"
          render={({ field }) => (
            <FormItem>
              <FormLabel className="text-foreground">
                Phone Number
              </FormLabel>
              <FormControl>
                <Input
                  {...field}
                  type="tel"
                  className="form-input"
                  placeholder="Enter your phone number"
                  disabled={isSubmitting}
                  data-testid="input-phone"
                  aria-describedby={form.formState.errors.phone ? "phone-error" : undefined}
                />
              </FormControl>
              <FormMessage id="phone-error" />
            </FormItem>
          )}
        />


        {/* Service Interest */}
        <FormField
          control={form.control}
          name="serviceInterest"
          render={({ field }) => (
            <FormItem>
              <FormLabel className="text-foreground">
                Service Interest
              </FormLabel>
              <Select 
                onValueChange={field.onChange} 
                value={field.value || ""} 
                disabled={isSubmitting}
              >
                <FormControl>
                  <SelectTrigger 
                    className="form-input"
                    data-testid="select-service-interest"
                    aria-describedby={form.formState.errors.serviceInterest ? "service-error" : undefined}
                  >
                    <SelectValue placeholder="Select a service of interest" />
                  </SelectTrigger>
                </FormControl>
                <SelectContent>
                  {serviceOptions.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <FormMessage id="service-error" />
            </FormItem>
          )}
        />

        {/* Message Field */}
        <FormField
          control={form.control}
          name="message"
          render={({ field }) => (
            <FormItem>
              <FormLabel className="text-foreground">
                Message *
              </FormLabel>
              <FormControl>
                <Textarea
                  {...field}
                  className="form-input resize-vertical min-h-[120px]"
                  placeholder="Tell us about your needs, questions, or how we can help you..."
                  disabled={isSubmitting}
                  data-testid="textarea-message"
                  aria-describedby={form.formState.errors.message ? "message-error" : "message-help"}
                />
              </FormControl>
              <div id="message-help" className="text-sm text-muted-foreground mt-1">
                Please provide as much detail as possible to help us assist you better.
              </div>
              <FormMessage id="message-error" />
            </FormItem>
          )}
        />

        {/* SMS Opt-in */}
        <FormField
          control={form.control}
          name="smsOptIn"
          render={({ field }) => (
            <FormItem className="flex flex-row items-start space-x-3 space-y-0">
              <FormControl>
                <Checkbox
                  checked={field.value}
                  onCheckedChange={field.onChange}
                  disabled={isSubmitting}
                  data-testid="checkbox-sms-opt-in"
                />
              </FormControl>
              <div className="space-y-1 leading-none">
                <FormLabel className="text-sm font-normal">
                  I consent to receive customer care text messages from TMH GLOBAL. Reply STOP to opt-out; Reply HELP for support; Message and data rates apply; Messaging frequency may vary. Visit /privacy for privacy policy and /terms for Terms of Service.
                </FormLabel>
              </div>
            </FormItem>
          )}
        />

        {/* Submit Button */}
        <Button
          type="submit"
          className="w-full btn-primary"
          disabled={isSubmitting}
          data-testid="button-submit-contact"
        >
          {isSubmitting ? (
            <>
              <Loader2 className="mr-2 h-4 w-4 animate-spin" aria-hidden="true" />
              Sending Message...
            </>
          ) : (
            <>
              <Send className="mr-2 h-4 w-4" aria-hidden="true" />
              Send Message
            </>
          )}
        </Button>

        {/* Additional Help Text */}
        <div className="text-center text-sm text-muted-foreground">
          <p>
            Need immediate assistance? Call us at{" "}
            <a 
              href="tel:+17038297277" 
              className="text-primary hover:underline focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring rounded"
              data-testid="link-phone-contact"
            >
              (703) 829-7277 ext 101
            </a>
          </p>
          <p className="text-accent mt-1">
            📱 Text preferred for quicker response
          </p>
        </div>
      </form>
    </Form>
  );
}
