import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { insertConsultationSchema, type InsertConsultation } from "@shared/schema";
import { Loader2, Calendar } from "lucide-react";

const serviceTypes = [
  { value: "general-consultation", label: "General Consultation" },
  { value: "bookkeeping", label: "Bookkeeping Services" },
  { value: "hr-services", label: "HR Services" },
  { value: "cfo-services", label: "CFO Services" },
  { value: "auditing", label: "Auditing" },
  { value: "business-development", label: "Business Development" },
  { value: "tax-preparation", label: "Tax Preparation" },
  { value: "sam-registration", label: "S.A.M. Registration" },
  { value: "government-contracting", label: "Government Contracting" },
  { value: "compliance-assessment", label: "Compliance Assessment" },
  { value: "vpat-documentation", label: "VPAT Documentation" },
];

const timeSlots = [
  { value: "09:00", label: "9:00 AM" },
  { value: "10:00", label: "10:00 AM" },
  { value: "11:00", label: "11:00 AM" },
  { value: "13:00", label: "1:00 PM" },
  { value: "14:00", label: "2:00 PM" },
  { value: "15:00", label: "3:00 PM" },
  { value: "16:00", label: "4:00 PM" },
];

export default function ConsultationForm() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isSubmitting, setIsSubmitting] = useState(false);

  const form = useForm<InsertConsultation>({
    resolver: zodResolver(insertConsultationSchema),
    defaultValues: {
      firstName: "",
      lastName: "",
      email: "",
      phone: "",
      companyName: "",
      serviceType: "",
      preferredDate: "",
      preferredTime: "",
      message: "",
    },
  });

  const consultationMutation = useMutation({
    mutationFn: async (data: InsertConsultation) => {
      const response = await apiRequest("POST", "/api/consultations", data);
      return response.json();
    },
    onSuccess: (result: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/consultations"] });
      
      const isConflict = result.hasConflict;
      const message = result.message || "Thank you! We'll contact you within 24 hours to confirm your consultation time.";
      
      toast({
        title: isConflict ? "Consultation Scheduled with Note" : "Consultation Scheduled",
        description: message,
        variant: isConflict ? "default" : "default", // Use warning-like styling for conflicts if needed
      });
      form.reset();
      setIsSubmitting(false);
    },
    onError: (error: any) => {
      let errorMessage = "Please try again later or contact us directly.";
      
      // Handle validation errors from server
      if (error.message && error.message.includes("Required")) {
        errorMessage = "Please fill in all required fields marked with an asterisk (*).";
      } else if (error.message) {
        errorMessage = error.message;
      }
      
      toast({
        title: "Error Scheduling Consultation",
        description: errorMessage,
        variant: "destructive",
      });
      setIsSubmitting(false);
    },
  });

  const onSubmit = async (data: InsertConsultation) => {
    // Check for validation errors before submitting
    const errors = form.formState.errors;
    if (Object.keys(errors).length > 0) {
      // Show validation errors to user
      const errorFields = Object.keys(errors).map(field => {
        const fieldLabels: Record<string, string> = {
          firstName: "First Name",
          lastName: "Last Name", 
          email: "Email Address",
          serviceType: "Consultation Type"
        };
        return fieldLabels[field] || field;
      });
      
      toast({
        title: "Please Fix the Following Errors",
        description: `Missing required fields: ${errorFields.join(", ")}`,
        variant: "destructive",
      });
      setIsSubmitting(false);
      return;
    }

    setIsSubmitting(true);
    consultationMutation.mutate(data);
  };

  // Generate date options starting 10 business days from today (excluding weekends)
  const getDateOptions = () => {
    const dates = [];
    const today = new Date();
    let currentDate = new Date(today);
    
    // Start from 10 business days from today
    let businessDaysAdded = 0;
    while (businessDaysAdded < 10) {
      currentDate.setDate(currentDate.getDate() + 1);
      const dayOfWeek = currentDate.getDay();
      // Count only weekdays (Monday = 1, Friday = 5)
      if (dayOfWeek >= 1 && dayOfWeek <= 5) {
        businessDaysAdded++;
      }
    }

    // Now generate 20 available date options from that point
    while (dates.length < 20) {
      const dayOfWeek = currentDate.getDay();
      // Only include weekdays (Monday = 1, Friday = 5)
      if (dayOfWeek >= 1 && dayOfWeek <= 5) {
        dates.push({
          value: currentDate.toISOString().split('T')[0],
          label: currentDate.toLocaleDateString('en-US', { 
            weekday: 'long', 
            year: 'numeric', 
            month: 'long', 
            day: 'numeric' 
          })
        });
      }
      currentDate.setDate(currentDate.getDate() + 1);
    }
    return dates;
  };

  const dateOptions = getDateOptions();

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6" noValidate>
        {/* Name Fields */}
        <div className="grid md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="firstName"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-foreground">
                  First Name *
                </FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    className="form-input"
                    placeholder="Enter your first name"
                    disabled={isSubmitting}
                    data-testid="input-consultation-first-name"
                    aria-describedby={form.formState.errors.firstName ? "consultation-first-name-error" : undefined}
                  />
                </FormControl>
                <FormMessage id="consultation-first-name-error" />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="lastName"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-foreground">
                  Last Name *
                </FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    className="form-input"
                    placeholder="Enter your last name"
                    disabled={isSubmitting}
                    data-testid="input-consultation-last-name"
                    aria-describedby={form.formState.errors.lastName ? "consultation-last-name-error" : undefined}
                  />
                </FormControl>
                <FormMessage id="consultation-last-name-error" />
              </FormItem>
            )}
          />
        </div>

        {/* Email and Phone */}
        <div className="grid md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="email"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-foreground">
                  Email Address *
                </FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    type="email"
                    className="form-input"
                    placeholder="Enter your email"
                    disabled={isSubmitting}
                    data-testid="input-consultation-email"
                    aria-describedby={form.formState.errors.email ? "consultation-email-error" : undefined}
                  />
                </FormControl>
                <FormMessage id="consultation-email-error" />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="phone"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-foreground">
                  Phone Number
                </FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    value={field.value || ""}
                    type="tel"
                    className="form-input"
                    placeholder="Enter your phone number"
                    disabled={isSubmitting}
                    data-testid="input-consultation-phone"
                    aria-describedby={form.formState.errors.phone ? "consultation-phone-error" : undefined}
                  />
                </FormControl>
                <FormMessage id="consultation-phone-error" />
              </FormItem>
            )}
          />
        </div>

        {/* Company Name */}
        <FormField
          control={form.control}
          name="companyName"
          render={({ field }) => (
            <FormItem>
              <FormLabel className="text-foreground">
                Company Name
              </FormLabel>
              <FormControl>
                <Input
                  {...field}
                  value={field.value || ""}
                  className="form-input"
                  placeholder="Enter your company name (optional)"
                  disabled={isSubmitting}
                  data-testid="input-consultation-company"
                  aria-describedby={form.formState.errors.companyName ? "consultation-company-error" : undefined}
                />
              </FormControl>
              <FormMessage id="consultation-company-error" />
            </FormItem>
          )}
        />

        {/* Service Type */}
        <FormField
          control={form.control}
          name="serviceType"
          render={({ field }) => (
            <FormItem>
              <FormLabel className="text-foreground">
                Consultation Type *
              </FormLabel>
              <Select 
                onValueChange={field.onChange} 
                defaultValue={field.value || ""} 
                disabled={isSubmitting}
              >
                <FormControl>
                  <SelectTrigger 
                    className="form-input"
                    data-testid="select-consultation-service-type"
                    aria-describedby={form.formState.errors.serviceType ? "consultation-service-error" : undefined}
                  >
                    <SelectValue placeholder="Select consultation type" />
                  </SelectTrigger>
                </FormControl>
                <SelectContent>
                  {serviceTypes.map((option) => (
                    <SelectItem key={option.value} value={option.value}>
                      {option.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <FormMessage id="consultation-service-error" />
            </FormItem>
          )}
        />

        {/* Preferred Date and Time */}
        <div className="grid md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="preferredDate"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-foreground">
                  Preferred Date
                </FormLabel>
                <Select 
                  onValueChange={field.onChange} 
                  defaultValue={field.value || ""} 
                  disabled={isSubmitting}
                >
                  <FormControl>
                    <SelectTrigger 
                      className="form-input"
                      data-testid="select-consultation-date"
                      aria-describedby={form.formState.errors.preferredDate ? "consultation-date-error" : undefined}
                    >
                      <SelectValue placeholder="Select preferred date" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    {dateOptions.map((date) => (
                      <SelectItem key={date.value} value={date.value}>
                        {date.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <FormMessage id="consultation-date-error" />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="preferredTime"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="text-foreground">
                  Preferred Time
                </FormLabel>
                <Select 
                  onValueChange={field.onChange} 
                  defaultValue={field.value || ""} 
                  disabled={isSubmitting}
                >
                  <FormControl>
                    <SelectTrigger 
                      className="form-input"
                      data-testid="select-consultation-time"
                      aria-describedby={form.formState.errors.preferredTime ? "consultation-time-error" : undefined}
                    >
                      <SelectValue placeholder="Select preferred time" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    {timeSlots.map((time) => (
                      <SelectItem key={time.value} value={time.value}>
                        {time.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <FormMessage id="consultation-time-error" />
              </FormItem>
            )}
          />
        </div>

        {/* Additional Information */}
        <FormField
          control={form.control}
          name="message"
          render={({ field }) => (
            <FormItem>
              <FormLabel className="text-foreground">
                Additional Information
              </FormLabel>
              <FormControl>
                <Textarea
                  {...field}
                  value={field.value || ""}
                  className="form-input resize-vertical min-h-[100px]"
                  placeholder="Please share any specific topics you'd like to discuss or questions you have..."
                  disabled={isSubmitting}
                  data-testid="textarea-consultation-message"
                  aria-describedby={form.formState.errors.message ? "consultation-message-error" : "consultation-message-help"}
                />
              </FormControl>
              <div id="consultation-message-help" className="text-sm text-muted-foreground mt-1">
                Optional: Help us prepare for your consultation by sharing your specific needs.
              </div>
              <FormMessage id="consultation-message-error" />
            </FormItem>
          )}
        />

        {/* Submit Button */}
        <Button
          type="submit"
          className="w-full btn-primary"
          disabled={isSubmitting}
          data-testid="button-submit-consultation"
        >
          {isSubmitting ? (
            <>
              <Loader2 className="mr-2 h-4 w-4 animate-spin" aria-hidden="true" />
              Scheduling Consultation...
            </>
          ) : (
            <>
              <Calendar className="mr-2 h-4 w-4" aria-hidden="true" />
              Schedule Consultation
            </>
          )}
        </Button>

        {/* Additional Information */}
        <div className="bg-muted/50 p-4 rounded-lg">
          <h4 className="font-semibold text-foreground mb-2">What to Expect:</h4>
          <ul className="text-sm text-muted-foreground space-y-1">
            <li>• Free 30-minute initial consultation</li>
            <li>• Discussion of your specific needs and goals</li>
            <li>• Overview of our services and how we can help</li>
            <li>• No obligation - just helpful information</li>
          </ul>
        </div>
      </form>
    </Form>
  );
}
