import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { AlertCircle, Calendar, CheckCircle, ExternalLink } from "lucide-react";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { useToast } from "@/hooks/use-toast";

interface CalendarStatus {
  connected: boolean;
  email: string | null;
}

export function CalendarSetup() {
  const [status, setStatus] = useState<CalendarStatus>({ connected: false, email: null });
  const [loading, setLoading] = useState(false);
  const [authUrl, setAuthUrl] = useState<string>("");
  const { toast } = useToast();

  useEffect(() => {
    checkStatus();
  }, []);

  const checkStatus = async () => {
    try {
      const response = await fetch('/api/calendar/status');
      const data = await response.json();
      setStatus(data);
    } catch (error) {
      console.error('Failed to check calendar status:', error);
    }
  };

  const initiateAuth = async () => {
    setLoading(true);
    try {
      const response = await fetch('/api/auth/google');
      const data = await response.json();
      
      if (data.authUrl) {
        setAuthUrl(data.authUrl);
        
        // Listen for messages from OAuth popup
        const handleMessage = (event: MessageEvent) => {
          if (event.data?.type === 'GOOGLE_AUTH_SUCCESS') {
            checkStatus();
            toast({
              title: "Calendar Connected",
              description: "Google Calendar is now connected to prevent double bookings.",
            });
            window.removeEventListener('message', handleMessage);
            setAuthUrl("");
            setLoading(false);
          } else if (event.data?.type === 'GOOGLE_AUTH_ERROR') {
            toast({
              title: "Authentication Failed",
              description: event.data.error || "Failed to connect Google Calendar.",
              variant: "destructive",
            });
            window.removeEventListener('message', handleMessage);
            setAuthUrl("");
            setLoading(false);
          }
        };
        
        window.addEventListener('message', handleMessage);
        
        // Open in new window for OAuth flow
        const popup = window.open(data.authUrl, 'google-auth', 'width=500,height=600');
        
        // Fallback: check if popup was closed manually
        const popupCheck = setInterval(() => {
          if (popup?.closed) {
            clearInterval(popupCheck);
            window.removeEventListener('message', handleMessage);
            setAuthUrl("");
            setLoading(false);
          }
        }, 1000);
        
        // Clear listeners after 5 minutes
        setTimeout(() => {
          clearInterval(popupCheck);
          window.removeEventListener('message', handleMessage);
          setLoading(false);
        }, 300000);
      }
    } catch (error) {
      console.error('Failed to initiate auth:', error);
      toast({
        title: "Authentication Failed",
        description: "Failed to start Google Calendar authentication.",
        variant: "destructive",
      });
      setLoading(false);
    }
  };

  const disconnect = async () => {
    try {
      const response = await fetch('/api/calendar/disconnect', {
        method: 'POST',
      });
      
      if (response.ok) {
        setStatus({ connected: false, email: null });
        toast({
          title: "Calendar Disconnected",
          description: "Google Calendar has been disconnected.",
        });
      }
    } catch (error) {
      console.error('Failed to disconnect:', error);
      toast({
        title: "Disconnect Failed",
        description: "Failed to disconnect Google Calendar.",
        variant: "destructive",
      });
    }
  };

  return (
    <Card data-testid="calendar-setup-card">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Calendar className="h-5 w-5" />
          Google Calendar Integration
        </CardTitle>
        <CardDescription>
          Connect your Google Calendar to prevent double bookings and sync consultation appointments
        </CardDescription>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            <span>Status:</span>
            {status.connected ? (
              <Badge variant="default" className="bg-green-100 text-green-800">
                <CheckCircle className="h-3 w-3 mr-1" />
                Connected
              </Badge>
            ) : (
              <Badge variant="secondary" className="bg-yellow-100 text-yellow-800">
                <AlertCircle className="h-3 w-3 mr-1" />
                Not Connected
              </Badge>
            )}
          </div>
          
          {status.connected ? (
            <Button
              variant="outline"
              size="sm"
              onClick={disconnect}
              data-testid="button-disconnect-calendar"
            >
              Disconnect
            </Button>
          ) : (
            <Button
              onClick={initiateAuth}
              disabled={loading}
              data-testid="button-connect-calendar"
            >
              {loading ? "Connecting..." : "Connect Calendar"}
            </Button>
          )}
        </div>

        {status.connected && status.email && (
          <Alert>
            <CheckCircle className="h-4 w-4" />
            <AlertDescription>
              Connected to: <strong>{status.email}</strong>
              <br />
              Consultation bookings will now check for conflicts and automatically create calendar events.
            </AlertDescription>
          </Alert>
        )}

        {!status.connected && (
          <Alert>
            <AlertCircle className="h-4 w-4" />
            <AlertDescription>
              Connect your Google Calendar to enable automatic conflict detection and prevent double bookings from other scheduling tools.
            </AlertDescription>
          </Alert>
        )}

        {authUrl && (
          <Alert>
            <ExternalLink className="h-4 w-4" />
            <AlertDescription>
              Complete the authorization in the popup window. If it didn't open automatically,{" "}
              <a 
                href={authUrl} 
                target="_blank" 
                rel="noopener noreferrer"
                className="underline text-blue-600 hover:text-blue-800"
                data-testid="link-manual-auth"
              >
                click here to authorize
              </a>
            </AlertDescription>
          </Alert>
        )}

        <div className="text-sm text-muted-foreground">
          <p className="font-medium mb-1">Benefits of calendar integration:</p>
          <ul className="list-disc list-inside space-y-1">
            <li>Automatic conflict detection before booking confirmations</li>
            <li>Two-way sync: consultations create calendar events automatically</li>
            <li>Prevents double bookings from external scheduling tools</li>
            <li>Email notifications to attendees via Google Calendar</li>
          </ul>
        </div>
      </CardContent>
    </Card>
  );
}