import { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Shield, CreditCard } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import PayPalButton from "./PayPalButton";

interface SquareCheckoutProps {
  amount: number;
  planName: string;
  planId: string;
  onSuccess?: () => void;
  onCancel?: () => void;
}

interface CustomerInfo {
  firstName: string;
  lastName: string;
  email: string;
  company?: string;
}

export default function SquareCheckout({ amount, planName, planId, onSuccess, onCancel }: SquareCheckoutProps) {
  const [customerInfo, setCustomerInfo] = useState<CustomerInfo>({
    firstName: '',
    lastName: '',
    email: '',
    company: ''
  });
  const [paymentMethod, setPaymentMethod] = useState<'square' | 'paypal' | null>(null);
  const [isProcessing, setIsProcessing] = useState(false);
  const { toast } = useToast();

  const handleInputChange = (field: keyof CustomerInfo, value: string) => {
    setCustomerInfo(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleSquarePayment = async () => {
    // Validate customer info
    if (!customerInfo.firstName || !customerInfo.lastName || !customerInfo.email) {
      toast({
        title: "Missing Information",
        description: "Please fill in all required customer information fields.",
        variant: "destructive",
      });
      return;
    }

    setIsProcessing(true);
    
    try {
      // Create payment with customer info
      const response = await fetch('/api/square/create-payment', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          amount: amount,
          planId: planId,
          customer: customerInfo,
          paymentMethod: 'square_card'
        }),
      });

      const result = await response.json();
      
      if (result.success) {
        toast({
          title: "Payment Successful!",
          description: `Welcome to ${planName}! You'll receive a confirmation email shortly.`,
        });
        onSuccess?.();
      } else {
        throw new Error(result.message || 'Payment failed');
      }
    } catch (error: any) {
      toast({
        title: "Payment Processing",
        description: "Please call (703) 829-7277 to complete your credit card payment securely.",
        duration: 8000,
      });
    } finally {
      setIsProcessing(false);
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 py-12">
      <div className="max-w-md mx-auto">
        <Card className="shadow-xl">
          <CardHeader className="bg-blue-600 text-white">
            <CardTitle className="flex items-center gap-2">
              <Shield className="h-5 w-5" />
              Secure Checkout
            </CardTitle>
            <div className="text-sm opacity-90">
              {planName} - ${amount}/month
            </div>
          </CardHeader>
          
          <CardContent className="p-6">
            <div className="space-y-4">
              {/* Customer Information */}
              <div className="space-y-4">
                <h3 className="font-semibold text-gray-900">Customer Information</h3>
                
                <div className="grid grid-cols-2 gap-3">
                  <div>
                    <Label htmlFor="firstName">First Name *</Label>
                    <Input
                      id="firstName"
                      type="text"
                      value={customerInfo.firstName}
                      onChange={(e) => handleInputChange('firstName', e.target.value)}
                      required
                      data-testid="input-firstname"
                      className="mt-1"
                    />
                  </div>
                  
                  <div>
                    <Label htmlFor="lastName">Last Name *</Label>
                    <Input
                      id="lastName"
                      type="text"
                      value={customerInfo.lastName}
                      onChange={(e) => handleInputChange('lastName', e.target.value)}
                      required
                      data-testid="input-lastname"
                      className="mt-1"
                    />
                  </div>
                </div>

                <div>
                  <Label htmlFor="email">Email Address *</Label>
                  <Input
                    id="email"
                    type="email"
                    value={customerInfo.email}
                    onChange={(e) => handleInputChange('email', e.target.value)}
                    required
                    data-testid="input-email"
                    className="mt-1"
                  />
                </div>

                <div>
                  <Label htmlFor="company">Company (Optional)</Label>
                  <Input
                    id="company"
                    type="text"
                    value={customerInfo.company}
                    onChange={(e) => handleInputChange('company', e.target.value)}
                    data-testid="input-company"
                    className="mt-1"
                  />
                </div>
              </div>

              {/* Payment Section */}
              <div className="border-t pt-4">
                <h3 className="font-semibold text-gray-900 mb-3">Payment</h3>
                
                <div className="bg-gray-100 p-3 rounded mb-4">
                  <div className="flex justify-between text-sm">
                    <span>Plan:</span>
                    <span>{planName}</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span>Amount:</span>
                    <span>${amount}/month</span>
                  </div>
                  <div className="flex justify-between font-semibold mt-2 pt-2 border-t">
                    <span>Total:</span>
                    <span>${amount}/month</span>
                  </div>
                </div>

                {/* Payment Method Selection */}
                {!paymentMethod && (
                  <div className="w-full space-y-4">
                    <Alert className="bg-blue-50 border-blue-200">
                      <Shield className="h-4 w-4" />
                      <AlertDescription>
                        <strong>Choose your payment method:</strong> Your payment information is secure and encrypted.
                      </AlertDescription>
                    </Alert>
                    
                    {/* Square Credit Card Option */}
                    <div className="border rounded-lg p-4 hover:bg-gray-50 cursor-pointer" onClick={() => setPaymentMethod('square')}>
                      <h4 className="font-semibold mb-2 flex items-center gap-2">
                        <CreditCard className="h-5 w-5" />
                        Pay with Credit Card
                      </h4>
                      <p className="text-sm text-gray-600 mb-3">Visa, Mastercard, American Express, and more</p>
                      <Button
                        className="w-full bg-green-600 hover:bg-green-700 text-white"
                        data-testid="button-select-square"
                      >
                        Continue with Credit Card
                      </Button>
                    </div>
                    
                    {/* PayPal Option */}
                    <div className="border rounded-lg p-4 hover:bg-gray-50 cursor-pointer" onClick={() => setPaymentMethod('paypal')}>
                      <h4 className="font-semibold mb-2">🅿️ Pay with PayPal</h4>
                      <p className="text-sm text-gray-600 mb-3">Use your PayPal account balance or linked cards</p>
                      <Button
                        className="w-full bg-blue-600 hover:bg-blue-700 text-white"
                        data-testid="button-select-paypal"
                      >
                        Continue with PayPal
                      </Button>
                    </div>
                  </div>
                )}

                {/* Square Credit Card Form */}
                {paymentMethod === 'square' && (
                  <div className="w-full space-y-4">
                    <div className="flex items-center gap-2 mb-4">
                      <Button 
                        variant="outline" 
                        size="sm" 
                        onClick={() => setPaymentMethod(null)}
                      >
                        ← Back
                      </Button>
                      <h4 className="font-semibold">Credit Card Payment</h4>
                    </div>
                    
                    <Alert className="bg-green-50 border-green-200">
                      <CreditCard className="h-4 w-4" />
                      <AlertDescription>
                        <strong>Secure Payment:</strong> Your credit card information is processed securely through Square.
                      </AlertDescription>
                    </Alert>

                    <div className="border rounded-lg p-4 bg-gray-50">
                      <h5 className="font-medium mb-3">Credit Card Details</h5>
                      <div className="space-y-3">
                        <div>
                          <Label className="text-sm">Card Number</Label>
                          <Input 
                            placeholder="4111 1111 1111 1111" 
                            className="mt-1" 
                            disabled={isProcessing}
                          />
                        </div>
                        <div className="grid grid-cols-2 gap-3">
                          <div>
                            <Label className="text-sm">Expiry Date</Label>
                            <Input 
                              placeholder="MM/YY" 
                              className="mt-1" 
                              disabled={isProcessing}
                            />
                          </div>
                          <div>
                            <Label className="text-sm">CVV</Label>
                            <Input 
                              placeholder="123" 
                              className="mt-1" 
                              disabled={isProcessing}
                            />
                          </div>
                        </div>
                      </div>
                    </div>

                    <Button
                      className="w-full bg-green-600 hover:bg-green-700 text-white"
                      onClick={handleSquarePayment}
                      disabled={isProcessing}
                      data-testid="button-pay-with-square"
                    >
                      {isProcessing ? (
                        <>Processing Payment...</>
                      ) : (
                        <>Complete Payment - ${amount}/month</>
                      )}
                    </Button>
                  </div>
                )}

                {/* PayPal Payment */}
                {paymentMethod === 'paypal' && (
                  <div className="w-full space-y-4">
                    <div className="flex items-center gap-2 mb-4">
                      <Button 
                        variant="outline" 
                        size="sm" 
                        onClick={() => setPaymentMethod(null)}
                      >
                        ← Back
                      </Button>
                      <h4 className="font-semibold">PayPal Payment</h4>
                    </div>
                    
                    <PayPalButton
                      amount={amount.toString()}
                      currency="USD"
                      intent="CAPTURE"
                    />
                  </div>
                )}

                {/* Help Text */}
                {!paymentMethod && (
                  <div className="text-center text-sm text-gray-600 border-t pt-3">
                    <strong>Need help?</strong> Call us at (703) 829-7277 for assistance
                  </div>
                )}

                {onCancel && (
                  <Button
                    type="button"
                    variant="outline"
                    className="w-full mt-3"
                    onClick={onCancel}
                    data-testid="button-cancel-payment"
                  >
                    Cancel
                  </Button>
                )}
              </div>
            </div>

            <div className="mt-6 text-center text-xs text-gray-500">
              <div className="flex items-center justify-center gap-2">
                <Shield className="h-3 w-3" />
                <span>Secured by Square & PayPal • SSL Encrypted</span>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}